function [k,c_array] = poly2lat( arg1, arg2 )
%POLY2LAT Compute reflection and "c" coefficients from polynomial coefficients.
%        [K,C] = POLY2LAT( B, A ) computes the reflection coefficients K
%        and the "C"-coefficients of the IIR filter with transfer function:
%
%                  b0 + b1(z)^-1 + b2(z)^-2 + ... + bN(z)^-N
%           H(z) = ------------------------------------------         
%                  a0 + a1(z)^-1 + a2(z)^-2 + ... + aN(z)^-N
%
%        A and B polynomial must have real-valued coefficients.
%        K is a row vector of length one less than B.  Length of C equals N.
%
%        CAUTION: If abs(K(i)) == 1 for any i, finding the reflection
%        coefficients is an ill conditioned problem. POLY2LAT will return
%        some NaNs and provide a warning message in this case.
%
%        For an FIR filter use [K,C] = POLY2LAT( B ) or POLY2LAT( B, 1 ). 
%        In the case of an FIR filter POLY2LAT returns a two element vector 
%        in C. C(1) is the feedforward gain and C(2) is +/- 1, such that
%              y(n) = C(1)x(n) + C(2)[ x(n) + ... + bN x(n-N) ]
%        and C(1)+C(2) = b0.
%
%        For an ALL-POLE IIR filter use [K,C] = POLY2LAT( 1, A ).
%        In this case K again returns the reflection coefficients, and C = 1.
%        For IIR filters we always assume that a0 = A(1) equals 1,
%        otherwise LAT2POLY will not return the correct A.
%
%        See also LAT2POLY.

%
%    Author: Mehmet Zeytinoglu (mzeytin@ee.ryerson.ca)
%            Department of Electrical and Computer Engineering
%            Ryerson Polytechnic University
%            Toronto, Ontario, M5B 2K3 
%            CANADA
%
%    Copyright (c) 1997 Mehmet Zeytinoglu
%
%    $Revision: 1.3 $    $Date: 1997/03/11 16:18:10 $
%

if (nargin == 1)                           % ==== FIR ====

   poly1 = arg1;
   poly2 = zeros(size(poly1));
   type  = 1;

elseif (nargin == 2)                       % ==== IIR or FIR      ====

   if ( arg2 == 1 )                        % ==== FIR: again      ====
      poly1 = arg1;
      poly2 = zeros(size(poly1));
      type  = 1;
   elseif ( arg1 == 1 )                    % ==== IIR: (all-pole) ====
      poly1 = arg2;
      poly2 = zeros(size(poly1));
      type  = 2;
   elseif ( length(arg1) == length(arg2) ) % ==== IIR: (pole+zero) ====
      poly2 = arg1;
      poly1 = arg2;
      type  = 3;
   end

end

%============================================================
%  Special FIR case 
%============================================================
if ( type == 1 )   

   N = length(poly1);
   if ( poly1(1) ~= 1 )
     c_array = [ poly1(1)-sign(poly1(1))  sign(poly1(1)) ];
       poly1 = [ 1 sign(poly1(1))*poly1(2:N) ];
   else
     c_array = [ 0 1 ];
   end
   k = poly2rc(poly1);
   return

end

%============================================================
%  These are the IIR cases
%============================================================
poly1(1) = [];
c = poly2;

for i = length(poly1):-1:2
    k(i) = poly1(i);
    c(1:i) = c(1:i) - c(i+1)*poly1(i:-1:1);
    poly1(i) = [];
    poly1 = (poly1 - k(i)*poly1(i-1:-1:1))/(1 - k(i)^2);
end 

k(1) = poly1(1);
c(1) = c(1)-c(2)*poly1(1);

if ( type == 2 )
   c_array = 1;
elseif ( type == 3 )
   c_array = c;
end
