function [ N ] = hilbord( deltadB, df, fs )
%HILBORD estimates the order of optimal FIR Hilbert Transformer (HT) as a 
%        function of peak error (DE) and edge frequencies.
%
%        N = HILBORD( DE, DF, FS ) estimates N, the order of optimal FIR HT.
%        DF represents the lower and upper edge frequencies. 
%        FS is the sampling frequency such that DF <= FS/4.
%        If FS is not specified as an input argument then it is assumed that
%        FS = 2.  DE represents the allowable peak relative error expressed
%        in dB.  Acceptable values for DE are between -Inf and 0.
%
%        The normalized values for FP defined as FP/(FS/2) are realistically
%        limited to between 0.01 and 0.2.  
%
%        The N values corresponding to DF/(FS/2) = {0.01,0.02,0.05,0.10} are
%        "exact".  Other intermediate values of DF/(FS/2) will result in 
%        linear interpolation between these exact values.  
%        N values are limited to the range 3--128 (arbitrary values).
%        The "exact" relations for N are based on the design curves 
%        taken from the text Rabiner and Gold (Figure 3.79 on page 177).

%
%    Author: Mehmet Zeytinoglu (mzeytin@ee.ryerson.ca)
%            Department of Electrical and Computer Engineering
%            Ryerson Polytechnic University
%            Toronto, Ontario, M5B 2K3 
%            CANADA
%
%    Copyright (c) 1997 Mehmet Zeytinoglu
%
%    $Revision: 1.3 $    $Date: 1997/03/11 16:18:10 $
%

if (nargin == 2)
   fs = 2;
end

%==============================================
%  Check the input parameters for consistency
%==============================================

if ( df < 0 )
   error('Lower and upper edge frequencies must be positive.');
end
if ( fs < 0 )
   error('Sampling frequency must be positive.');
end
if ( df > fs/4 )
   fprintf(['\07' ...
      'Lower edger frequency (%g) must be less than (1/4) times the\n' ...
      'sampling frequency (%g).\n'], df, fs );
   return;
end
if ( deltadB >= 0 )
   error('The peak error (expressed in dB) must be negative.');
end


df = df / (fs/2);
Nmin = 3;
Nmax = 128;

%==================
% CASE 1: df = 0.01
%==================

N = [-130/45 -130/82 -78/120 -37/120]*deltadB;

%============================
%  Interpolate if necessary
%============================

if ( df < 0.01 ) 

   N = min( 128 + df*(N(1)-128)/0.01, Nmax);

elseif ( df == 0.01 ) 

   N = N(1);

elseif ( df < 0.02 ) 

   N = N(1) + (df-0.01)*((N(2)-N(1))/0.01);

elseif ( df == 0.02 ) 

   N = N(2);

elseif ( df < 0.05 ) 

   N = N(2) + (df-0.02)*((N(3)-N(2))/0.03);

elseif ( df == 0.05 ) 

   N = N(3);

elseif ( df < 0.10 ) 

   N = N(3) + (df-0.05)*((N(4)-N(3))/0.05);

elseif ( df == 0.10 ) 

   N = N(4);

elseif ( df > 0.10 ) 

   N = max( Nmin, N(4) + (df-0.10)*(-N(4)/0.05) );

end

N = limiter( ceil(N), Nmin, Nmax );
