function [ Nodd, Neven ] = difford( deltadB, fp, fs )
%DIFFORD estimates the order of optimal FIR differentiators as a function
%        of peak error (DE) and passband cut-off frequency (FP).
%
%        [ NO, NE ] = DIFFORD( DE, FP, FS ) estimates NO, the order of optimal
%        Type III (odd filter order) FIR differentiator, and NE, the
%        order of optimal Type IV (even filter order) FIR differentiator.
%        FP represents the highest frequency for which differentiator action
%        is required.  FS is the sampling frequency such that FP <= FS/2.
%        If FS is not specified as an input argument then it is assumed that
%        FS = 2.  DE represents the allowable peak relative error expressed
%        in dB.  Acceptable values for DE are between -Inf and 0.
%
%        The normalized values for FP defined as FP/(FS/2) are realistically
%        limited to between 0.8 and 1.0.  While for Type IV filters the
%        normalized FP can be exactly equal to 1.0, for Type III filters
%        the normalized FP must be strictly less than 1.0.  If Fp/(FS/2)=1,
%        then NO = 128.  (An arbitrarily selected value.)
%
%        The NO and NE values corresponding to Fp/(FS/2) = {0.8,0.9,1.0} are
%        "exact".  Other intermediate values of Fp/(FS/2) > 0.8 will result in 
%        linear interpolation between these exact values.  For Fp/(FS/2) < 0.8
%        NO = NO(0.8), and NE = NE(0.8).  Therefore, the interpolated and
%        thresholded filter order estimates must be used with caution.
%        The "exact" relations for NO and NE are based on the design curves 
%        taken from the text Rabiner and Gold (Figure 3.71 on page 169).
%
%        The NO and NE values are also bound below such that
%        as DE-->0, NO-->12 and NE-->11.

%
%    Author: Mehmet Zeytinoglu (mzeytin@ee.ryerson.ca)
%            Department of Electrical and Computer Engineering
%            Ryerson Polytechnic University
%            Toronto, Ontario, M5B 2K3 
%            CANADA
%
%    Copyright (c) 1997 Mehmet Zeytinoglu
%
%    $Revision: 1.3 $    $Date: 1997/03/11 16:18:10 $
%

if (nargin == 2)
   fs = 2;
end

%==============================================
%  Check the input parameters for consistency
%==============================================

if ( fp < 0 )
   error('Passband frequency must be positive.');
end
if ( fs < 0 )
   error('Sampling frequency must be positive.');
end
if ( fp > fs/2 )
   fprintf(['\07' ...
      'Passband frequency (%g) must be less than half the\n' ...
      'sampling frequency (%g).\n'], fp, fs );
   return;
end
if ( deltadB > 0 )
   error('The peak error (expressed in dB) must be negative.');
end


fp = fp / (fs/2);
NOmin = 11;
NEmin = 12;
NOmax = 129;

%==================
% CASE 1: fp = 0.80
%==================

Nodd  = max( NOmin, -(30/115)*(deltadB+15 ) );
Neven = max( NEmin, -(40/126)*(deltadB-8.6) );

%==================
% CASE 2: fp = 0.90
%==================

Nodd  = [ Nodd max( NOmin, -(54/112)*(deltadB+ 18 )) ];
Neven = [ Neven  max( NEmin, -(83/128)*(deltadB-4.17)) ];

%==================
% CASE 2: fp = 1.00
%==================

if ( deltadB >= -40 )
   Nodd = [ Nodd max( NOmin, -(deltadB+ 18 )) ];
else
   Nodd = [ Nodd max( NOmin, -(130/21)*(deltadB+36)) ];
end
Neven = [ Neven  128 ];

%============================
%  Interpolate if necessary
%============================

if ( fp < 0.8 ) 

   fprintf('\07 The peak relative error is outside the design range.\n');
   Nodd  = Nodd(1);
   Neven = Neven(1);

elseif ( fp == 0.8 ) 

   Nodd  = Nodd(1);
   Neven = Neven(1);

elseif ( fp < 0.90 ) 

   Nodd  = Nodd(1) + (fp-0.8)*((Nodd(2)-Nodd(1))/0.1);
   Neven = Neven(1)  + (fp-0.8)*((Neven(2) -Neven(1))/0.1);

elseif ( fp == 0.90 ) 

   Nodd  = Nodd(2);
   Neven = Neven(2);

elseif ( fp < 1.00 ) 

   Nodd  = Nodd(2) + (fp-0.90)*((Nodd(3)-Nodd(2))/0.1);
   Neven = Neven(2);

elseif ( fp == 1.00 ) 

   Nodd  = Nodd(3);
   Neven = Neven(3);

end

Neven = 2*ceil(Neven/2);
Nodd  = 2*fix(Nodd/2)+1;
