function flag = d_tspchk( FILTtypeNumber, SpecData )
%D_TSPCHK checks the values entered into the editable fields of the
%         "TYPE and SPECIFICATIONS" window for consistency.  
%
%         D_TSPCHK( FILTtypeNumber, FILTDATA ) initiates the check for
%         the filter FILTtypeNumber (FILTtypeNumber=1 -> LPF, and so on)
%         with filter specifications given in the array FILTDATA.   For
%         the contents of FILTDATA as a function of FILTtypeNumber see
%         either D_TSPSUP or D_MTDSUP.
%
%         FLAG = D_TSPCHK( ... ) return 0 if a fatal error is found, which
%         should cause an return from the calling function or it returns 
%         which may indicate that no problems have been found or only a
%         condition worthy of a WARNING has been found.  WARNINGs are either
%         self-correcting or does not cause a suspension of the calling 
%         function.   In either case D_TSPCHK makes the appropriate calls
%         to D_ERROR which prints out the error or warning explanation 
%         message to the command window.

%
%    Author: Mehmet Zeytinoglu (mzeytin@ee.ryerson.ca)
%            Department of Electrical and Computer Engineering
%            Ryerson Polytechnic University
%            Toronto, Ontario, M5B 2K3 
%            CANADA
%
%    Copyright (c) 1997 Mehmet Zeytinoglu
%
%    $Revision: 1.3 $    $Date: 1997/03/11 16:18:10 $
%

flag = 1;

%
% Sampling rate has to be positive
%=================================
fs = SpecData(1);

if ( fs <= 0 )
   d_error( 'ET-01' )
   flag = 0;
   return
end

if ( FILTtypeNumber <= 4 )   % One of four basic types: LPF-HPF-BPF-BSF

   if     ( FILTtypeNumber == 1 )
      fre_idx = [2 4];
      pas_idx =   3  ;
      stp_idx =   5  ;
   elseif ( FILTtypeNumber == 2 )
      fre_idx = [2 4];
      pas_idx =   5  ;
      stp_idx =   3  ;
   elseif ( FILTtypeNumber == 3 )
      fre_idx = [2 4 6 7];
      pas_idx =   5  ;
      stp_idx = [3 8];
   elseif ( FILTtypeNumber == 4 )
      fre_idx = [2 4 6 7];
      pas_idx = [3 8];
      stp_idx =   5  ;
   end

   fre_vector = SpecData(fre_idx);

   %
   % Any intermediate frequencies exceeding (fs/2) ?
   %================================================
   if ( any( fre_vector >= fs/2 ) )
      d_error( 'ET-02' )  
      flag = 0;
   end
   %
   % Are intermediate frequencies non-decreasing?
   %================================================
   if ( any( diff([0 fre_vector]) <= 0 ) )
      d_error( 'ET-03' )
      flag = 0;
   end

   %
   % Are Attenuation parameters non-negative ?
   %================================================
   if ( any( SpecData([pas_idx stp_idx]) <= 0 ) )
      d_error( 'ET-04' )
      flag = 0;
   end

   %
   % Are Stopband/Passband Attenuation parameters comparable ?
   %===========================================================
   if ( max(SpecData(pas_idx)) >= min(SpecData(stp_idx)) )
      d_error( 'ET-05' )
      flag = 0;
   end
   
elseif ( sum(FILTtypeNumber == [5 6]) )    % Differentiator or Hilbert Transf.

   %
   % Is the specified TYPE III or IV only ?
   %=========================================
   if ( ~sum(SpecData(2) == [3 4]) )
      d_error( 'ET-06' )
      flag = 0;
   end

   %
   % Is the specified peak (relative) error expressed in dB negative ?
   %===================================================================
   if ( SpecData(3) >= 0 )
      d_error( 'ET-07' )
      flag = 0;
   end

   if     ( FILTtypeNumber == 5 )   % diff

      %
      % While checking the Differentiator, is slope non-negative?
      %==========================================================
      if ( SpecData(5) <= 0 )
         d_error( 'ET-08' )
         flag = 0;
      end
      %
      %  Is fc <= fs/2 ?
      %============================
      if ( SpecData(4) > (fs/2) )
         d_error( 'ET-09' )
         flag = 0;
      end
      %
      % If Type III, is fc < fs/2 ?
      %============================
      if ( (SpecData(2)==3) & (SpecData(4)==fs/2) )
         d_error( 'WT-01' )
         flag = 1;
      end

   elseif ( FILTtypeNumber == 6 )   % Hilbert

      %
      %  Is 0 < fl ? 
      %==============
      if ( SpecData(4) <= 0 )
         d_error( 'ET-10' )
         flag = 0;
      end
      %
      %  Is fh <= fs/2 ?
      %============================
      if ( SpecData(5) > (fs/2) )
         d_error( 'ET-11' )
         flag = 0;
      end
      %
      % If Type III, is fh < fs/2 ?
      %============================
      if ( (SpecData(2)==3) & (SpecData(5)==fs/2) )
         d_error( 'WT-02' )
         flag = 1;
      end
      %
      %  Is fl < (fs/4) < fh ? 
      %========================
      if ( any(diff([SpecData(4) (fs/4) SpecData(5)]) <= 0) )
         d_error( 'ET-12' )
         flag = 0;
      end
      %
      %  Are fl and fh symmetric ? 
      %==========================================================
      if ( (SpecData(2)==3) & (SpecData(4)~=(fs/2-SpecData(5))) & ...
           (SpecData(5)~=fs/2) )
         d_error( 'WT-03' )
         flag = 1;
      end

   end


end
