function [x, y, z] = d_tgtgen( arg1, arg2 )
%D_TGTGEN prepares target data from filter specifications.
%
%        D_TGTGEN( FILTtype, FILTdata ) generates the target for
%        an FIR (if DFP_fir=1) or for an IIR (if DFP_fir=0) filter, with
%        specifications given in FILTdata.
%
%        D_TGTGEN generates an FIR/IIR filter target (based on DFP_fir)
%        where the FILTtype is obtained from Figure 2 Pop-Up-Menu and
%        FILTdata is determined from Figure 2 Userdata.
%
%        The output from the function are [X, Y, Z] such that:
%          X = is the matrix of x-coordinates;
%          Y = [ Ylin; Ylog ] is the augmented matrix of y-coordinates
%          Z = [fs YMINlin YMAXlin; fs YMINlog YMAXlog]
%        Each column of X corresponds to the x-coordinates of the vertices 
%        that make up the target-zone(column #);  If M = # of columns of X,
%        then Y is a (# of rows of X) x (2*M) array where the column "m" 
%        of Y is the y-coordinates of the vertices that make up the 
%        target-zone(m) on LINEAR scale and the column "m+M" is the 
%        y-coordinates of the vertices that make up the same zone on LOG-scale.
%        If for some reason the target on a LINEAR or LOG scale does not exist
%        then set the corresponding portion of Y to NaNs.

%
%    Author: Mehmet Zeytinoglu (mzeytin@ee.ryerson.ca)
%            Department of Electrical and Computer Engineering
%            Ryerson Polytechnic University
%            Toronto, Ontario, M5B 2K3 
%            CANADA
%
%    Copyright (c) 1997 Mehmet Zeytinoglu
%
%    $Revision: 1.3 $    $Date: 1997/03/11 16:18:10 $
%

%=================
% GLOBAL variables
%=================
global  DFP_Targetmax  DFP_Targetmin  DFP_fir

if ( nargin == 2 ) 
    FILTtypeNumber = arg1;
    FILTdata       = arg2;
elseif ( nargin == 0 ) 
    FILTtypeNumber = get(findobj( ...
       get(findobj(get(0,'Children'),'flat','Tag','fig2'),...
       'Children'),'flat','Style','popupmenu'), 'Value');
    FILTdata = get( findobj(get(0,'Children'),'flat','Tag','fig2'), ...
                         'Userdata' );
    FILTdata = FILTdata( FILTtypeNumber, : );
end

if ( FILTtypeNumber == 1 )  % LPF

    fs = FILTdata(1);
    fp = FILTdata(2);
    Ap = FILTdata(3);
    fc = FILTdata(4);
    Ac = FILTdata(5);

    Ymin = -( fix(abs(-Ac+DFP_Targetmin)/10)*10 );
    Ymax =  ( fix((  Ap/2+DFP_Targetmax)/10)*10 ); 

    ApLIN = 10^(( Ap/2)/20);
    AcLIN = 10^(   -Ac/20);
    YminLIN = 0;
    YmaxLIN = ceil( ApLIN*10 )/10;
    YmaxLIN = 1.5;

    x = [  0       0       fc   ;
           fp      fp      fc   ;
           fp      fp     fs/2  ];

    if ( DFP_fir )

         yLIN = [ ApLIN     1/ApLIN   YmaxLIN  ;
                  ApLIN     1/ApLIN   AcLIN    ;
                  YmaxLIN   YminLIN   AcLIN    ];

         yLOG = [ Ap/2   -Ap/2    Ymax  ;
                  Ap/2   -Ap/2   -Ac    ;
                  Ymax    Ymin   -Ac    ];
    else

         yLIN = [ 1        (1/ApLIN)^2    1.5  ;
                  1        (1/ApLIN)^2    AcLIN    ;
                  1.5      YminLIN        AcLIN    ];

         yLOG = [ 0          -Ap           Ymax  ;
                  0          -Ap          -Ac    ;
                  Ymax-Ap/2   Ymin-Ap/2   -Ac    ];

    end

elseif ( FILTtypeNumber == 2 )  % HPF

    fs = FILTdata(1);
    fc = FILTdata(2);
    Ac = FILTdata(3);
    fp = FILTdata(4);
    Ap = FILTdata(5);

    Ymin = -( fix(abs(-Ac+DFP_Targetmin)/10)*10 );
    Ymax =  ( fix((  Ap/2+DFP_Targetmax)/10)*10 ) + Ap/2;

    ApLIN = 10^(( Ap/2)/20);
    AcLIN = 10^(   -Ac/20);
    YminLIN = 0;
    YmaxLIN = ceil( ApLIN*10 )/10;
    YmaxLIN = 1.5;

    x = [  fp     fp     0    ;
           fp     fp     fc   ;
          fs/2   fs/2    fc   ];

    if ( DFP_fir )

         yLIN = [ 
                  YmaxLIN   YminLIN   AcLIN    ;
                  ApLIN     1/ApLIN   AcLIN    ;
                  ApLIN     1/ApLIN   YmaxLIN  ;
                ];

         yLOG = [ Ymax   Ymin   -Ac   ;
                  Ap/2  -Ap/2   -Ac   ;
                  Ap/2  -Ap/2    Ymax ];
    else

         yLIN = [ 
                  1.5      YminLIN        AcLIN;
                  1        (1/ApLIN)^2    AcLIN;
                  1        (1/ApLIN)^2    1.5;
                ];

         yLOG = [ Ymax-Ap/2   Ymin-Ap/2   -Ac   ;
                  0          -Ap          -Ac   ;
                  0          -Ap           Ymax ];

    end

elseif ( FILTtypeNumber == 3 )  % BPF

    fs  = FILTdata(1);
    fc1 = FILTdata(2);
    Ac1 = FILTdata(3);
    fp1 = FILTdata(4);
    Ap  = FILTdata(5);
    fp2 = FILTdata(6);
    fc2 = FILTdata(7);
    Ac2 = FILTdata(8);

    Ymin = min( -fix(abs(-Ac1+DFP_Targetmin)/10)*10 , ...
                -fix(abs(-Ac2+DFP_Targetmin)/10)*10 );
    Ymax =  ( fix((  Ap/2+DFP_Targetmax)/10)*10 ) + Ap/2;

    ApLIN  = 10^(( Ap/2)/20);
    Ac1LIN = 10^(   -Ac1/20);
    Ac2LIN = 10^(   -Ac2/20);
    YminLIN = 0;
    YmaxLIN = ceil( ApLIN*10 )/10;
    YmaxLIN = 1.5;

    x = [ fp1    fp1     0     fc2   ;
          fp1    fp1     0     fc2   ;
          fp2    fp2    fc1    fs/2  ;
          fp2    fp2    fc1    fs/2  ];

    if ( DFP_fir )

       yLOG = [ Ymax   Ymin  -Ac1    Ymax  ;
                Ap/2  -Ap/2  -Ac1   -Ac2   ;
                Ap/2  -Ap/2  -Ac1   -Ac2   ;
                Ymax   Ymin   Ymax  -Ac2   ];

       yLIN = [ YmaxLIN   YminLIN  Ac1LIN   YmaxLIN  ;
                 ApLIN    1/ApLIN  Ac1LIN   Ac2LIN   ;
                 ApLIN    1/ApLIN  Ac1LIN   Ac2LIN   ;
                YmaxLIN   YminLIN  YmaxLIN  Ac2LIN   ];

    else

       yLOG = [ Ymax   Ymin   -Ac1   Ymax  ;
                  0    -Ap    -Ac1   -Ac2  ;
                  0    -Ap    -Ac1   -Ac2  ;
                Ymax   Ymin   Ymax   -Ac2  ];

       yLIN = [  YmaxLIN   YminLIN       Ac1LIN    YmaxLIN  ;
                   1       (1/ApLIN)^2   Ac1LIN    Ac2LIN   ;
                   1       (1/ApLIN)^2   Ac1LIN    Ac2LIN   ;
                 YmaxLIN   YminLIN       YmaxLIN   Ac2LIN   ];
    end


elseif ( FILTtypeNumber == 4 )  % BSF

    fs  = FILTdata(1);
    fp1 = FILTdata(2);
    Ap1 = FILTdata(3);
    fc1 = FILTdata(4);
    Ac  = FILTdata(5);
    fc2 = FILTdata(6);
    fp2 = FILTdata(7);
    Ap2 = FILTdata(8);

    Ymax = max( fix((Ap1/2+DFP_Targetmax)/10)*10 + Ap1/2 , ...
                fix((Ap2/2+DFP_Targetmax)/10)*10 +Ap2/2  );
    Ymin = -fix(abs(-Ac+DFP_Targetmin)/10)*10;

    Ap1LIN  = 10^(( Ap1/2)/20);
    Ap2LIN  = 10^(( Ap2/2)/20);
    AcLIN = 10^(   -Ac/20);
    Ac2LIN = 10^(   -Ac2/20);
    YminLIN = 0;
    YmaxLIN = ceil( ApLIN*10 )/10;
    YmaxLIN = 1.5;

    x = [  0       0      fc1    fp2     fp2    ;
          fp1     fp1     fc1    fp2     fp2    ;
          fp1     fp1     fc2    fs/2    fs/2   ;
          fp1     fp1     fc2    fs/2    fs/2   ];

    if ( DFP_fir )

       yLOG = [ Ap1/2   -Ap1/2    Ymax    Ymax     Ymin   ;
                Ap1/2   -Ap1/2    -Ac     Ap2/2   -Ap2/2  ;
                Ymax     Ymin     -Ac     Ap2/2   -Ap2/2  ;
                Ymax     Ymin     Ymax    Ap2/2   -Ap2/2  ];

       yLIN = [ Ap1LIN   1/Ap1LIN  YmaxLIN  YmaxLIN   YminLIN;
                Ap1LIN   1/Ap1LIN   AcLIN    Ap2LIN   1/Ap2LIN;
                YmaxLIN  YminLIN    AcLIN    Ap2LIN   1/Ap2LIN;
                YmaxLIN  YminLIN   YmaxLIN   Ap2LIN   1/Ap2LIN ];

    else

       yLOG = [   0    -Ap1    Ymax   Ymax   Ymin ;
                  0    -Ap1    -Ac     0     -Ap2 ;
                Ymax   Ymin    -Ac     0     -Ap2 ;
                Ymax   Ymin    Ymax    0     -Ap2 ];


       yLIN = [   1      (1/Ap1LIN)^2   YmaxLIN  YmaxLIN    YminLIN    ;
                  1      (1/Ap1LIN)^2    AcLIN    1       (1/Ap2LIN)^2 ;
                YmaxLIN     YminLIN      AcLIN    1       (1/Ap2LIN)^2 ;
                YmaxLIN     YminLIN     YmaxLIN   1       (1/Ap2LIN)^2 ];

    end

elseif ( FILTtypeNumber == 5 )  % Differentiator

    fs  = FILTdata(1);
    ty  = FILTdata(2);
    er  = 10^(FILTdata(3)/20);
    fp  = FILTdata(4);
    sl  = FILTdata(5);

    YmaxLIN = 2*sl; %10^( DFP_Targetmax/20 ); 
    Ymax = DFP_Targetmax;
    Ymin = DFP_Targetmin;

    sl = sl*fp/(fs/2);
    x = [  0    0  ;
          fp   fp  ;
          fp  fs/2 ];

    yLIN = [ 0            0      ;
             sl*(1+er) sl*(1-er) ;
             Ymax         0      ];

    yLOG = NaN * yLIN;

elseif ( FILTtypeNumber == 6 )  % Hilbert Transformer

    fs  = FILTdata(1);
    ty  = FILTdata(2);
    er  = FILTdata(3);
    fl  = FILTdata(4);
    fh  = FILTdata(5);
     
    erLIN = 10^( er/20 );
    er    = max(abs(20*log10([ 1+erLIN 1-erLIN]))); 

    Ymax = ( fix((er+DFP_Targetmax)/10)*10 );
    Ymin = DFP_Targetmin;
    YminLIN = 0;
    YmaxLIN = ceil( 1+erLIN );

    x = [ fl  fl ;
          fl  fl ;
          fh  fh ;
          fh  fh ];

    yLIN = [ YmaxLIN  YminLIN ;
             1+erLIN  1-erLIN ;
             1+erLIN  1-erLIN ;
             YmaxLIN  YminLIN ]; 
    yLOG = [ Ymax     Ymin    ;
              er       -er    ;
              er       -er    ;
             Ymax     Ymin    ]; 

end

y = [ yLIN; yLOG ];
z = [ fs 0 YmaxLIN; fs Ymin Ymax ];
