function [out,OverFlow] = d_quant(in,NumberBitsArg,ModeArg,CoeffArg)
%D_QUANT ..... Maps an analog sequence sequence into a quantized sequence.
%
%D_QUANT(X,N)  Takes the analog input sequence (analog within the limits
%              of machine accuracy) and quantizes each element of X using an
%              N-bit, mid-rise type uniform quantizer. Number of quantization
%              levels is 2^N.  The quantizer range is the interval [-1, 1).  
%              The minimum quantization level is -1, and the maximum level 
%              equals 1-(2/2^N).  Xq is the array that contains the quantized 
%              values corresponding to the elements in X.
%
%D_QUANT(X,[N1 N2]) quantizes the array X using N1 total bits with N2 bits
%              representing the integer part.  Number of quantization levels
%              is 2^N1. The quantizer range is the interval 2^N2*[-1,1).
%
%D_QUANT(X,[N1 N2],MODE) uses the quantizer type MODE.  MODE options are
%                    'rounding'   -->  mid-riser type quantization (default)
%                    'truncation' -->  "TWO's COMPLEMENT with TRUNCATION"
%                    'mid-riser'  -->  "TWO's COMPLEMENT with ROUNDING"
%
%D_QUANT(X,[N1 N2],MODE,COEFF) uses the quantizer type MODE where the 
%              quantizer coefficients are to be represented in COEFF mode.
%              Valid COEFF mode options ore: 
%                    'unsigned' -> quantizer range [   0 , 2^N2)
%                    'twoscomp' -> quantizer range [-2^N2, 2^N2) (default)

%
%    Author: Mehmet Zeytinoglu (mzeytin@ee.ryerson.ca)
%            Department of Electrical and Computer Engineering
%            Ryerson Polytechnic University
%            Toronto, Ontario, M5B 2K3 
%            CANADA
%
%    Copyright (c) 1997 Mehmet Zeytinoglu
%
%    $Revision: 1.3 $    $Date: 1997/03/11 16:18:10 $
%

%===========================================================================
% Modifications history:
% ----------------------
%        o   Added 'scaling' input parameter. 06.06.91 MZ
%        o   Changed input N to "quantization levels". 06.17.91 MZ
%        o   Changed input N to "bits/sample". 06.19.91 MZ
%        o   Added "checking"  11.30.1992 MZ
%        o   Tested (and modified) under MATLAB 4.0/4.1 08.16.1993 MZ
% ------------------------- Version 2.0
%       o   Complete rewrite of the quantization routine, infinitely
%           faster than version 1.0.  Added 'rounding', 'truncation'
%           'integer bits' options, removed restriction on the number 
%           of bits.   09.08.1995 MZ
%        o   Added "unsigned integer" option 02.25.1996 MZ
%        o   Changed name from "quantize" to "d_quant" to avoid naming
%            conflicts with simulink/blocks/quantize.m/quantize.mex??
%            12.08.96 MZ
%===========================================================================

%---------------------------------------------------------------------------
%        Default values
%---------------------------------------------------------------------------

IntegerBitsDefault = 0;
ModeDefault        = 'rounding';
CoeffDefault       = 'twoscomp';

%---------------------------------------------------------------------------
%        Input parameter check and parameter set-up
%---------------------------------------------------------------------------

if ( length(NumberBitsArg) == 1 )
   IntegerBits = IntegerBitsDefault;
   TotalBits   = NumberBitsArg;
elseif ( length(NumberBitsArg) == 2)
   TotalBits   = NumberBitsArg(1);
   IntegerBits = NumberBitsArg(2);
end

if (nargin == 2)
   Mode      = ModeDefault;
   CoeffMode = CoeffDefault;
elseif (nargin == 3)  
   Mode = ModeArg;
   CoeffMode = CoeffDefault;
elseif (nargin == 4)  
   Mode = ModeArg;
   CoeffMode = CoeffArg;
end

Range    = 2^IntegerBits;
StepSize = 2*Range/(2^TotalBits);

%---------------------------------------------------------------------------
%        DC shift input samples if "unsigned" quantization is requested
%       will shift the DC level back top where it was after quantization.
%---------------------------------------------------------------------------

if (strcmp(CoeffMode,'unsigned'))  

    in = in - Range;

end

%---------------------------------------------------------------------------
%        If there are any overflows: How many?
%---------------------------------------------------------------------------

OverFlow = sum(sum( abs(in)>Range ));

%---------------------------------------------------------------------------
%        Now let us quantize the samples
%---------------------------------------------------------------------------

if (strcmp(Mode,'rounding'))       

    in = limiter(in,-Range,Range-StepSize);
    ss = [(in >= 0) - (in < 0)];
    out = ss .* ( round(abs(in)/StepSize)*StepSize  );

elseif (strcmp(Mode,'mid-riser'))  

    in = limiter(in,-Range,Range-eps);
    ss = [(in >= 0) - (in < 0)];
    out = ss .* ( fix(abs(in)/StepSize)*StepSize + StepSize/2 );
    out = limiter(out,-Range+StepSize/2,Range-StepSize/2);

elseif (strcmp(Mode,'truncation'))  

    in = limiter(in,-Range,Range-eps);
    ss = [(in >= 0) - (in < 0)];
    out = ss .* ( fix(abs(in)/StepSize)*StepSize + StepSize/2 );
    idxn = (in < 0);
    idx = (fix(abs(in(idxn))/StepSize)*StepSize == (abs(in(idxn))));
    out(find(idx)) = out(find(idx)) + StepSize;
    out = out - StepSize/2;

else

    error('Unknown quantization Mode.');
    return;

end

if (strcmp(CoeffMode,'unsigned'))  

    out = out + Range;

end

