function FIGhan = d_navgtr()
%D_NAVGTR creates and initializes the "NAVIGATOR" window and its children.
%
%         FIG = D_NAVGTR returns the handle to the figure object which has 
%         been created by calling the function D_NAVGTR.

%
%    Author: Mehmet Zeytinoglu (mzeytin@ee.ryerson.ca)
%            Department of Electrical and Computer Engineering
%            Ryerson Polytechnic University
%            Toronto, Ontario, M5B 2K3 
%            CANADA
%            
%    Copyright (c) 1997 Mehmet Zeytinoglu
%
%    $Revision: 1.4 $    $Date: 1997/03/11 16:18:10 $
%

%=================
% GLOBAL variables
%=================
global  DFP_ArrowColor1      DFP_ArrowColor2       DFP_FIGbgroundcolor ...
        DFP_Frame2button     DFP_UIcontrolBox_X    DFP_UIcontrolBox_Y  ...
        DFP_WinBorderOffset  DFP_Offset1           DFP_Offset2         ...
        DFP_Linewidth        DFP_FigureResize      DFP_ScaleX          ...
        DFP_ScaleY

%==============
% FIGURE NUMBER
%==============
FigNo = 1;

%==============================
% Dimension definitions (local)
%==============================
CBx  = DFP_UIcontrolBox_X;       % Checkbox (CB) x-dim
CBy1 = DFP_UIcontrolBox_Y;       % standard CB y-dim
CBy  = 2*CBy1;                   % combined "Coeef/PoleZero" CB y-dim
PBy  = 1.5*DFP_UIcontrolBox_Y;   % pushbutton (PB) y-dim on the initial screen
PBy1 =     DFP_UIcontrolBox_Y;   % standard PB y-dim

Dim1  = 4*DFP_Frame2button;      % FR        to l/r FIG edge
Dim2  = 4*DFP_Frame2button;      % FR( [2] ) to b/t FIG edge
Dim21 = (CBy - CBy1)/2 + Dim2;   % FR([1 3]) to b/t FIG edge
Dim3  = 1*DFP_Frame2button;      % CB to b/t FR edge
Dim4  = 1*DFP_Frame2button;      % CB to l/r FR edge
Dim5  = 8*DFP_Frame2button;      % FR separation
Dim9  = 0;                       % Status text frame height in pixels
Dim10 = 20*DFP_ScaleY;           % separation between turques and yellow lines

FRx  = CBx +2*Dim4;              % Frame (FR) x-dim
FRy  = PBy1+2*Dim3+2*CBy1+CBy+(2*DFP_ScaleY); % FR( [2])   y-dim
FRy1 = PBy1+2*Dim3+3*CBy1    +(2*DFP_ScaleY); % FR([1 3])  y-dim

PBx  = FRx;                      % PB (on the initial screen) x-dim     
PBx1 = FRx - (3*DFP_ScaleX);     % PB x-dim on top of the FRs (3 is empirical)

FIGx = 3*FRx+2*Dim5+2*Dim1;      % Figure (FIG) x-dim
FIGy =   FRy+2*Dim2;             % FIG y-dim

%===============================================================
%  Now that we have defined all specified/derived dimensions
%  let us proceed to place and define other properties of the
%  UICONTROL objects.
%     Remark:  To be able to turn object visibility on/off
%              in groups we use the following TAGS:
%                grp(k):  Frame(k) and all the CBs on it k=1,2,3
%                grp5  :  PB(1) (DESIGN)    to control grp1
%                grp6  :  PB(2) (DISPLAY)   to control grp2
%                grp7  :  PB(3) (IMPLEMENT) to control grp3
%              Some of the line/arrow elements will be attached 
%              to appropriate group TAGS to achieve the desired
%              visual effect with the connections. 
%     Remark:  All positions are to be defined relative to the
%              FR positions.
%===============================================================

%  FRAME objects
%===============
FRpos = [ 
          Dim1               Dim9+Dim21  FRx  FRy1 ;
          Dim1+  (Dim5+FRx)  Dim9+Dim21  FRx  FRy1 ;
          Dim1+2*(Dim5+FRx)  Dim9+Dim21  FRx  FRy1 ;
        ];

FRtag = str2mat( 'grp1', 'grp2', 'grp3' );


%  PUSHBUTTON objects
%====================
Dim6  = (FIGy+Dim9-PBy)/2;            % (FIG bottom)->(big-PB bottom)
Dim7  = FRpos(2,2)+2*Dim3+2*CBy1+CBy; % (FIG bottom)->(top-small-PB on FR([2])
Dim71 = FRpos(1,2)+2*Dim3+3*CBy1;     % (FIG bottom)->(top-small-PB on FR([1 3])

ax    = 2 * DFP_ScaleX;               % ax (=2*scale) and ay (=1*scale) are
ay    = 1 * DFP_ScaleY;               % empirical and for asthetic reasons
PBpos = [ 
          FRpos(1,1)    Dim6     PBx  PBy  ;
          FRpos(2,1)    Dim6     PBx  PBy  ;
          FRpos(3,1)    Dim6     PBx  PBy  ;
          ...
          FRpos(1,1)+ax Dim71+ay PBx1 PBy1 ;
          FRpos(2,1)+ax Dim71+ay PBx1 PBy1 ;
          FRpos(3,1)+ax Dim71+ay PBx1 PBy1 ;
        ];
PBstr = str2mat( ...
                 'DESIGN' , 'DISPLAY' , 'IMPLEMENT' , ...
                 'FILTER' , 'DISPLAY' , 'IMPLEMENT'  ...
               );
PBtag = str2mat( 'grp5', 'grp6', 'grp7', 'grp1', 'grp2', 'grp3' );


%  CHECKBOX objects
%==================
Dim8  = FRpos(1,2)+Dim3+2*CBy1;      % (FIG bottom)->(top-CB on FR([1 3])
Dim81 = FRpos(3,2)+Dim3+1.5*CBy1;

CBpos = [ 
          FRpos(1,1)+Dim4   Dim81         CBx  CBy1 ;
          FRpos(1,1)+Dim4   Dim81-CBy1    CBx  CBy1 ;
          ...
          FRpos(2,1)+Dim4   Dim8-CBy1*2  CBx  CBy1 ;
          ...
          FRpos(3,1)+Dim4   Dim81        CBx  CBy1 ;
          FRpos(3,1)+Dim4   Dim81-CBy1   CBx  CBy1 ;
        ];  
CBusr = str2mat( ...
                 'fig2' , 'fig3' , ...
                 'poze'  ,         ...
                 'fig5' , 'fig6'   ...
                 );
CBstr = str2mat( ...
                 'Type/Spec'   , 'Method' , ...
                 'PoZe/Coeff',              ...
                 'Real/Quant', 'O/P Format' ...
                 );
        s1 = 'grp1'; s2 = 'grp2'; s3 = 'grp3';
CBtag = [ s1(ones(1,2),:); s2; s3(ones(1,2),:) ];


%  POP-UP MENU objects
%=====================
PUpos = [ FRpos(2,1)+Dim4   Dim8-CBy1  CBx  CBy1 ];
PUstr = str2mat( 'Phase|GDelay|Impulse' );
PUtag = 'grp2';
PUusr = 'fig1:popup';


%  TEXT objects
%=================================================================
%       Remark:  I assume that if I leave DFP_Offset1 (=2 pixels)
%                between the box surrounding the TEXT object and 
%                the boundaries of FRAME and CHECKBOXES
%=================================================================
TXpos = PUpos + [  0  CBy1  0  0  ];
TXstr = 'Magnitude';
TXtag = 'grp2';


%  Create FIGURE
%===============
ScreenSize = get(0,'Screensize');
FIGpos = [ DFP_WinBorderOffset   ScreenSize(4)-FIGy-DFP_WinBorderOffset ];
FIGhan = figure( ...
                 'Position',     [FIGpos FIGx FIGy],     ...
                 'Name',          d_namesv(FigNo), ...
                 'MenuBar',      'none',                 ...
                 'Resize',       'off',                  ...
                 'BackingStore', 'off',                  ...
                 'NumberTitle',  'off',                  ...
                 'Visible',      'off',                  ...
                 'Resize',        DFP_FigureResize,      ...
                 'Tag',          ['fig' num2str(FigNo)], ...
                 'NextPlot',     'new'                   ...
               );


%========================================================
%  Now the FIGURE has been generated let us now start
%  placing the UIMENU and UICONTROL objects
%========================================================

%  UIMENU object
%===============
MNhan = uimenu( FIGhan, ...
                                 'Label',         'File',          ...
                                 'Accelerator',   'F'                  );
                uimenu( MNhan,   'Label',         'Open ...',      ...
                                 'CallBack',      'd_fileio(0);'       );
                uimenu( MNhan,   'Label',         'Save ...',      ...
                                 'Enable',        'off',           ...
                                 'Tag',           'fig1:uimenu:save', ...
                                 'CallBack',      'd_fileio(1);'       );
                uimenu( MNhan,   'Label',         'Clear ',        ...
                                 'CallBack',      'd_fileio(2);'       );
                h = uimenu(MNhan,'Label',         'Properties',    ...
                                 'Separator',     'on'                 );
                            uimenu( h,'Label',    'Customize ... ',...
                                 'CallBack',      'd_propts;'          );
                            uimenu( h,'Label',    'Default values ',...
                                 'CallBack',      'd_rstdef(0); d_propsv;');
                uimenu( MNhan,   'Label',         'Close DFP',     ...
                                 'Separator',     'on',            ...
                                 'CallBack',      'd_fileio(3);'       );
                uimenu( MNhan,   'Label',         'Exit MATLAB   ',...
                                 'Separator',     'on',            ...
                                 'CallBack',      'exit'               );
        uimenu( FIGhan, ...
                                 'Label',         'Window',        ...
                                 'Tag',           'winmenu',       ...
                                 'CallBack',       winmenu('callback'),...
                                 'Accelerator',   'W'                    );
        h = uimenu( FIGhan, ...
                                 'Label',         'Help',          ...
                                 'CallBack',      '',              ...
                                 'Tag',           'help',          ...
                                 'Accelerator',   'H'                    );
                uimenu( h,       'Label',         'About ...  ',   ...
                                 'CallBack',      'd_logo'               );
drawnow

%  FRAME objects
%===============
VisMatrix = str2mat( 'off' , 'off' , 'off' , 'on' );
for ii = 1:size(FRpos,1)
         uicontrol( FIGhan,                        ...
              'Style',           'Frame',          ...
              'Tag',              FRtag(ii,:),     ...
              'Visible',          VisMatrix(ii,:), ...
              'Position',         FRpos(ii,:)      ...
              );
end

%  PUSHBUTTON objects
%====================
VisMatrix = str2mat( 'on', 'on', 'on', 'off' , 'off' , 'off' , 'off' );
EnbMatrix = str2mat( 'on', 'on', 'on', 'on'  , 'on'  , 'on'  , 'off' );
for ii = 1:size(PBpos,1)
         uicontrol( FIGhan,                                   ...
              'Style',               'pushbutton',            ...
              'String',               deblank(PBstr(ii,:)),   ...
              'Tag',                  PBtag(ii,:),            ...
              'CallBack',             sprintf('d_navgcb(%i);',ii), ...
              'Visible',              VisMatrix(ii,:),        ...
              'Enable',               EnbMatrix(ii,:),        ...
              'Position',             PBpos(ii,:)             ...
              );
end

%  CHECKBOX objects
%==================
for ii = 1:size(CBpos,1)
         uicontrol( FIGhan,                                   ...
              'Style',               'checkbox',              ...
              'String',               deblank(CBstr(ii,:)),   ...
              'HorizontalAlignment', 'left',                  ...
              'Tag',                  CBtag(ii,:),            ...
              'CallBack',             sprintf('d_navgcb(%i);',6+ii), ...
              'UserData',             deblank(CBusr(ii,:)),   ...
              'Visible',             'off',                   ...
              'Position',             CBpos(ii,:)             ...
              );
end

%  POP-UP MENU objects
%=====================
         uicontrol( FIGhan,                                   ...
              'Style',               'popupmenu',             ...
              'HorizontalAlignment', 'center',                ...
              'String',               PUstr,                  ...
              'Tag',                  PUtag,                  ...
              'CallBack',            'd_dplydr([0 NaN 0])', ...
              'Userdata',             PUusr,                  ...
              'Visible',             'off',                   ...
              'Position',             PUpos                   ...
              );                    

%  TEXT objects
%==============
         uicontrol( FIGhan,                                   ...
              'Style',               'text',                  ...
              'HorizontalAlignment', 'center',                ...
              'String',               TXstr,                  ...
              'Tag',                  TXtag,                  ...
              'Visible',             'off',                   ...
              'Position',             TXpos                   ...
              );

%  LINE and PATCH (=arrow) objects
%====================================================================
%  We first define the co-ordinates of the start and stop points
%  which are to be connected by LINES and ARROWS.  I did not try 
%  to define the points as a matrix as the LINE and ARROW functions
%  have different argument formats.
%  Therefore: 1. Define points co-ordinates Pkx, Pky, k =1,...14
%             2. Create argument matrices for LINE and ARROW
%                These matrices must be _normalized_ by FIGx and FIGy
%                as the axis limits are Xlim = [0 1], YLim = [0 1].
%             3. Create AXES and then draw the LINE and ARROW objects
%====================================================================
P1x  = Dim1 + FRx/2;               P1y  = Dim6 + PBy;
P2x  = P1x;                        P2y  = Dim6 + PBy + Dim10;
P3x  = Dim1 + 2  *FRx + 2*Dim5;    P3y  = P2y;
P4x  = Dim1 + 2.5*FRx + 2*Dim5;    P4y  = P2y;
P5x  = P4x;                        P5y  = P1y;
P6x  = P4x;                        P6y  = Dim6;
P7x  = P4x;                        P7y  = Dim6 - Dim10;
P8x  = Dim1 + FRx;                 P8y  = P7y;
P9x  = P1x;                        P9y  = P7y;
P10x = P1x;                        P10y = Dim6;
P11x = P8x;                        P11y = Dim6 + PBy/2;
P12x = Dim1 +   FRx +   Dim5;      P12y = P11y; 
P13x = Dim1 + 2*FRx +   Dim5;      P13y = P11y;
P14x = Dim1 + 2*FRx + 2*Dim5;      P14y = P11y;

%  ARROW definitions
%===================
ARROWstart = [ 
                P2x      P2y  ;
                P4x      P4y  ;
                P7x      P7y  ;
                P9x      P9y  ;
                P11x     P11y ;
                P14x     P14y ] * [ (1/FIGx) 0; 0 (1/FIGy) ];
ARROWstop  = ([  
                P3x-2    P3y  ;    %====================================
                P5x      P5y+2;    % These +/- 2 or other are for hand-
                P8x      P8y  ;    % tuning the arrow positions, just
                P10x     P10y ;    % for appearance.
                P12x-4   P12y ;    %====================================
                P13x     P13y ] - (ones(6,1)*[0*DFP_Linewidth 0])) ...
                                * [ (1/FIGx) 0; 0 (1/FIGy) ];

ARROWcolor = [ 
                DFP_ArrowColor1;
                DFP_ArrowColor1;
                DFP_ArrowColor1;
                DFP_ArrowColor1;
                ...
                DFP_ArrowColor2;
                DFP_ArrowColor2;
             ];

%  LINE definitions
%==================
LINEx = (1/FIGx) * [ P1x   P2x   P6x   P7x ;
                     P2x   P4x   P7x   P9x ];
LINEy = (1/FIGy) * [ P1y   P2y   P6y   P7y ;
                     P2y   P4y   P7y   P9y ];

%  Create AXES
%=============
axes( ...
                'Position',  [0 0 1 1],       ...
                'Visible',  'off',            ...
                'Drawmode', 'fast',           ...
                'Xlim',      [0 1],           ...
                'Ylim',      [0 1]            ...
                );  

%  Draw ARROWs
%=============
ARROWhan = arrow(...
                'Start',     ARROWstart,      ...
                'Stop',      ARROWstop,       ...
                'Linewidth', DFP_Linewidth*ones(size(ARROWstart,1),1), ...
                'LineStyle','-',              ...
                'Length',    10*(DFP_ScaleX+DFP_ScaleY)/2,  ... 
                'Tipangle',  30,              ...
                'Color',     ARROWcolor,      ...
                'Width',     (DFP_Linewidth-2)*ones(size(ARROWstart,1),1) ...
                );
           set(ARROWhan(1),'tag','grp3','visible','off'); ...
           set(ARROWhan(3),'tag','grp1','visible','off');

%  Draw LINEs
%============
LINEhan  = line(...
                LINEx,LINEy,                 ...
               'Color',     DFP_ArrowColor1, ...
               'Visible',  'on',             ...
               'LineStyle','-',              ...
               'Linewidth', DFP_Linewidth    ...
                );
           set(LINEhan(2),'tag','grp7'); ...
           set(LINEhan(4),'tag','grp5');

%==================================================================
%  Everything is set up and drawn properly. 
%==================================================================

set( [ findobj(get(FIGhan,'Children'),'flat','Type','uicontrol'); ...
       findobj(get(FIGhan,'Children'),'flat','Type','axes')],     ...
     'Units','normalized' );
