function d_implmt()
%D_IMPLMT generates the user specified filter realisation.
%
%         When the user selects a filter design method from the "DESIGN METHOD"
%         (see the M-file D_MTDCB.M) or when a realization and/or coefficient
%         format choice is made in the "IMPLEMENTATION" window by pressing the
%         APPLY pushbutton, a call to this function is initiated.
%         
%         At the entry to D_IMPLMT function the filter's B ( = DFP_bb ) and 
%         A ( = DFP_aa ) polynomials have been calculated.  D_IMPLMT first
%         checks all the UICONTROL objects in the "IMPLEMENTATION" window.
%         It then implements the corresponding transformation, such as
%         Polynomial-->Lattice, Float-->Fixed, and then the inverse transfor-
%         mation such that the new polynomial arrays DFP_bbQ and DFP_aaQ are
%         to be used as the basis for frequency response calculations.
%
%         This function also determines which one of the coefficient windows,
%         their corresponding sliders etc will be visible.

%
%    Author: Mehmet Zeytinoglu (mzeytin@ee.ryerson.ca)
%            Department of Electrical and Computer Engineering
%            Ryerson Polytechnic University
%            Toronto, Ontario, M5B 2K3 
%            CANADA
%            
%    Copyright (c) 1997 Mehmet Zeytinoglu
%
%    $Revision: 1.3 $    $Date: 1997/03/11 16:18:10 $
%

%==================
%  GLOBAL variables
%==================
global   DFP_bb          DFP_aa            DFP_fir          ...
         DFP_bbQ         DFP_aaQ           DFP_firlat       ...
         DFP_top         DFP_bottom        DFP_gain         ...
         DFP_topsymbol   DFP_bottomsymbol  DFP_scrolltextlines 

FIG5han  =      findobj(get(0,'children'),'flat','Tag','fig5');
CHLDhan  = get(FIG5han,'children');
RBhan    = sort(findobj(CHLDhan,'flat','Style','radiobutton','Tag','format'));
SELhan   =      findobj(RBhan,'Flat','Value',1);
RealType = find( RBhan==SELhan );   %=== Realization Format ===%
CoefType = get(findobj(CHLDhan,'flat','Style','popupmenu'),'Value'); 

FIG4han = findobj(get(0,'children'),'flat','Tag','fig4');
SLhan   = sort(findobj(get(FIG4han,'children'),'flat','Style','slider'));
SThan   = sort(findobj(get(FIG4han,'children'),'flat','Style','edit')  );
LBhan   = sort(findobj(get(FIG4han,'children'),'flat','Style','text', ...
                       'Tag','fig4:label')  );
TXhan   = sort(findobj(get(FIG4han,'children'),'flat','Style','text', ...
                       'Tag','fig4:value')  );
MNhan   = sort(findobj(FIG4han,'Tag','format:coeff'));

CHD6han = get( findobj(get(0,'Children'),'flat','Tag','fig6'), 'children' );
CB6han  = sort(findobj(CHD6han, 'flat', 'Style','checkbox' ));
ET6han  = sort(findobj(CHD6han, 'flat', 'Style','edit' ));

set( get(0,'Children'), 'pointer', 'watch' ); 

STpos_top = get(SThan(1), 'position'); 
STpos_bot = get(SThan(2), 'position');
LBpos_top = get(LBhan(1), 'position'); 
LBpos_bot = get(LBhan(2), 'position'); 
TXpos_top = get(TXhan(1), 'position'); 
TXpos_bot = get(TXhan(2), 'position'); 
SLpos_bot = get(SLhan(2), 'position');

%=================================================================
%
% STEP 1:  Implement the requested REALISATION format
%               RealType == 1 --> Polynomial
%               RealType == 2 --> Lattice
%               RealType == 3 --> SoS
%          Compute the realisation coefficients. 
%
%=================================================================

if ( RealType == 1 )       % === Polynomial ===

   STpos_bot_new = [ STpos_bot(1:3) STpos_top(4) ];
   LBpos_top_new = [ LBpos_top(1:2) LBpos_bot(3) LBpos_top(4)]; 
   LBpos_bot_new = [ LBpos_bot(1) sum(STpos_bot_new([2 4])) LBpos_bot(3:4) ];
   TXpos_top_new = [ TXpos_bot(1) TXpos_top(2) TXpos_bot(3:4) ];
   TXpos_bot_new = [ TXpos_bot(1) sum(STpos_bot_new([2 4])) TXpos_bot(3:4) ];
   SLpos_bot_new = [ SLpos_bot(1:3) STpos_bot_new(4) ];

   DFP_top    = DFP_bb;  DFP_topsymbol    = ' b';  MM = length(DFP_top);
   DFP_bottom = DFP_aa;  DFP_bottomsymbol = ' a';  NN = length(DFP_bottom); 

   LBstr    = str2mat( 'Numerator Terms:', 'Denominator Terms:' );
   TXstr    = str2mat(  sprintf('%i',MM)  ,  sprintf('%i',NN)   );
   STvis    = 'on';
   SLVIShan = [1 1];
   SL2cb    = 'd_csldcb(2)';
   SLmax    = NN;

elseif ( RealType == 2 )       % === Lattice ===

   %=============================================================
   % case for a all-pole filter
   % --------------------------
   % this should ensure that for an all-pole filter the c-array
   % will equal to [ b(1) 0 .... 0 ] 
   %=============================================================
   if ( (length(DFP_bb)==1) & (length(DFP_aa)>1) )
      DFP_bb = [ DFP_bb zeros(1,length(DFP_aa)-1) ];
   end

   [k,c] = poly2lat( DFP_bb, DFP_aa );

   if ( DFP_fir ) 

      LBpos_top_new = [ LBpos_top(1:2) LBpos_bot(3)*3/5 LBpos_top(4)]; 
      LBpos_bot_new = [ LBpos_bot(1) sum(STpos_top([2 4]))-LBpos_top(4) ...
                                                           LBpos_bot(3:4)];
      TXpos_top_new = [ TXpos_bot(1)-LBpos_bot(3)*2/5 TXpos_top(2) ...
                       TXpos_bot(3)+LBpos_bot(3)*2/5 TXpos_top(4) ];
      TXpos_bot_new = [ TXpos_bot(1) sum(STpos_top([2 4]))-LBpos_top(4) ...
                                                           TXpos_bot(3:4)];
      STpos_bot_new = [ STpos_bot(1:3) LBpos_bot_new(2)-STpos_bot(2)];
      SLpos_bot_new = [ SLpos_bot(1:3) LBpos_bot_new(2)-STpos_bot(2)];

      DFP_top    = NaN;  DFP_topsymbol    = ' ';   MM = 1;
      DFP_bottom = k;    DFP_bottomsymbol = ' k';  NN = length(DFP_bottom);
      DFP_firlat = c;
      LBstr    = str2mat( 'Fwd Gain:', 'Reflection Coeff:' );
      TXstr    = str2mat(  sprintf('%.5f',DFP_firlat(1)), sprintf('%i',NN));
      STvis    = 'off';
      SLVIShan = [0 1];
      SL2cb    = 'd_csldcb(4)';
      SLmax    = NN;

   else

      STpos_bot_new = [ STpos_bot(1:3) STpos_top(4) ];
      LBpos_top_new = [ LBpos_top(1:2) LBpos_bot(3) LBpos_top(4)]; 
      LBpos_bot_new = [ LBpos_bot(1) sum(STpos_bot_new([2 4])) LBpos_bot(3:4) ];
      TXpos_top_new = [ TXpos_bot(1) TXpos_top(2) TXpos_bot(3:4) ];
      TXpos_bot_new = [ TXpos_bot(1) sum(STpos_bot_new([2 4])) TXpos_bot(3:4) ];
      SLpos_bot_new = [ SLpos_bot(1:3) STpos_bot_new(4) ];

      DFP_top    = k;  DFP_topsymbol    = ' k';  MM = length(DFP_top);
      DFP_bottom = c;  DFP_bottomsymbol = ' c';  NN = length(DFP_bottom);

      LBstr    = str2mat( 'Reflection Coeff:',     'c-Coeff:'    );
      TXstr    = str2mat(  sprintf('%i',MM)  ,  sprintf('%i',NN) );

      STvis    = 'on';
      VIShan   = [SThan; LBhan];
      SLVIShan = [1 1];
      SL2cb    = 'd_csldcb(2)';
      SLmax    = NN;

   end

elseif ( RealType == 3 )      % === SoS ===

   h = sort(findobj(CHLDhan,'flat','Style','radiobutton','Tag','gain'));
   DistributeGain = get( h(1), 'Value' );
   [z,p,k] = tf2zp( DFP_bb, DFP_aa );
   if ( DistributeGain )
      SOSarray = zp2sos( z,p,k );
      DFP_gain = 1;
   else
      SOSarray = zp2sos( z,p,1 );
      s = SOSarray(:,1);
      SOSarray(:,1:3) = SOSarray(:,1:3) ./ s(:,ones(3,1));
      DFP_gain = k;
   end

   LBpos_top_new = [ LBpos_top(1:2) LBpos_bot(3)*3/5 LBpos_top(4)]; 
   LBpos_bot_new = [ LBpos_bot(1) sum(STpos_top([2 4]))-LBpos_bot(4) ...
                                                    LBpos_bot(3:4)];
   TXpos_top_new = [ TXpos_bot(1)-LBpos_bot(3)*2/5 TXpos_top(2) ...
                       TXpos_bot(3)+LBpos_bot(3)*2/5 TXpos_top(4) ];
   TXpos_bot_new = [ TXpos_bot(1) sum(STpos_top([2 4]))-LBpos_bot(4) ...
                                                    TXpos_bot(3:4)];
   STpos_bot_new = [ STpos_bot(1:3) LBpos_bot_new(2)-STpos_bot(2)];
   SLpos_bot_new = [ SLpos_bot(1:3) LBpos_bot_new(2)-STpos_bot(2)];

   DFP_top    = [NaN];     DFP_topsymbol    = ' ';  MM = 1;
   DFP_bottom = SOSarray;  DFP_bottomsymbol = ' ';  Ns = size(DFP_bottom,1);
                                                    NN = Ns*7;
   LBstr    = str2mat(     'Filter Gain:'    ,    'SOS Sections:' );
   TXstr    = str2mat( sprintf('%.5f',DFP_gain),   sprintf('%i',Ns) );
   STvis    = 'off';
   SLVIShan = [0 1];
   SL2cb    = 'd_csldcb(3)';
   SLmax    = Ns;

end

%=====================================================================
%
% STEP 2:  Coefficient Quantization if required
%               CoefType == 1 --> floating point (don't do anything)
%               CoefType == 2 --> fixed point
%
%=====================================================================

d_setlbl(3, RealType);

if ( CoefType == 1 )

   d_setlbl(4,1);
   set( MNhan(2:4),'Enable','off');

elseif ( CoefType == 2 )           % === Fixed Point ===

   RBhan  = sort(findobj(CHLDhan,'flat','Style','radiobutton','Tag','fixed'));
   EThan  = sort(findobj(CHLDhan,'flat','Style','edit'));
   Ntotal = str2num(get(EThan(1),'String'));
   if ( isempty(Ntotal) )
      d_error( 'EQU-01' );
      set( get(0,'Children'), 'pointer', 'arrow' ); 
      return;
   end
   Nint   = str2num(get(EThan(2),'String'));
   if ( isempty(Nint) )
      d_error( 'EQU-02' );
      set( get(0,'Children'), 'pointer', 'arrow' ); 
      return;
   end
   if ( (Ntotal<=0) | (Nint<0) | (Nint>Ntotal) | ...
      (Ntotal ~= fix(Ntotal)) | (Nint ~= fix(Nint)) )
      d_error( 'EQU-03' );
      set( get(0,'Children'), 'pointer', 'arrow' ); 
      return;
   end

   Tsc    = get(RBhan(1),'Value');
   if ( Tsc )
      flag = 'twoscomp';
      str  = sprintf('Coeff: [%i,%i,%s]',Ntotal,Nint,'t');
   else
      flag = 'unsigned';
      str  = sprintf('Coeff: [%i,%i,%s]',Ntotal,Nint,'u');
   end
   [ DFP_top   , N1 ] = d_quant(DFP_top,   [Ntotal Nint],'rounding',flag);
   [ DFP_bottom, N2 ] = d_quant(DFP_bottom,[Ntotal Nint],'rounding',flag);
   TotalOverFlow      = N1 + N2;
   if ( TotalOverFlow )
      d_error( 'WQU-01', TotalOverFlow, Nint, flag );
   end
   if ( RealType == 2 & DFP_fir ) 
      [DFP_firlat(1),N1]=d_quant(DFP_firlat(1),[Ntotal Nint],'rounding',flag);
      if ( N1 )
         d_error( 'WQU-01', N1, Nint, flag );
      end
      TXstr = str2mat(  sprintf('%.5f',DFP_firlat(1)), sprintf('%i',NN));
   elseif ( RealType == 3 ) 
      %===============================================
      % is this (not to quantize DFP_gain) OK ? verify
      %===============================================
      %DFP_gain = d_quant(DFP_gain,[Ntotal Nint],'rounding',flag);
      TXstr = str2mat( sprintf('%.5f',DFP_gain),   sprintf('%i',Ns) );
   end
   d_setlbl(4,str); 
   set( MNhan(2:4),'Enable','on');

end

%=====================================================================
%
% STEP 3:  Inverse transform step.  Take the quantized DFP_top and
%          DFP_bottom, and convert back to DFP_bbQ and DFP_aaQ.
%          these will be the basis for frequency response displays.
%        
%          Also in this STEP we enable/disable the CBs and ETs in the
%          OUTPUT window based on RealType.
%
%=====================================================================

if ( RealType == 1 )

   DFP_bbQ = DFP_top;
   DFP_aaQ = DFP_bottom;
   set( [CB6han([2 3]); ET6han([2 3])], 'Enable', 'off' );

elseif ( RealType == 2 )

   if ( DFP_fir ) 
      [ DFP_bbQ, DFP_aaQ ] = lat2poly( DFP_bottom, DFP_firlat );
   else
      [ DFP_bbQ, DFP_aaQ ] = lat2poly( DFP_top, DFP_bottom );
   end
   set( [CB6han(2); ET6han(2)], 'Enable', 'on'  );
   set( [CB6han(3); ET6han(3)], 'Enable', 'off' );
 
elseif ( RealType == 3 )

   [z,p,k] = sos2zp( DFP_bottom );
   if ( DistributeGain )
      gain = k;
   else
      gain = DFP_gain;
   end
   [ DFP_bbQ, DFP_aaQ ] = zp2tf( z, p, gain );

   %===================================================================
   % The next two steps, particularly with the DFP_aaQ is necessary, as
   % earlier we converted a = [1] to a = [1 0 0 ... 0], now eliminate.
   %===================================================================
   DFP_bbQ = DFP_bbQ(1:length(DFP_bb));
   DFP_aaQ = DFP_aaQ(1:length(DFP_aa));
   set( [CB6han(2); ET6han(2)], 'Enable', 'off' );
   set( [CB6han(3); ET6han(3)], 'Enable', 'on'  );

end

%=====================================================================
%
% STEP 4:  set-up the "scrollable" text fields of "fig4" according
%          the parameters and the "realisation" format selected.
%          Also, reset all the CBs in the OUTPUT window, as this is
%          a new design.  Change the "Generate Filter Code" pop-up
%          menus to reflect the new design.
%
%=====================================================================

idx = ( [MM NN] > DFP_scrolltextlines*[1 1] );

set( LBhan(1), 'String',  deblank(LBstr(1,:)), 'Position',  LBpos_top_new );
set( LBhan(2), 'String',  deblank(LBstr(2,:)), 'Position',  LBpos_bot_new );

set( TXhan(1), 'String',  deblank(TXstr(1,:)), 'Position',  TXpos_top_new );
set( TXhan(2), 'String',  deblank(TXstr(2,:)), 'Position',  TXpos_bot_new );

set( SThan(1), 'Visible', STvis );
set( SThan(2), 'Position',STpos_bot_new )

set( SLhan(1), 'Max', 1, 'Min', MM,    'Value', 1 , 'Visible', 'off' );
set( SLhan(2), 'Max', 1, 'Min', SLmax, 'Value', 1,  'Visible', 'off', ...
               'Position', SLpos_bot_new, 'Callback', SL2cb );
set( SLhan( SLVIShan & idx ), 'Visible',  'on' );
set( CB6han, 'Value', 0 );

d_cfmtcb(0);
d_outpcb(1);

set( get(0,'Children'), 'pointer', 'arrow' );

