function FIGhan = d_disply()
%D_DISPLY creates and initializes the "DISPLAY" window and its children.
%
%         In addition to the "standard" global variables such as colors, 
%         this function generates two new global variables during set-up,
%         namely, "DFP_alpha" and "DFP_scrolltextlines".  The first variable
%         determines the resize ratio as the "PoZe/Coeff" checkbox in the
%         NAVIGATOR window is clicked on/off.  The latter variable determines 
%         the number of text lines which can be displayed in the coefficient
%         sub-windows.  The reason why we determine these two variables here
%         is simple: only when we generate the DISPLAY window we have an idea
%         what they will be equal to.  They are a function of the ScreenSize.
%
%         FIG = D_DISPLY returns the handle to the figure object which has 
%         been created by calling the function D_DISPLY.

%
%    Author: Mehmet Zeytinoglu (mzeytin@ee.ryerson.ca)
%            Department of Electrical and Computer Engineering
%            Ryerson Polytechnic University
%            Toronto, Ontario, M5B 2K3 
%            CANADA
%            
%    Copyright (c) 1997 Mehmet Zeytinoglu
%
%    $Revision: 1.4 $    $Date: 1997/03/11 16:18:10 $
%

%=================
% GLOBAL variables
%=================
global  DFP_FIGbgroundcolor   DFP_AXbgroundcolor   DFP_ETbgroundcolor  ...
        DFP_scrolltextlines   DFP_FigureResize     DFP_WinBorderOffset ...
        DFP_alpha             DFP_UIbgroundcolor   DFP_ColorMonitor    ...
        DFP_ScaleX            DFP_ScaleY


%==============
% FIGURE NUMBER
%==============
FigNo = 4;


%====================
% Display definitions
%====================

%  AXES
%======
Dim1     = 0.12;    % figure left border to axes right edge

Dim20    = 0.80;    % MAG and PHASE axes                 x-dimension
Dim21    = 0.38;    % POLE-ZERO     axes and COEFF frame x-dimension

Dim30    = 0.30;    % MAG       axes y-dimension
Dim31    = 0.11;    % PHASE     axes y-dimension 
Dim32    = 0.32;    % POLE-ZERO axes and COEFF frame y-dimension

Dim40    = 0.05;    % POLE-ZERO axes bottom to FIG bottom
Dim41    = 0.07;    % Vertical separation between axes

AXpozPos = [ Dim1  Dim40                       Dim21 Dim32 ];
AXphaPos = [ Dim1  sum(AXpozPos([2 4]))+Dim41  Dim20 Dim31 ];
AXmagPos = [ Dim1  sum(AXphaPos([2 4]))+Dim41  Dim20 Dim30 ];

%  FIGURE
%========
Dim1      = 52 * DFP_ScaleX;  % approximate number of pixels for the window
                              % titlebar also used in determining Fig x-size
SCRsize   = get(0,'Screensize');
REFpos    = get(findobj(get(0,'Children'),'flat','Tag','fig1'),'Position');
FIGx      = SCRsize(3) - REFpos(3) - 2*DFP_WinBorderOffset - Dim1;
FIGy      = (SCRsize(4) - DFP_WinBorderOffset - Dim1); 
FIGpos    = [ sum(REFpos([1 3]) + 2*DFP_WinBorderOffset) ... 
                         (SCRsize(4) - FIGy - DFP_WinBorderOffset) ];
FIGbgc    = DFP_FIGbgroundcolor;
DFP_alpha = (1-sum(AXpozPos([2 4])))*FIGy / ...
                 (SCRsize(4)-DFP_WinBorderOffset-Dim1);

%================================================================ 
% Remark: DFP_alpha is the ratio of (MAG+PHASE)/FULL-SIZE figure.
%         This parameter is to be used by the d_resize function 
%         which is linked to the "PoZe/Coeff" button on the
%         NAVIGATOR window.  This ratio is determined from the
%         physical screen size.
% DFP_alpha < 1, i.e., HALF -> FULL use "d_resize(1/DFP_alpha)"
%                      FULL -> HALF use "d_resize(DFP_alpha)"
%================================================================

%  FRAME
%========
Dim1  = 30 * DFP_ScaleY;                    % height of status display frame
FRbgc = [ FIGbgc; DFP_UIbgroundcolor ];
FRuni = str2mat( 'norm', 'pixels' );
FRtag = str2mat( 'pole', 'frame4:stat' );
FRpos = [
         0.54    Dim40     Dim21   Dim32;   % normalized
          0     FIGy-Dim1  FIGx    Dim1 ;   % pixel
        ]; 
%
% SCROLLABLE TEXT objects
%========================
A1 =  8 * DFP_ScaleX;
A2 = 12 * DFP_ScaleY;
A3 = 24 * DFP_ScaleY;

STx   =  FRpos(1,3)*FIGx-5*A1;
STy   = (FRpos(1,4)*FIGy-2*(A2+A3))/2;
STpos = [ 
          FRpos(1,1)*FIGx+2*A1 FRpos(1,2)*FIGy+1.5*A2+A3+STy  STx  STy;
          FRpos(1,1)*FIGx+2*A1 FRpos(1,2)*FIGy+A2             STx  STy;
        ];
STtag = str2mat( 'scroll:top', 'scroll:bottom' );

if ( DFP_ColorMonitor )
   STbgc = DFP_UIbgroundcolor*1.05;
else
   STbgc = DFP_UIbgroundcolor;
end

DFP_scrolltextlines = fix( STy/(17.5*DFP_ScaleY) );
%===========================================================================
% REMARK:
%      The determination of DFP_scrolltextlines as shown is experimental.
%      Further study may be required.  I empirically determined that with
%      MATLAB default values for the uicontrol text objects each line of
%      uicontrol text occupies about 17.5 pixels in the vertical direction.
%===========================================================================

%
% SLIDER objects
%===============
SLx   = 1.5*A1;                  %=== SLIDER object width
SLy   = STy;                     %=== SLIDER object height
SLbgc = 0.8*DFP_UIbgroundcolor;  %=== to get a decent slider color that
                                 %=== matches to the figure colors
SLpos = [
         sum(STpos(1,[1 3]))+(2*DFP_ScaleX) STpos(1,2) SLx SLy;
         sum(STpos(2,[1 3]))+(2*DFP_ScaleX) STpos(2,2) SLx SLy;
        ];
SLtag = str2mat( 'slider:top',  'slider:bottom' );

%
% TEXT objects
%===============
TXx   = 130 * DFP_ScaleX;  
TXx1  = STx+SLx-TXx;
TXy   = 24 * DFP_ScaleY;
TXbgc = [ 
          DFP_UIbgroundcolor;
          DFP_UIbgroundcolor;
          DFP_UIbgroundcolor;
          DFP_UIbgroundcolor;
          FIGbgc; 
          FIGbgc; 
          FIGbgc; 
          FIGbgc; 
         ];
if ( DFP_ColorMonitor )

   TXfgc = [ 
             1 1 0 ;
             1 1 0 ;
             1 1 0 ;
             1 1 0 ;
             1 1 1 ; 
             1 1 1 ; 
             1 1 0 ;  
             1 1 0 ;
           ];

else

   TXfgc = [ zeros(4,3); ones(4,3) ];

end

TXalg = str2mat( 'center','center','center','center', ...
                 'left', 'left', 'right', 'right' );
TXstr = str2mat( '_______', ' _______', '  _______', '  _______', ...
                 'Numerator Terms:', 'Denominator Terms:' , '', '' );
LBpos = [ 
          STpos(1,1) sum(STpos(1,[2 4])) TXx TXy;
          STpos(2,1) sum(STpos(2,[2 4])) TXx TXy;
        ];
A=FIGx/5;  B=FRpos(2,2)+(FRpos(2,4)-TXy)/2;  
TXpos = [
          (A/2)+A*[0:3]'  B*ones(4,1)  A*ones(4,1)  TXy*ones(4,1);
          LBpos;
          sum(LBpos(1,[1 3])) LBpos(1,2) TXx1 TXy;
          sum(LBpos(2,[1 3])) LBpos(2,2) TXx1 TXy;
        ];
TXtag = str2mat( 'fig4:sta1',  'fig4:sta2',  'fig4:sta3',  'fig4:sta4', ...
                 'fig4:label', 'fig4:label', 'fig4:value', 'fig4:value' );

%=======================================================
% Create and place the FIGURE and UIMENU objects
%=======================================================

FIGhan = figure( ...
            'Name',                   d_namesv(FigNo),  ...
            'Tag',                  ['fig' num2str(FigNo)],   ...
            'NumberTitle',           'off',                   ...
            'MenuBar',               'none',                  ...
            'Backingstore',          'on',                    ...
            'WindowButtonDownFcn',   'zoom(''down'')',        ...
            'Resize',                 DFP_FigureResize,       ...
            'Color',                  FIGbgc,                 ...
            'DefaultAxesColor',       DFP_AXbgroundcolor,     ...
            'DefaultAxesBox',        'on',                    ...
            'Visible',               'off',                   ...
            'Position',             [ FIGpos FIGx FIGy ]      ...
            );

%=============================
% MAGNITUDE menu and sub-menus
%=============================
MNhan = uimenu( FIGhan, ...
                             'Label',         'Magnitude',    ...
                             'Accelerator',   'M' ,           ...
                             'Visible',       'on',           ...
                             'Tag',           'menu4:mag'       );
            uimenu( MNhan,   'Label',         'dB  ',         ...
                             'Checked',       'on',           ...
                             'Tag',           'menu4:db',     ...
                             'Userdata',       1 ,            ...
                             'Callback',      'd_scmncb(1);' );
            uimenu( MNhan,   'Label',         'Linear  ',     ...
                             'Checked',       'off',          ...
                             'Tag',           'menu4:lin',    ...
                             'Userdata',       0 ,            ...
                             'Callback',      'd_scmncb(0);' );
            uimenu( MNhan,   'Label',         'Target  ',     ...
                             'Separator',     'on',           ...
                             'Enable',        'off',          ...
                             'Checked',       'off',          ...
                             'Tag',           'menu4:target', ...
                             'Callback',      'd_tgmncb;' );
            uimenu( MNhan,   'Label',         'Grid    ',     ...
                             'Separator',     'on',           ...
                             'Checked',       'off',          ...
                             'Tag',           'menu4:grid1',  ...
                             'Callback',      'd_grmncb(1);' );
            uimenu( MNhan,   'Label',         'Limits ... ',  ...
                             'Separator',     'off',          ...
                             'Checked',       'off',          ...
                             'Tag',           'menu4:lim1',    ...
                             'Callback',      'd_almncb(1);'   );
%=========================
% PHASE menu and sub-menus
%=========================
MNhan = uimenu( FIGhan, ...
                             'Label',         'Phase',        ...
                             'Accelerator',   'P' ,           ...
                             'Visible',       'on',           ...
                             'Tag',           'menu4:phase'         );
            uimenu(MNhan,    'Label',         'Radians  ',    ...
                             'Checked',       'off',          ...
                             'Tag',           'menu4:rad',    ...
                             'Callback',                      ...
                                    'd_dplydr( [0 1 0], [NaN 1])' );
            uimenu(MNhan,    'Label',         'Degrees  ',    ...
                             'Checked',       'off',          ...
                             'Tag',           'menu4:deg',    ...
                             'Callback',                      ...
                                    'd_dplydr( [0 1 0], [NaN 0])' );
            uimenu(MNhan,    'Label',         'Wrap  ',       ...
                             'Checked',       'off',          ...
                             'Separator',     'on',           ...
                             'Tag',           'menu4:wrap',   ...
                             'Callback',                      ...
                                    'd_dplydr( [0 1 0], [1 NaN])' );
            uimenu(MNhan,    'Label',         'Unwrap  ',     ...
                             'Checked',       'off',          ...
                             'Tag',           'menu4:unwrap', ...
                             'Callback',                      ...
                                    'd_dplydr( [0 1 0], [0 NaN])' );
            uimenu( MNhan,   'Label',         'Grid    ',     ...
                             'Separator',     'on',           ...
                             'Checked',       'off',          ...
                             'Tag',           'menu4:grid2',  ...
                             'Callback',      'd_grmncb(2);' );
            uimenu( MNhan,   'Label',         'Limits ... ',  ...
                             'Separator',     'off',          ...
                             'Checked',       'off',          ...
                             'Tag',           'menu4:lim2',    ...
                             'Callback',      'd_almncb(2);'    );

%==================================
% GDELAY/IMPULSE menu and sub-menus
%==================================
MNhan = uimenu( FIGhan, ...
                             'Label',         'Gdelay',    ...
                             'Accelerator',   'P' ,        ...
                             'Visible',       'off',       ...
                             'Tag',           'menu4:points'  );
            uimenu( MNhan,   'Label',         'Points'        );
            uimenu( MNhan,   'Label',         '128',       ...
                             'Checked',       'off',       ...
                             'Tag',           '128',       ...
                             'Separator',     'on',        ...
                             'Userdata',       128,        ...
                             'Callback',                   ...
                                    'd_dplydr( [0 NaN 0], 128 )' );
            uimenu( MNhan,   'Label',         '256',       ...
                             'Checked',       'off',       ...
                             'Tag',           '256',       ...
                             'Userdata',       256,        ...
                             'Callback',                   ...
                                    'd_dplydr( [0 NaN 0], 256 )' );
            uimenu( MNhan,   'Label',         '512',       ...
                             'Checked',       'off',       ...
                             'Tag',           '512',       ...
                             'Userdata',       512,        ...
                             'Callback',                   ...
                                    'd_dplydr( [0 NaN 0], 512 )' );
            uimenu( MNhan,   'Label',         '1024',      ...
                             'Checked',       'off',       ...
                             'Tag',           '1024',      ...
                             'Userdata',       1024,       ...
                             'Callback',                   ...
                                    'd_dplydr( [0 NaN 0], 1024 )' );
            uimenu( MNhan,   'Label',         'Other ... ',...
                             'Checked',       'off',       ...
                             'Tag',           'menu4:other',...
                             'Userdata',       0,          ...
                             'Callback',                   ...
                                     sprintf('d_inpdlg(%s,%s,%s,%s);', ...
                                     '''Enter Number of Points:''',    ...
                                     'd_namesv(14)',                   ...
                                     '''d_numncb;''',                  ...
                                     '''  ''')                            );
            uimenu( MNhan,   'Label',         'Grid    ',     ...
                             'Separator',     'on',           ...
                             'Checked',       'off',          ...
                             'Tag',           'menu4:grid2',  ...
                             'Callback',      'd_grmncb(2);' );
            uimenu( MNhan,   'Label',         'Limits ... ',  ...
                             'Separator',     'off',          ...
                             'Checked',       'off',          ...
                             'Tag',           'menu4:lim3',   ...
                             'Callback',      'd_almncb(3);'   );

%================================
% COEFFICIENTS menu and sub-menus
%================================
MNhan = uimenu( FIGhan, ...
                             'Label',         'Coefficients',  ...
                             'Accelerator',   'C' ,            ...
                             'Visible',       'off',           ...
                             'Tag',           'menu4:coeff'       );
            uimenu( MNhan,   'Label',         'Float  ',       ...
                             'Checked',       'on' ,           ...
                             'Tag',           'format:coeff',  ...
                             'Callback',      'd_cfmtcb(0)' );
            uimenu( MNhan,   'Label',         'Hex  ',         ...
                             'Checked',       'off',           ...
                             'Enable',        'off',          ...
                             'Tag',           'format:coeff',  ...
                             'Separator',     'on',            ...
                             'Callback',      'd_cfmtcb(1)' );
            uimenu( MNhan,   'Label',         'Octal  ',       ...
                             'Checked',       'off',           ...
                             'Enable',        'off',          ...
                             'Tag',           'format:coeff',  ... 
                             'Callback',      'd_cfmtcb(2)' );
            uimenu( MNhan,   'Label',         'Binary  ',      ...
                             'Checked',       'off',           ...
                             'Enable',        'off',          ...
                             'Tag',           'format:coeff',  ...
                             'Callback',      'd_cfmtcb(3)' );

%============================
% SNAPSHOT menu and sub-menus
%============================
MNhan = uimenu( FIGhan, ...
                             'Label',         'Snapshot',      ...
                             'Accelerator',   'S' ,            ...
                             'Visible',       'on',            ...
                             'Tag',           'menu4:snapshot'      );
            uimenu( MNhan,   'Label',         'Magnitude  ',   ...
                             'Visible',       'on',            ...
                             'Tag',           'snp:mag',       ...
                             'Callback',      'd_spmncb(1)'   );
            uimenu( MNhan,   'Label',         'Phase  ',       ...
                             'Visible',       'on',            ...
                             'Tag',           'snp:secondary', ...
                             'Callback',      'd_spmncb(2)'   );
            uimenu(MNhan,    'Label',         'Pole Zero  ',   ...
                             'Visible',       'off',           ...
                             'Tag',           'snp:pozero',    ...
                             'Callback',      'd_spmncb(3)'   );

drawnow

%
%  AXES objects
%==============

axes( ...
              'Drawmode',           'fast',         ...
              'Position',            AXmagPos ,     ...
              'Userdata',           [0 1 0 1; 0 1 0 1], ...
              'Tag',                'axes:mag',     ...
              'Visible',            'on'               );
               ylabel('Magnitude [dB]'); 
               xlabel('Frequency [Hz]');
axes( ...
              'Drawmode',           'fast',         ...
              'Position',            AXphaPos,      ...
              'Tag',                'axes:pha',     ...
              'Visible',            'on'               );
               ylabel('f/p','FontName','symbol'); 
               xlabel('Frequency [Hz]');
axes( ...
              'Drawmode',           'fast',         ...
              'Position',            AXpozPos,      ...
              'Tag',                'axes:poz',     ...
              'Ytick',               [],            ...
              'Visible',            'on'               );

%
% FRAME objects
%==============
for ii = 1:size(FRpos,1)
    uicontrol( FIGhan,                                     ...
              'Style',              'frame',               ...
              'Backgroundcolor',     FRbgc(ii,:),          ...
              'Units',               deblank(FRuni(ii,:)), ...
              'Tag',                 deblank(FRtag(ii,:)), ...
              'Position',            FRpos(ii,:)           ...
             );
end

%
% SCROLLABLE TEXT objects
%========================
for ii = 1:size(STpos,1)
    uicontrol( FIGhan,                                     ...
              'Style',              'edit',                ...
              'BackGroundColor',     STbgc,                ...
              'HorizontalAlignment','left',                ...
              'Units',              'pixels',              ...
              'Tag',                 deblank(STtag(ii,:)), ...
              'Min',                 0,                    ...
              'Max',                 2,                    ...
              'Position',            STpos(ii,:)           ...
             );
end

%
% SLIDER objects
%===============
for ii = 1:size(SLpos,1)
    uicontrol( FIGhan,                                     ...
              'Style',              'slider',              ...
              'BackGroundColor',     SLbgc,                ...
              'Tag',                 deblank(SLtag(ii,:)), ...
              'Callback',            sprintf('d_csldcb(%i);',ii),...
              'Visible',            'off',                 ...
              'Value',               1,                    ...
              'Position',            SLpos(ii,:)           ...
             );
end

%
% TEXT objects
%===============
for ii = 1:size(TXpos,1)
    uicontrol( FIGhan,                                     ...
              'Style',              'text',                ...
              'BackGroundColor',     TXbgc(ii,:),          ...
              'ForeGroundColor',     TXfgc(ii,:),          ...
              'HorizontalAlignment', deblank(TXalg(ii,:)), ...
              'String',              deblank(TXstr(ii,:)), ...
              'Clipping',           'off',                 ...
              'Tag',                 deblank(TXtag(ii,:)), ...
              'Position',            TXpos(ii,:)           ...
             );
end

set( findobj(get(FIGhan,'Children'),'flat','Type','uicontrol'), ...
    'Units', 'normalized' );

d_resize(DFP_alpha);
