function  d_codgen( CoeffArray, FilterType, Fid, Options )
%D_CODGEN generates filter code according to user requested options.
%      D_CODGEN( C, FILTTYPE, Fid, OPTIONS ) generates the filter code derived
%        from the string array C representing filter coefficients.  FILTTYPE 
%        specifies the filter type (FIR/IIR/Direct Form/Lattice/Biquad).  The 
%        filter code is written to the file with File Identifier Fid which must
%        have been opened by the calling program "d_outpcb".  Finally OPTIONS
%        represent different code generation options.
%
%        FILTTYPE = 0  --> FIR Direct Form
%                 = 1  --> FIR Lattice 
%                 = 2  --> IIR Direct Canonic Form with scaling
%                 = 3  --> IIR Lattice 
%                 = 4  --> IIR Cascade of Biquad Sections with scaling
%
%        FIRTYPE = 0 --> OPTIONS = [ CodeType M ]
%        FIRTYPE = 1 --> OPTIONS = [ CodeType K G1 G2 ]
%        FIRTYPE = 2 --> OPTIONS = [ CodeType M ]
%        FIRTYPE = 3 --> OPTIONS = [ CodeType K C ]
%        FIRTYPE = 4 --> OPTIONS = [ CodeType L ]
%
%        where
%
%           CodeType = 1/2  for (macro code)/(full code)
%            M = number of numerator coefficients; 
%            K = number of reflection coefficients;
%            C = number of c-coefficients in an IIR lattice filter;
%           G1 = feedforward gain for the FIR lattice;
%           G2 = lattice gain for the FIR lattice;
%            L = number of biquad sections.

%
%    Author: Mehmet Zeytinoglu (mzeytin@ee.ryerson.ca)
%            Department of Electrical and Computer Engineering
%            Ryerson Polytechnic University
%            Toronto, Ontario, M5B 2K3 
%            CANADA
%            
%    Copyright (c) 1997 Mehmet Zeytinoglu
%
%    $Revision: 1.3 $    $Date: 1997/03/11 16:18:10 $
%

global   DFP_ver  DFP_TemplatesDir

h = sort( findobj(get(findobj(get(0,'Children'),'flat','Tag','fig4'), ...
                                    'Children'),'flat','Style','text'));
h = h(1:4);
s = get( h(4), 'String' );
s = s(findstr(s,'['):length(s));
ht = sort( findobj(get(findobj(get(0,'Children'),'flat','Tag','fig2'), ...
                       'Children'),'flat','Visible','on','Style','text'));
he = sort( findobj(get(findobj(get(0,'Children'),'flat','Tag','fig2'), ...
                       'Children'),'flat','Visible','on','Style','edit'));

CodeType = Options(1);

if ( FilterType == 0 )        %==== FIR Direct Form ===
   LL = Options(2);
elseif ( FilterType == 1 )    %==== FIR Lattice Form ===
   LL = Options(2);
elseif ( FilterType == 2 )    %==== IIR Direct Canonic Form ===
   LL = 2*Options(2)-1;
elseif ( FilterType == 3 )    %==== IIR Lattice Form ===
   LL = Options(2)+Options(3);
elseif ( FilterType == 4 )    %==== IIR Cascade of Biquad Sections ===
   LL = size(CoeffArray,1);
end
str    = [];
for ii = 1:LL
    str = str2mat( str, sprintf('%s', CoeffArray(ii,:)) );
end
str = str(2:size(str,1),:);

fprintf( Fid, ';**************************************************\n' );
fprintf( Fid, ';                                                  \n' );
fprintf( Fid, '; Created by     : Digital Filter Package (DFP)        \n' );
fprintf( Fid, ';                  Version %s  \n', DFP_ver  );
fprintf( Fid, ';                                                  \n' );
fprintf( Fid, '; Created on     : %s  \n', timedate );
fprintf( Fid, ';                                                  \n' );
fprintf( Fid, ';--------------------------------------------------\n' );
fprintf( Fid, ';                                                  \n' );
fprintf( Fid, '; Filter Type    : %s  \n', get(h(1),'String') );
fprintf( Fid, ';                                                  \n' );
for ii = 1:length(ht)
    fprintf( Fid, ';     %s = %s \n', ...
             get(ht(ii),'String'),get(he(ii),'String'));
end
fprintf( Fid, ';                                                  \n' );
fprintf( Fid, '; Design Method  : %s  \n', get(h(2),'String') );
fprintf( Fid, '; Implementation : %s  \n', get(h(3),'String') );
fprintf( Fid, '; Coefficients   : %s  \n', s                  );
fprintf( Fid, ';                                                  \n' );
fprintf( Fid, ';**************************************************\n\n' );

%==========================================================================
%  If the environment variable MOT56k_MACLIB that defines the location of
%  the MACRO library for the Motorola DSP56k has been set and if we want 
%  to generate full assembler code, include the following line in the
%  output file.
%==========================================================================
maclib = getenv( 'MOT56k_MACLIB' );
if ( ~isempty(maclib) & (CodeType == 2) )
   fprintf( Fid,'  MACLIB  ''%s''\n\n', maclib );
elseif ( isempty(maclib) & (CodeType == 2) )
   fprintf( Fid,'  INCLUDE ''aiset''   ; SSI port initialization macro\n\n');
end

%===========================================
%  Now start generating the filter code
%===========================================

if ( FilterType == 0 )    %==== FIR Direct Form ===

   NN = Options(2);
   if ( CodeType == 2 )
      s = [ DFP_TemplatesDir 'fir_d00.asm' ];
      if ( exist(s) )
         d_append( Fid, s ); 
      else
         d_error('EO-06', s, DFP_TemplatesDir );
         return;
      end
   end
   fprintf( Fid,['CoeffArrayAddress  equ  $0\n' ...
             'StateArrayAddress  equ  $0\n'   ] );
   if ( CodeType == 2 )
    fprintf( Fid,'ProgramAddress     equ  $100\n');
   end
   fprintf( Fid,'ntaps              equ  %i\n\n\n', NN );
   fprintf( Fid,'       org  Y:CoeffArrayAddress  \n'           );
   fprintf( Fid,'coeff  dc   %s     ; b(%i)\n', str(1,:),  1   );
   for ii = 2:NN
       fprintf( Fid,'       dc   %s     ; b(%i)\n', str(ii,:), ii);
   end
   fprintf( Fid,'\n       org  X:StateArrayAddress  \n');
   fprintf( Fid,'state  dsm  ntaps\n' );
   if ( CodeType == 1 )
      fprintf( Fid,'r0str  dc   state\n' );
   elseif ( CodeType == 2 )
      s = [ DFP_TemplatesDir 'fir_d02.asm' ];
      if ( exist(s) )
         d_append( Fid, s ); 
      else
         d_error('EO-06', s, DFP_TemplatesDir );
         return;
      end
   end

elseif ( FilterType == 1 )    %==== FIR Lattice Form ===

   NN = Options(2);
   if ( CodeType == 2 )
      s = [ DFP_TemplatesDir 'fir_l00.asm' ];
      if ( exist(s) )
         d_append( Fid, s ); 
      else
         d_error('EO-06', s, DFP_TemplatesDir );
         return;
      end
   end
   fprintf( Fid,['CoeffArrayAddress  equ  $0\n' ...
             'StateArrayAddress  equ  $0\n'   ] );
   if ( CodeType == 2 )
    fprintf( Fid,'ProgramAddress     equ  $100\n');
   end
   fprintf( Fid,'ncoef              equ  %i\n', NN );
   fprintf( Fid,'GainF              equ  %.14f\n',   Options(3) );
   fprintf( Fid,'GainL              equ  %.14f\n\n', Options(4) );
   fprintf( Fid,'       org  Y:CoeffArrayAddress  \n'           );
   fprintf( Fid,'coeff  dc   %s     ; k(%i)\n', str(1,:),  1   );
   for ii = 2:NN
       fprintf( Fid,'       dc   %s     ; k(%i)\n', str(ii,:), ii);
   end
   fprintf( Fid,'\n       org  X:StateArrayAddress  \n');
   fprintf( Fid,'state  dsm  ncoef+1\n' );
   if ( CodeType == 1 )
      fprintf( Fid,'r0str  dc   state\n' );
   elseif ( CodeType == 2 )
      s = [ DFP_TemplatesDir 'fir_l02.asm' ];
      if ( exist(s) )
         d_append( Fid, s ); 
      else
         d_error('EO-06', s, DFP_TemplatesDir );
         return;
      end
   end

elseif ( FilterType == 2 )    %==== IIR Direct Canonic Form ===

   NN = Options(2);
   if ( CodeType == 2 )
      s = [ DFP_TemplatesDir 'iir_d00.asm' ];
      if ( exist(s) )
         d_append( Fid, s ); 
         fprintf(  Fid,'igain              equ  0.5 \n');
      else
         d_error('EO-06', s, DFP_TemplatesDir );
         return;
      end
   end
   fprintf( Fid,['CoeffArrayAddress  equ  $0\n' ...
             'StateArrayAddress  equ  $0\n'   ] );
   if ( CodeType == 2 )
    fprintf( Fid,'ProgramAddress     equ  $100\n');
   end
   fprintf( Fid, 'nstate             equ  %i\n\n', NN );
   fprintf( Fid, '       org  Y:CoeffArrayAddress  \n'           );
   fprintf( Fid, 'coeff  dc   %s     ; -a(%i)/2\n', str(1,:),  2   );
   for ii = 2:NN-1
       fprintf( Fid,'       dc   %s     ; -a(%i)/2\n', str(ii,:), ii+1 );
   end
   for ii = NN:2*NN-1
       fprintf( Fid,'       dc   %s     ;  b(%i)/2\n', str(ii,:), ii-NN+1 );
   end
   fprintf( Fid, '\n       org  X:StateArrayAddress  \n'           );
   fprintf( Fid, 'state  ds   nstate\n' );
   if ( CodeType == 1 )
      fprintf( Fid,'r0str  dc   state\n' );
   elseif ( CodeType == 2 )
      s = [ DFP_TemplatesDir 'iir_d02.asm' ];
      if ( exist(s) )
         d_append( Fid, s ); 
      else
         d_error('EO-06', s, DFP_TemplatesDir );
         return;
      end
   end

elseif ( FilterType == 3 )    %==== IIR Lattice Form ===

   KK = Options(2);
   CC = Options(3);
   if ( CodeType == 2 )
      s = [ DFP_TemplatesDir 'iir_l00.asm' ];
      if ( exist(s) )
         d_append( Fid, s ); 
      else
         d_error('EO-06', s, DFP_TemplatesDir );
         return;
      end
   end
   fprintf( Fid,['CoeffArrayAddress  equ  $0\n' ...
             'StateArrayAddress  equ  $0\n'   ] );
   if ( CodeType == 2 )
    fprintf( Fid,'ProgramAddress     equ  $100\n');
   end
   fprintf( Fid,'order              equ  %i\n\n', KK );
   fprintf( Fid,'       org  Y:CoeffArrayAddress  \n'           );
   fprintf( Fid,'coeff  dc   %s     ; k(%i)\n', str(1,:),  KK   );
   for ii = 2:KK
       fprintf( Fid,'       dc   %s     ; k(%i)\n', str(ii,:), KK-ii+1 );
   end
   fprintf( Fid,'       dc   %s     ; c(%i)\n', str(KK+1,:),  CC   );
   for ii = 2:CC
       fprintf( Fid,'       dc   %s     ; c(%i)\n', str(KK+ii,:), CC-ii+1 );
   end
   fprintf( Fid,'\n       org  X:StateArrayAddress  \n');
   fprintf( Fid,'state  dsm  order+1\n' );
   if ( CodeType == 1 )
      fprintf( Fid,'r0str  dc   state\n' );
   elseif ( CodeType == 2 )
      s = [ DFP_TemplatesDir 'iir_l02.asm' ];
      if ( exist(s) )
         d_append( Fid, s ); 
      else
         d_error('EO-06', s, DFP_TemplatesDir );
         return;
      end
   end

elseif ( FilterType == 4 )    %==== IIR Cascade of Biquad Sections ===

   ss  = blanks(length(str(1,:)));
   if ( CodeType == 2 )
      s = [ DFP_TemplatesDir 'iir_s00.asm' ];
      if ( exist(s) )
         d_append( Fid, s ); 
         fprintf(  Fid,'igain              equ  0.5 \n');
      else
         d_error('EO-06', s, DFP_TemplatesDir );
         return;
      end
   end
   fprintf( Fid,['CoeffArrayAddress  equ  $0\n' ...
             'StateArrayAddress  equ  $0\n'   ] );
   if ( CodeType == 2 )
    fprintf( Fid,'ProgramAddress     equ  $100\n');
   end
   fprintf( Fid,'biquad             equ  %i\n\n', Options(2) );
   fprintf( Fid,'      org  Y:CoeffArrayAddress  \n'           );
   fprintf( Fid,'           %s     ;=== Biquad 1 ===\n', ss    );
   fprintf( Fid,'coeff dc   %s     ; -a(1,3)/2\n', str(1,:) );
   fprintf( Fid,'      dc   %s     ; -a(1,2)/2\n', str(2,:) );
   fprintf( Fid,'      dc   %s     ;  b(1,3)/2\n', str(3,:) );
   fprintf( Fid,'      dc   %s     ;  b(1,2)/2\n', str(4,:) );
   fprintf( Fid,'      dc   %s     ;  b(1,1)/2\n', str(5,:) );
   fprintf( Fid,'                        \n'           );

   for ii = 2:Options(2)
   
   fprintf( Fid,'           %s     ;=== Biquad %.0f ===\n', ss, ii );
   fprintf( Fid,'      dc   %s     ; -a(%.0f,3)/2\n', str((ii-1)*5+1,:), ii);
   fprintf( Fid,'      dc   %s     ; -a(%.0f,2)/2\n', str((ii-1)*5+2,:), ii);
   fprintf( Fid,'      dc   %s     ;  b(%.0f,3)/2\n', str((ii-1)*5+3,:), ii);
   fprintf( Fid,'      dc   %s     ;  b(%.0f,2)/2\n', str((ii-1)*5+4,:), ii);
   fprintf( Fid,'      dc   %s     ;  b(%.0f,1)/2\n', str((ii-1)*5+5,:), ii);
   fprintf( Fid,'                        \n'           );

   end
   
   fprintf( Fid,'\n       org  X:StateArrayAddress  \n'           );
   fprintf( Fid,'state  ds   2*biquad\n\n' );
   if ( CodeType == 2 )
      s = [ DFP_TemplatesDir 'iir_s02.asm' ];
      if ( exist(s) )
         d_append( Fid, s ); 
      else
         d_error('EO-06', s, DFP_TemplatesDir );
         return;
      end
   end

end
