/*
	NAME
		strip - Motorola DSP COFF file strip utility

	SYNOPSIS
		strip files

	DESCRIPTION

		This program strips line number and symbol table
		information from Motorola DSP COFF object files.

	OPTIONS
		q	- do not display signon banner.

		Error messages are sent to the standard error output when
		files cannot be open.

	HISTORY
		1.0	The beginning.
*/

#if !defined (LINT)
static char rcsid[] = "$Id: strip.c,v 1.23 1999/03/19 19:51:42 jay Exp $";	/* RCS data */
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <errno.h>
#if !defined (ICC) && !defined (WATCOM)
extern int errno;
#endif
#if !defined (VARARGS)
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#if defined (MPW)
#include <files.h>
#endif
#if defined (LINT) || defined (SUN)
#include "lint.h"	/* lint declarations */
#endif

/*
  Headers for working with COFF files
*/

#ifdef BIG_ENDIAN
#if !BIG_ENDIAN
#undef BIG_ENDIAN
#endif
#endif
#include "coreaddr.h"
#include "maout.h"
#include "dspext.h"

#include "strip.h"

/*
  Global variables
*/

char Progdef[] = "strip";		/* program default name */
char *Progname = Progdef;		/* pointer to program name */
char Progtitle[] = "DSP COFF File Strip Utility";
/*
  Put an extra space after the version number so that it is not
  compressed in the object file.  That way the strings program
  can find it.
*/
char Version[]   = "Version 6.3 ";	/* strip version number */
char Copyright[] = "(C) Copyright Motorola, Inc. 1991-1996.  All rights reserved.";

char *optarg = NULL;	/* command argument pointer */
int optind = 0;		/* command argument index */

FILE *ifile = NULL;	/* file pointer for input file */
char *ifn = NULL;	/* pointer to input file name */
FILE *ofile = NULL;	/* file pointer for output file */
char *ofn = NULL;	/* pointer to output file name */
char *tfn = NULL;      	/* pointer to temporary file name */
char quiet = NO;	/* signon banner flag */


int 
main (int argc, char *argv[])
{
	int c;			/* option character buffer */
	char ofb[L_tmpnam + 1];	/* output file name buffer */
	char tfb[L_tmpnam + 1];	/* temporary file name buffer */
	char *tmpnam (char *);

/*
	set up for signals, save program name, check for command line options
*/

	(void)signal (SIGINT, onsig);
	(void)signal (SIGSEGV, onsig);
	/* scan for quiet flag on command line */
	quiet = cmdarg ('q', argc, argv) ? YES : NO;

#if defined (MSDOS) || defined (TOS)
	if (!quiet)
		(void)fprintf (stderr, "%s  %s\n%s\n",
			       Progtitle, Version, Copyright);
#endif
	while ((c = getopts (argc, argv, "Qq")) != EOF) {
		if (isupper (c))
			c = tolower (c);
		switch (c) {
		case 'q':
			quiet = YES;
			break;
		case '?':
		default:
			usage ();
			break;
		}
	}

	/* no more args?  error */
	if (optind >= argc)
		usage ();

	/* dash for input filename? use stdio */
	if (*argv[optind] == '-') {

		ifn = "stdin";
		ifile = stdin;
		ofn = "stdout";
		ofile = stdout;
		strip_file ();

	} else while (optind < argc) {	/* process input files */

		/* open input and scratch files */
		ifn = argv[optind++];
		if (!(ifile = fopen (ifn, "rb")))
			error ("cannot open input file %s", ifn);
#if defined (APOLLO)
		(void)strcpy (ofb, "stXXXXXX");
#endif
		if ((ofn = basename (tmpnam (ofb))) == NULL)
			error ("cannot create output file name");
#if defined (UNIX)
		(void)strcat (tfb, "a");	/* to insure uniqueness */
#endif
		if (!(ofile = fopen (ofn, "wb")))
			error ("cannot open output file %s", ofn);
		(void) setfile (ofn, "COFF", "MPS ");

		strip_file ();

		/* rename scratch file to original file name */
		(void)fclose (ifile);
		(void)fclose (ofile);
#if defined (APOLLO)
		(void)strcpy (tfb, "stXXXXXX");
#endif
		if ((tfn = basename (tmpnam (tfb))) == NULL)
			error ("cannot create temporary file name");
#if defined (UNIX)
		(void)strcat (tfb, "b");	/* to insure uniqueness */
#endif
		if (rename (ifn, tfn) != 0)
			error ("cannot rename input file %s", ifn);
		if (rename (ofn, ifn) != 0) {
			if (rename (tfn, ifn) != 0)
				error ("cannot undo rename of %s to %s",
				       ifn, tfn);
			error ("cannot rename output file %s", ofn);
		}
		if (remove (tfn) != 0)
			error ("cannot remove renamed input file %s", tfn);
	}
	exit (0);
	/*NOTREACHED*/	/* for lint, Watcom */
	return (0);
}

static void 
strip_file (void)
{
	FILHDR	file_header;
	AOUTHDR	opt_header;
	XCNHDR	*scn_header, *sh;
	XCNHDR	*new_scn_header;
	long opthdr, nscns, dsize = 0L;
	long sec_data_offset;
	int i;

	/* read and write file headers */
	if (freads ((char *)&file_header, sizeof (FILHDR), 1, ifile) != 1)
		error ("cannot read file header");
	errno = 0;	/* set errno here for non-system errors */
	if (!(file_header.f_flags & F_RELFLG)) /* must be absolute object */
		error ("invalid object file type");
	if (!file_header.f_nsyms)	/* already stripped */
		error ("%s: already stripped", ifn);
	file_header.f_flags |= F_LNNO;	/* indicate line numbers stripped */
	file_header.f_nsyms = 0L;	/* zero count of symbol entries */
	/*
	  We have to save the next two fields before the write because
	  bytes may be swapped on the write and they are swapped *in place*.
	 */
	opthdr = file_header.f_opthdr;	/* save optional header length */
	nscns = file_header.f_nscns;	/* save number of sections */
	if (fwrites ((char *)&file_header, sizeof (FILHDR), 1, ofile) != 1)
		error ("cannot write file header");
	if (file_header.f_opthdr) {	/* optional header present */
		if (freads ((char *)&opt_header, (int)opthdr,
			    1, ifile) != 1)
			error ("cannot read optional header");
		if (fwrites ((char *)&opt_header, (int)opthdr,
			    1, ofile) != 1)
			error ("cannot read optional header");
	}

	/* read and write section headers */
	if ((scn_header =
	     (XCNHDR *)malloc ((unsigned)(nscns * sizeof (XCNHDR)))) == NULL)
		error ("cannot allocate section headers");
	if ((new_scn_header =
	     (XCNHDR *)malloc ((unsigned)(nscns * sizeof (XCNHDR)))) == NULL)
		error ("cannot allocate new section headers");
	if (fseek (ifile, (long)(FILHSZ + opthdr), 0) != 0)
	{
	    error ("cannot seek to section headers");
	}
	if (freads ((char *)scn_header, (int)(nscns * sizeof (XCNHDR)),
		    1, ifile) != 1)
		error ("cannot read section headers");
	(void) memcpy(new_scn_header,scn_header,(int)(nscns * sizeof (XCNHDR)));
	sec_data_offset = (long) (FILHSZ + opthdr + (nscns * sizeof (XCNHDR)));
	
	for (i = 0, sh = new_scn_header; i < nscns; i++, sh++) 
	{
		if (sh->_s.s_scnptr)
		{
		    /* real data exists */
		    sh->_s.s_scnptr = sec_data_offset;
		    dsize += sh->_s.s_size;	/* cumulate data size */
		    sec_data_offset += sh->_s.s_size * sizeof (long);
		}
		sh->_s.s_nreloc = sh->_s.s_nlnno = 0L;
	}
	if (fwrites ((char *)new_scn_header, (int)(nscns * sizeof (XCNHDR)),
		    1, ofile) != 1)
		error ("cannot write section headers");
	free ((char *)new_scn_header);

	/* read and write section data */
	for (i = 0, sh = scn_header; i < nscns; i++, sh++) 
	{
	    if (sh->_s.s_scnptr && sh->_s.s_size)
	    {
		long *raw_data;	
		raw_data = (long *)malloc((unsigned)(sh->_s.s_size * sizeof (long)));
		if (!raw_data)
		{
		    error ("cannot allocate raw data");
		}
		if (fseek (ifile, sh->_s.s_scnptr, 0) != 0)
		{
		    error ("cannot seek to raw data");
		}
		if (freads ((char *)raw_data, (int)sh->_s.s_size,
			    sizeof (long), ifile) != sizeof (long))
		{
		    error ("cannot read raw data ");
		}
		if (fwrites ((char *)raw_data, (int)sh->_s.s_size,
			    sizeof (long), ofile) != sizeof (long))
		{
		    error ("cannot write raw data");
		}
		free ((char *)raw_data);
	    }
	}
	free ((char *)scn_header);
}

/**
*
* name		cmdarg --- scan command line for argument
*
* synopsis	argp = cmdarg (arg, argc, argv)
*		char *argp;	pointer to argument
*		char arg;	argument to find
*		int argc;	count of command line arguments
*		char **argv;	pointer to command line arguments
*
* description	Takes a pointer to and count of the command line arguments.
*		Scans command line looking for argument character.  Returns
*		pointer to argument if found, NULL otherwise.
*
**/
static char *
cmdarg (char arg, int argc, char **argv)
{
	--argc;	/* skip over program name */
	++argv;
	while (argc > 0) {	/* scan args */
		if (**argv == '-') {
			char *p;
			for (p = *argv + 1; *p; p++)
				if ((isupper (*p) ? tolower (*p) : *p) == arg)
					return (*argv);
		}
		--argc;
		++argv;
	}
	return (NULL);
}

static int 
getopts (int argc, char *argv[], char *optstring)
/* get option letter from argv */
{
	register char c;
	register char *place;
	static char *scan = NULL;

	optarg = NULL;

	if (scan == NULL || *scan == '\0') {
		if (optind == 0)
			optind++;
	
		if (optind >= argc || argv[optind][0] != '-' || argv[optind][1] == '\0')
			return(EOF);
		if (strcmp(argv[optind], "--")==0) {
			optind++;
			return(EOF);
		}
	
		scan = argv[optind]+1;
		optind++;
	}

	c = *scan++;
	place = strchr(optstring, c);

	if (place == NULL || c == ':') {
		(void)fprintf(stderr, "%s: unknown option -%c\n", Progname, c);
		return('?');
	}

	place++;
	if (*place == ':') {
		if (*scan != '\0') {
			optarg = scan;
			scan = NULL;
		} else if (optind < argc) {
			optarg = argv[optind];
			optind++;
		} else {
			(void)fprintf( stderr, "%s: -%c argument missing\n", 
					Progname, c);
			return('?');
		}
	}

	return(c);
}


#if defined (APOLLO)
/**
*
* name		tmpnam - create temporary file name
*
* synopsis	fn = tmpnam (template)
*		char *fn;		pointer to temporary file name
*		char *template;		template for temporary file name
*
* description	Creates a temporary file name based on template.  Template
*		has the form yyyXXXXXX, where yyy is arbitrary text and
*		the literal XXXXXX substring is replaced by a letter
*		and a five digit number.  If a duplicate identifier is
*		formed in the same process, the routine substitutes the
*		next alphabetic character in sequence for the leading letter.
*		Returns pointer to revised template on success, NULL if
*		template is badly formed or other error.
*
**/
char *
tmpnam (char *template)
{
	static char oldfn[MAXSTR] = "";
	static char subpat[] = "XXXXXX";
	char *p, *subp;
	char tbuf[32];
	long t;
	int i;
	extern long time ();

	if ((subp = strchr (template, subpat[0])) == NULL)
		return (NULL);

	if (strcmp (subp, subpat) != 0)
		return (NULL);

	(void) time (&t);
	if (t < 0L)
		t *= -1L;
	sprintf (tbuf, "%05ld", t);
	for (i = strlen (subpat) - 1, p = tbuf + strlen (tbuf);
	     i > 0;
	     i--, p--);
	i = 'a';
	do {
		sprintf (subp, "%c%s", i++, p);
	} while (i <= 'z' && strcmp (oldfn, template) == 0);
	if (i > 'z')
		return (NULL);

	strcpy (oldfn, template);
	return (template);
}
#endif


static char *
basename (			/* return base part of file name in str */
    char *str
)
{
	register char *p;

	if (!str)		/* empty input */
		return (NULL);

	for (p = str + strlen (str); p >= str; --p)
#if defined (MSDOS) || defined (TOS)
		if( *p == '\\' || *p == ':')
#endif
#if defined (VMS)
		if( *p == ']' || *p == ':')
#endif
#if defined (UNIX) || defined (MACH)
		if( *p == '/' )
#endif
#if defined (MPW)
		if( *p == ':' )
#endif
			break;

	return (p < str ? str : ++p);
}


/**
* name		setfile --- set the file type and creator if necessary
*
* synopsis	yn = setfile (fn, type, creator)
*		int yn;		YES on success, NO on failure
*		char *fn;	pointer to file name
*		char *type;	pointer to file type
*		char *creator;	pointer to file creator
*
* description	Sets the file type and creator for newly-created Macintosh
*		output files.  Simply returns YES on other hosts.
*
**/
static int 
setfile (char *fn, char *type, char *creator)
{
#if defined (MPW)
	int i;
	short status;
	struct FInfo finfo;
	OSType val;
	static char buf[256];

	(void) strcpy (buf, fn);
	c2pstr(buf);
	if ((status = GetFInfo ((unsigned char *) &buf[0], (short)0, &finfo)) != 0)
		return (NO);
	i = 0;
	val = (OSType) 0;
	while (i < sizeof (OSType)) {
		val <<= (OSType) 8;
		val += (OSType) type[i++];
	}
	finfo.fdType = val;
	i = 0;
	val = (OSType) 0;
	while (i < sizeof (OSType)) {
		val <<= (OSType) 8;
		val += (OSType) creator[i++];
	}
	finfo.fdCreator = val;
	if ((status = SetFInfo ((unsigned char *) &buf[0], (short)0, &finfo)) != 0)
		return (NO);
#endif
#if defined (LINT)
	fn = fn;
	type = type;
	creator = creator;
#endif
	return (YES);
}

/**
*
* name		freads - swap bytes and read
*
* synopsis	freads (ptr, size, nitems, stream)
*		char *ptr;		pointer to buffer
*		int size;		size of buffer
*		int nitems;		number of items to read
*		FILE *stream;		file pointer
*
* description	Treats ptr as reference to union array; if necessary,
*		swaps bytes to maintain base format byte ordering
*		(big endian).  Calls fread to do I/O.
*
**/
static int 
freads (char *ptr, int size, int nitems, FILE *stream)
{
	int rc;

	rc = fread (ptr, size, nitems, stream);
#if !defined (BIG_ENDIAN)
	swapw (ptr, size, nitems);
#endif
	return (rc);
}

/**
*
* name		fwrites - swap bytes and write
*
* synopsis	fwrites (ptr, size, nitems, stream)
*		char *ptr;		pointer to buffer
*		int size;		size of buffer
*		int nitems;		number of items to write
*		FILE *stream;		file pointer
*
* description	Treats ptr as reference to union array; if necessary,
*		swaps bytes to maintain base format byte ordering
*		(big endian).  Calls fwrite to do I/O.
*
**/
static int 
fwrites (char *ptr, int size, int nitems, FILE *stream)
{
#if !defined (BIG_ENDIAN)
	swapw (ptr, size, nitems);
#endif
	return (fwrite (ptr, size, nitems, stream));
}

#if !defined (BIG_ENDIAN)
/**
*
* name		swapw - swap bytes in words
*
* synopsis	swapw (ptr, size, nitems)
*		char *ptr;		pointer to buffer
*		int size;		size of buffer
*		int nitems;		number of items to write
*
* description	Treats ptr as reference to union array; if necessary,
*		swaps bytes to maintain base format byte ordering
*		(big endian).
*
**/
static void 
swapw (char *ptr, int size, int nitems)
{
	union wrd *w;
	union wrd *end = (union wrd *)ptr +
		((size * nitems) / sizeof (union wrd));
	unsigned i;

	for (w = (union wrd *)ptr; w < end; w++) {
		i = w->b[0];
		w->b[0] = w->b[3];
		w->b[3] = i;
		i = w->b[1];
		w->b[1] = w->b[2];
		w->b[2] = i;
	}
}
#endif

/**
*
* name		onsig - handle signals
*
* description	Displays a message on the standard error output and
*		exits with the error count.
*
**/
static void
onsig (int sig)
{
	switch (sig) {
	case SIGINT:
#if defined (WATCOM)	/* Watcom doesn't like variadic calls in signal handlers */
		exit (1);
#else
		(void) fprintf (stderr, "\n%s: Interrupted\n", Progname);
#if defined (SIOW)
		longjmp (Env, 1);
#else
		exit (1);
#endif
#endif
		break;
	case SIGSEGV:
		(void) fprintf (stderr, "%s: Fatal segmentation or protection fault; contact your tools vendor\n", Progname);
		exit (1);
		break;
	}
}


static void 
usage (void)			/* display usage on stderr, exit nonzero */
{
#if defined (MSDOS) || defined (TOS)
	if (quiet)
#endif
		(void)fprintf (stderr, "%s  %s\n%s\n",
			       Progtitle, Version, Copyright);
	(void)fprintf (stderr, "Usage:  %s [-q] files\n", Progname);
	(void)fprintf (stderr, "        q - do not display signon banner\n");
	exit (1);
}

#if defined (LINT)
#if defined (va_dcl)
#undef va_dcl
#define va_dcl char *va_alist;
#endif
#endif

/* VARARGS1 */
static void
#if !defined (VARARGS)
error (char *fmt, ...)		/* display error on stderr, exit nonzero */
#else
error (va_alist)		/* display error on stderr, exit nonzero */
va_dcl
#endif
{
	va_list ap;
#if !defined (LINT)
	int err = errno;
#endif
#if !defined (VARARGS)
	va_start (ap, fmt);
#else
	char *fmt;
	va_start (ap);
	fmt = va_arg (ap, char *);
#endif
	(void)fprintf  (stderr, "%s: ", Progname);
	(void)vfprintf (stderr, fmt, ap);
	(void)fprintf  (stderr, "\n");
	va_end (ap);
#if !defined (LINT)
	if (err) {
		errno = err;
		perror (Progname);
	}
#endif
	(void)remove (ofn);
	exit (1);
}

