/************************************************************************
*
*  FREESCALE SEMICONDUCTORS INC.
*  ALL RIGHTS RESERVED
*  (c) Copyright 2003 Freescale semiconductors, Inc.
*
*************************************************************************
*
*  FILE NAME  : BAND_FLTR
*
*  PURPOSE    : BANDPASS_FLTR, BANDREJECT_FLTR macros definition for 
*		CW compiler
*
*  AUTHOR     :  Andriy Tymkiv, Anatoly Khaynakov
*
***********************************************************************/

#ifndef __BAND_FLTR_H
#define __BAND_FLTR_H
#include "math.h"

/************************************************************************
* NAME: BANDPASS_FLTR
*
* DESCRIPTION: 	calculates the band-pass filter of src[0..size-1] array 
*		in dest[0..size-1] array with center frequency `freq' and
*		bandwith `bandw'
*
* NOTE: 	The elements type is assumed to be FRAC32
*
*************************************************************************/

#define BANDPASS_FLTR(dst,src,size,freq,bandw) do {	\
		FRAC32 *b = dst;			\
		FRAC32 *a = src;			\
		long n = (size);			\
		FRAC32 f = (freq);			\
		FRAC32 bw = (bandw);			\
		double d_a0, d_a1, d_a2, d_b1, d_b2;	\
		double K,R;				\
		R = 1 - 3.0 * F32_TO_D(bw);		\
		K = (1.0 - 2.0 * R * 			\
			cos(2.0 * PI * F32_TO_D(f))+  	\
			R * R) / (2.0 - 2.0 *		\
			cos(2.0 * PI * F32_TO_D(f)));	\
		d_a0 = 1.0 - K;				\
		d_a1 = 2.0 * (K - R) *			\
			cos(2.0 * PI * F32_TO_D(f));	\
		d_a2 = R * R - K;			\
		d_b1 = 2.0  * R *			\
			cos(2.0 * PI * F32_TO_D(f));	\
		d_b2 = - 1.0 * R * R;			\
		__IMPL_BAND_FLTR(b, a, n,		\
			D_TO_F32(d_a0),			\
			D_TO_F32(d_a1),			\
			D_TO_F32(d_a2),			\
			D_TO_F32(d_b1),			\
			D_TO_F32(d_b2));		\
		} while (0)
		



/************************************************************************
* NAME: BANDPASS_FLTR
*
* DESCRIPTION: 	calculates the band-reject filter of src[0..size-1] array 
*		in dest[0..size-1] array with center frequency `freq' and
*		bandwith `bandw'
*
* NOTE: 	The elements type is assumed to be FRAC32
*
*************************************************************************/

#define BANDREJECT_FLTR(dst,src,size,freq,bandw) do {	\
		FRAC32 *b = dst;			\
		FRAC32 *a = src;			\
		long n = (size);			\
		FRAC32 f = (freq);			\
		FRAC32 bw = (bandw);			\
		double d_a0, d_a1, d_a2, d_b1, d_b2;	\
		double K,R;				\
		R = 1 - 3.0 * F32_TO_D(bw);		\
		K = (1.0 - 2.0 * R * 			\
			cos(2.0 * PI * F32_TO_D(f))+  	\
			R * R) / (2.0 - 2.0 *		\
			cos(2.0 * PI * F32_TO_D(f)));	\
		d_a0 = K;				\
		d_a1 = -2.0 * K *			\
			cos(2.0 * PI * F32_TO_D(f));	\
		d_a2 = K;				\
		d_b1 = 2.0  * R *			\
			cos(2.0 * PI * F32_TO_D(f));	\
		d_b2 = - 1.0 * R * R;			\
		__IMPL_BAND_FLTR(b, a, n,		\
			D_TO_F32(d_a0),			\
			D_TO_F32(d_a1),			\
			D_TO_F32(d_a2),			\
			D_TO_F32(d_b1),			\
			D_TO_F32(d_b2));		\
		} while (0)


/*
	d0	y[i-1]
	d1	cntr
	d2	cntr
	d3	x[i]
	d4	x[i+1]
	d5	x[i+2]
	d6	coef b1
	d7	coef b2
	
	a0	dst
	a1	src
	a2	x[i+3]
	a3	coef a0
	a4	coef a1 || b3
	a5	coef a2	|| b4




*/


/************************************************************************
* NAME: __IMPL_BAND_FLTR
*
* DESCRIPTION: 	Implementation for BAND filters
*
* NOTE: 	The elements type is assumed to be FRAC32
*
*************************************************************************/


#define __IMPL_BAND_FLTR(dst,src,size,c_a0,c_a1,c_a2,c_b1,c_b2,) do {	\
		FRAC32 *bb = dst;			\
		FRAC32 *aa = src;			\
		long nn = (size);			\
		FRAC32 coef_a0 = c_a0;			\
		FRAC32 coef_a1 = c_a1;			\
		FRAC32 coef_a2 = c_a2;			\
		FRAC32 coef_b1 = c_b1;			\
		FRAC32 coef_b2 = c_b2;			\
	 	asm{					\
			/* Save registers */\
      lea -60(a7),a7			;\
			movem.l d0-d7/a0-a5,(a7)	;\
			/* Set eMAC to fractional mode */\
			move.l #0x20,d0			;\
			move.l d0,MACSR			;\
			/* Load function variables */\
			move.l bb,a0			;\
			move.l aa,a1			;\
			move.l coef_a0,a3		;\
			move.l coef_a1,a4		;\
			move.l coef_a2,a5		;\
			move.l coef_b1,d6		;\
			move.l coef_b2,d7		;\
			/* Clean registers and accumulators */  \
			moveq.l %0,d0			;\
			moveq.l %0,d3			;\
			moveq.l %0,d4			;\
			moveq.l %0,d5			;\
			move.l #0,ACC0			;\
			move.l #0,ACC1			;\
			move.l #0,ACC2			;\
			move.l #0,ACC3			;\
			/* Load counter */  \
			move.l nn,d1			;\
			move.l d1,d2			;\
			/* Load in d1 number of 2 operations */ \
			asr.l %1,d1			;\
			beq *+64			;\
			/* Perform the filter 2 by 2 operations */ \
			move.l (a1)+,a2			;\
			mac.l a3,a2,ACC0		;\
			mac.l a4,d4,ACC0		;\
			mac.l a5,d5,ACC0		;\
			mac.l d6,d0,ACC0		;\
			mac.l d7,d3,ACC0		;\
			/* Store result and clear accumulator */ \
			movclr.l ACC0,d3		;\
			move.l d3,(a0)+			;\
			move.l (a1)+,d5			;\
			mac.l a3,d5,ACC0		;\
			mac.l a4,a2,ACC0		;\
			mac.l a5,d4,ACC0		;\
			mac.l d6,d3,ACC0		;\
			mac.l d7,d0,ACC0		;\
			/* Store result and clear accumulator */ \
			movclr.l ACC0,d0		;\
			move.l d0,(a0)+			;\
			move.l d5,d4			;\
			move.l a2,d5			;\
			subq.l %1,d1			;\
			bne *-60			;\
			/* Load in d2 number of operations left */\
			and.l %1,d2			;\
			beq *+28			;\
			/* Perform last operations */\
			move.l (a1)+,a2			;\
			mac.l a3,a2,ACC0		;\
			mac.l a4,d4,ACC0		;\
			mac.l a5,d5,ACC0		;\
			mac.l d6,d0,ACC0		;\
			mac.l d7,d3,ACC0		;\
			/* Store result and clear accumulator */ \
			movclr.l ACC0,d3		;\
			move.l d3,(a0)+			;\
			/* Restore registers */  \
			movem.l (a7),d0-d7/a0-a5	;\
			lea 60(a7),a7	    		;\
			};				\
		} while (0)


#endif // __BAND_FLTR_H
