/*
 * File:		sysinit.c
 * Purpose:		Reset configuration of the M52233DEMO
 *
 * Notes:
 */

#include "m52233DEMO.h"
#include "common.h"

/********************************************************************/

void mcf52233_init(void);
void mcf52233_wtm_init(void);
void mcf52233_pll_init(void);
void mcf52233_uart_init(void);
void mcf52233_scm_init(void);
void mcf52233_gpio_init(void);

/********************************************************************/
void
mcf52233_init(void)
{
	extern char __DATA_ROM[];
	extern char __DATA_RAM[];
	extern char __DATA_END[];
	extern char __BSS_START[];
	extern char __BSS_END[];
	extern uint32 VECTOR_TABLE[];
	extern uint32 __VECTOR_RAM[];
	register uint32 n;
	register uint8 *dp, *sp;


    /* 
     * Allow interrupts from ABORT, SW1, SW2, and SW3 (IRQ[1,4,7,11]) 
     */
     
    /* Enable IRQ signals on the port */
    MCF_GPIO_PNQPAR = 0
        | MCF_GPIO_PNQPAR_IRQ1_IRQ1
        | MCF_GPIO_PNQPAR_IRQ4_IRQ4
        | MCF_GPIO_PNQPAR_IRQ7_IRQ7;
    
    MCF_GPIO_PGPPAR = 0
        | MCF_GPIO_PGPPAR_IRQ11_IRQ11;
    
    /* Set EPORT to look for rising edges */
    MCF_EPORT_EPPAR0 = 0
        | MCF_EPORT_EPPAR_EPPA1_RISING
        | MCF_EPORT_EPPAR_EPPA4_RISING
        | MCF_EPORT_EPPAR_EPPA7_RISING;
        
    MCF_EPORT_EPPAR1 = 0
        | MCF_EPORT_EPPAR_EPPA11_RISING;
        
    /* Clear any currently triggered events on the EPORT  */
    MCF_EPORT_EPIER0 = 0
        | MCF_EPORT_EPIER_EPIE1
        | MCF_EPORT_EPIER_EPIE4
        | MCF_EPORT_EPIER_EPIE7;
       
    MCF_EPORT_EPIER1 = 0
        | MCF_EPORT_EPIER_EPIE11;
       
    /* Enable interrupts in the interrupt controller */
    MCF_INTC0_IMRL &= ~(0
        | MCF_INTC_IMRL_MASK1 
        | MCF_INTC_IMRL_MASK4 
        | MCF_INTC_IMRL_MASK7 
        | MCF_INTC_IMRL_MASKALL);

    MCF_INTC1_IMRH &= ~(0
        | MCF_INTC_IMRH_MASK35);
        
	MCF_INTC1_ICR35 = MCF_INTC_ICR_IL(4);

	MCF_GPIO_PDDPAR = 0x0F;
	
	/* Set real time clock freq */

	MCF_CLOCK_RTCDR = 25000000;
	
	mcf52233_wtm_init();
	mcf52233_pll_init();
	mcf52233_scm_init();
	mcf52233_uart_init();
	mcf52233_gpio_init();
	
	/* Turn Instruction Cache ON */
	mcf5xxx_wr_cacr(0
		| MCF5XXX_CACR_CENB
		| MCF5XXX_CACR_CINV
		| MCF5XXX_CACR_DISD
		| MCF5XXX_CACR_CEIB
		| MCF5XXX_CACR_CLNF_00);

	/* Copy the vector table to RAM */
	if (__VECTOR_RAM != VECTOR_TABLE)
	{
		for (n = 0; n < 256; n++)
			__VECTOR_RAM[n] = VECTOR_TABLE[n];
	}
	mcf5xxx_wr_vbr((uint32)__VECTOR_RAM);

	/*
	 * Move initialized data from ROM to RAM.
	 */
	if (__DATA_ROM != __DATA_RAM)
	{
		dp = (uint8 *)__DATA_RAM;
		sp = (uint8 *)__DATA_ROM;
		n = (uint32)(__DATA_END - __DATA_RAM);
		while (n--)
			*dp++ = *sp++;
	}

	/*
	 * Zero uninitialized data
	 */
	if (__BSS_START != __BSS_END)
	{
		sp = (uint8 *)__BSS_START;
		n = (uint32)(__BSS_END - __BSS_START);
		while (n--)
			*sp++ = 0;
	}
}
/********************************************************************/
void
mcf52233_wtm_init(void)
{
	/*
	 * Disable Software Watchdog Timer
	 */
	MCF_SCM_CWCR = 0;
}
/********************************************************************/
void
mcf52233_pll_init(void)
{

//MCF_CLOCK_CCHR =0x05; // The PLL pre divider - 25MHz / 5 = 5MHz 

	/* The PLL pre-divider affects this!!! 
	 * Multiply 25Mhz reference crystal /CCHR by 12 to acheive system clock of 60Mhz
	 */

	MCF_CLOCK_SYNCR = MCF_CLOCK_SYNCR_MFD(4) | MCF_CLOCK_SYNCR_CLKSRC| MCF_CLOCK_SYNCR_PLLMODE | MCF_CLOCK_SYNCR_PLLEN ;

	while (!(MCF_CLOCK_SYNSR & MCF_CLOCK_SYNSR_LOCK))
	{
	}
}
/********************************************************************/
void
mcf52233_scm_init(void)
{
	/*
	 * Enable on-chip modules to access internal SRAM
	 */
	MCF_SCM_RAMBAR = (0
		| MCF_SCM_RAMBAR_BA(SRAM_ADDRESS)
		| MCF_SCM_RAMBAR_BDE);
}
/********************************************************************/
void
mcf52233_gpio_init(void)
{
  uint32 myctr; 		//generic counter variable

	/*
	 * Initialize PLDPAR to enable Ethernet Leds
	 */
  MCF_GPIO_PLDPAR = (0
  
		 | MCF_GPIO_PLDPAR_ACTLED_ACTLED 
		 | MCF_GPIO_PLDPAR_LNKLED_LNKLED 
		 | MCF_GPIO_PLDPAR_SPDLED_SPDLED 
		 | MCF_GPIO_PLDPAR_DUPLED_DUPLED 
		 | MCF_GPIO_PLDPAR_COLLED_COLLED 
		 | MCF_GPIO_PLDPAR_RXLED_RXLED   
		 | MCF_GPIO_PLDPAR_TXLED_TXLED);  

	/*
	 * Initialize Port TA to enable Axcel control
	 */
	MCF_GPIO_PTAPAR = 0x00; 
	MCF_GPIO_DDRTA  = 0x0F;
	MCF_GPIO_PORTTA = 0x04;
			  
  // set phy address to zero
  MCF_PHY_EPHYCTL1 = MCF_PHY_EPHYCTL1_PHYADDR(FEC_PHY0); //	FEC_PHY0 from m52233evb.h 

  //Enable EPHY module with PHY clocks disabled
  //Do not turn on PHY clocks until both FEC and EPHY are completely setup (see Below)
  MCF_PHY_EPHYCTL0 = (uint8)(MCF_PHY_EPHYCTL0_DIS100 | MCF_PHY_EPHYCTL0_DIS10);
  
  //Enable auto_neg at start-up
  MCF_PHY_EPHYCTL0 = (uint8)(MCF_PHY_EPHYCTL0 & (MCF_PHY_EPHYCTL0_ANDIS));

  //Enable EPHY module
  MCF_PHY_EPHYCTL0 = (uint8)(MCF_PHY_EPHYCTL0_EPHYEN | MCF_PHY_EPHYCTL0);
  //Let PHY PLLs be determined by PHY
  MCF_PHY_EPHYCTL0 = (uint8)(MCF_PHY_EPHYCTL0  & ~(MCF_PHY_EPHYCTL0_DIS100 | MCF_PHY_EPHYCTL0_DIS10)); 

  //printf("ePHY enabled \n\r");
	
  //DELAY, Delay start-up
  for (myctr=200000; myctr >0; myctr--)
  {
  }
  
  //printf("ePHY Ready \n\r");
}
/********************************************************************/
void
mcf52233_uart_init(void)
{
	/*
	 * Initialize all three UARTs for serial communications
	 */

	register uint16 ubgs;

	/*
	 * Set Port UA to initialize URXD0/UTXD0
	 */
    MCF_GPIO_PUAPAR = 0
        | MCF_GPIO_PUAPAR_RXD0_RXD0
        | MCF_GPIO_PUAPAR_TXD0_TXD0;

    MCF_GPIO_PUBPAR = 0
        | MCF_GPIO_PUBPAR_RXD1_RXD1
        | MCF_GPIO_PUBPAR_TXD1_TXD1;

    MCF_GPIO_PUCPAR = 0
        | MCF_GPIO_PUCPAR_RXD2_RXD2
        | MCF_GPIO_PUCPAR_TXD2_TXD2;

	/*
	 * Reset Transmitter
	 */
	MCF_UART0_UCR = MCF_UART_UCR_RESET_TX;
	MCF_UART1_UCR = MCF_UART_UCR_RESET_TX;
	MCF_UART2_UCR = MCF_UART_UCR_RESET_TX;

	/*
	 * Reset Receiver
	 */
	MCF_UART0_UCR = MCF_UART_UCR_RESET_RX;
	MCF_UART1_UCR = MCF_UART_UCR_RESET_RX;
	MCF_UART2_UCR = MCF_UART_UCR_RESET_RX;

	/*
	 * Reset Mode Register
	 */
	MCF_UART0_UCR = MCF_UART_UCR_RESET_MR;
	MCF_UART1_UCR = MCF_UART_UCR_RESET_MR;
	MCF_UART2_UCR = MCF_UART_UCR_RESET_MR;

	/*
	 * No parity, 8-bits per character
	 */
	MCF_UART0_UMR = (0
		| MCF_UART_UMR_PM_NONE
		| MCF_UART_UMR_BC_8 );
	MCF_UART1_UMR = (0
		| MCF_UART_UMR_PM_NONE
		| MCF_UART_UMR_BC_8 );
	MCF_UART2_UMR = (0
		| MCF_UART_UMR_PM_NONE
		| MCF_UART_UMR_BC_8 );

	/*
	 * No echo or loopback, 1 stop bit
	 */
	MCF_UART0_UMR = (0
		| MCF_UART_UMR_CM_NORMAL
		| MCF_UART_UMR_SB_STOP_BITS_1);
	MCF_UART1_UMR = (0
		| MCF_UART_UMR_CM_NORMAL
		| MCF_UART_UMR_SB_STOP_BITS_1);
	MCF_UART2_UMR = (0
		| MCF_UART_UMR_CM_NORMAL
		| MCF_UART_UMR_SB_STOP_BITS_1);

	/*
	 * Set Rx and Tx baud by SYSTEM CLOCK
	 */
	MCF_UART0_UCSR = (0
		| MCF_UART_UCSR_RCS_SYS_CLK
		| MCF_UART_UCSR_TCS_SYS_CLK);
	MCF_UART1_UCSR = (0
		| MCF_UART_UCSR_RCS_SYS_CLK
		| MCF_UART_UCSR_TCS_SYS_CLK);
	MCF_UART2_UCSR = (0
		| MCF_UART_UCSR_RCS_SYS_CLK
		| MCF_UART_UCSR_TCS_SYS_CLK);

	/*
	 * Mask all UART interrupts
	 */
	MCF_UART0_UIMR = 0;
	MCF_UART1_UIMR = 0;
	MCF_UART2_UIMR = 0;

	/*
	 * Calculate baud settings
	 */
	ubgs = (uint16)((SYSTEM_CLOCK*1000000)/(UART_BAUD * 32));

	MCF_UART0_UBG1 = (uint8)((ubgs & 0xFF00) >> 8);
	MCF_UART0_UBG2 = (uint8)(ubgs & 0x00FF);
	MCF_UART1_UBG1 = (uint8)((ubgs & 0xFF00) >> 8);
	MCF_UART1_UBG2 = (uint8)(ubgs & 0x00FF);
	MCF_UART2_UBG1 = (uint8)((ubgs & 0xFF00) >> 8);
	MCF_UART2_UBG2 = (uint8)(ubgs & 0x00FF);

	/*
	 * Enable receiver and transmitter
	 */
	MCF_UART0_UCR = (0
		| MCF_UART_UCR_TX_ENABLED
		| MCF_UART_UCR_RX_ENABLED);
	MCF_UART1_UCR = (0
		| MCF_UART_UCR_TX_ENABLED
		| MCF_UART_UCR_RX_ENABLED);
	MCF_UART2_UCR = (0
		| MCF_UART_UCR_TX_ENABLED
		| MCF_UART_UCR_RX_ENABLED);

}
/********************************************************************/
