/*
 * File:    mii.c
 * Purpose:     
 *
 * Notes:       
 */

#include "common.h"

/********************************************************************/
/*
 * Write a value to a PHY's MII register.
 *
 * Parameters:
 *  phy_addr    Address of the PHY.
 *  reg_addr    Address of the register in the PHY.
 *  data        Data to be written to the PHY register.
 *
 * Return Values:
 *  0 on failure
 *  1 on success.
 *
 * Please refer to your PHY manual for registers and their meanings.
 * mii_write() polls for the FEC's MII interrupt event and clears it. 
 * If after a suitable amount of time the event isn't triggered, a 
 * value of 0 is returned.
 */
int
fec_mii_write(int phy_addr, int reg_addr, int data)
{
    int timeout;
    uint32 eimr;

    /* Clear the MII interrupt bit */
    MCF_FEC_EIR = MCF_FEC_EIR_MII;

    /* Mask the MII interrupt */
    eimr = MCF_FEC_EIMR;
    MCF_FEC_EIMR &= ~MCF_FEC_EIMR_MII;

    /* Write to the MII Management Frame Register to kick-off the MII write */
    MCF_FEC_MMFR = (vuint32)(0
        | MCF_FEC_MMFR_ST_01
        | MCF_FEC_MMFR_OP_WRITE
        | MCF_FEC_MMFR_PA(phy_addr)
        | MCF_FEC_MMFR_RA(reg_addr)
        | MCF_FEC_MMFR_TA_10
        | MCF_FEC_MMFR_DATA(data));

    /* Poll for the MII interrupt (interrupt should be masked) */
    for (timeout = 0; timeout < FEC_MII_TIMEOUT; timeout++)
    {
        if (MCF_FEC_EIR & MCF_FEC_EIR_MII)
            break;
    }
    if(timeout == FEC_MII_TIMEOUT)
        return 0;

    /* Clear the MII interrupt bit */
    MCF_FEC_EIR = MCF_FEC_EIR_MII;

    /* Restore the EIMR */
    MCF_FEC_EIMR = eimr;

    return 1;
}
/********************************************************************/
/*
 * Read a value from a PHY's MII register.
 *
 * Parameters:
 *  phy_addr    Address of the PHY.
 *  reg_addr    Address of the register in the PHY.
 *  data        Pointer to storage for the Data to be read
 *              from the PHY register (passed by reference)
 *
 * Return Values:
 *  0 on failure
 *  1 on success.
 *
 * Please refer to your PHY manual for registers and their meanings.
 * mii_read() polls for the FEC's MII interrupt event and clears it. 
 * If after a suitable amount of time the event isn't triggered, a 
 * value of 0 is returned.
 */
int
fec_mii_read(int phy_addr, int reg_addr, uint16* data)
{
    int timeout;
    uint32 eimr;

    /* Clear the MII interrupt bit */
    MCF_FEC_EIR = MCF_FEC_EIR_MII;

    /* Mask the MII interrupt */
    eimr = MCF_FEC_EIMR;
    MCF_FEC_EIMR &= ~MCF_FEC_EIMR_MII;

    /* Write to the MII Management Frame Register to kick-off the MII read */
    MCF_FEC_MMFR = (vuint32)(0
        | MCF_FEC_MMFR_ST_01
        | MCF_FEC_MMFR_OP_READ
        | MCF_FEC_MMFR_PA(phy_addr)
        | MCF_FEC_MMFR_RA(reg_addr)
        | MCF_FEC_MMFR_TA_10);

    /* Poll for the MII interrupt (interrupt should be masked) */
    for (timeout = 0; timeout < FEC_MII_TIMEOUT; timeout++)
    {
        if (MCF_FEC_EIR & MCF_FEC_EIR_MII)
            break;
    }

    if(timeout == FEC_MII_TIMEOUT)
        return 0;

    /* Clear the MII interrupt bit */
    MCF_FEC_EIR = MCF_FEC_EIR_MII;

    /* Restore the EIMR */
    MCF_FEC_EIMR = eimr;

    *data = (uint16)(MCF_FEC_MMFR & 0x0000FFFF);

    return 1;
}
/********************************************************************/
/*
 * Initialize the MII interface controller
 *
 * Parameters:
 *  sys_clk System Clock Frequency (in MHz)
 */
void
fec_mii_init(int sys_clk)
{
    /*
     * Initialize the MII clock (EMDC) frequency
     *
     * Desired MII clock is 2.5MHz
     * MII Speed Setting = System_Clock / (2.5MHz * 2)
     * (plus 1 to make sure we round up)
     */
    MCF_FEC_MSCR = MCF_FEC_MSCR_MII_SPEED((uint32)((sys_clk/5)+1));
    /*
     * Make sure the external MII interface signals are enabled
     */
// not needed on 5223
//	MCF_GPIO_PASPAR = (MCF_GPIO_PASPAR & 0xF0FF) | 0x0F00;
}
/********************************************************************/
/********************************************************************/
/*
 * Print the MII interface controller registers
 *
 * Parameters:
 *  none
 */
void
fec_mii_reg_printf(void)
{
 uint16 settings;


        if (!fec_mii_read(0, 0, &settings))
            printf("Control Register =  Error  \n\r\r");
        else
			printf("Control Register = %04x\n\r\r",settings);

        if (!fec_mii_read(0, 1, &settings))
            printf("Status Register  =  Error  \n\r\r");
        else
			printf("Status Register  = %04x\n\r\r",settings);

        if (!fec_mii_read(0, 2, &settings))
            printf("PHY Identifier 1 Register        =  Error  \n\r\r");
        else
			printf("PHY Identifier 1 Register        = %04x\n\r\r",settings);

        if (!fec_mii_read(0, 3, &settings))
            printf("PHY Identifier 2 Register        =  Error  \n\r");
        else
			printf("PHY Identifier 2 Register        = %04x\n\r",settings);

        if (!fec_mii_read(0, 4, &settings))
            printf("Auto-Neg. Advertisement Register =  Error  \n\r");
        else
			printf("Auto-Neg. Advertisement Register = %04x\n\r",settings);

        if (!fec_mii_read(0, 5, &settings))
            printf("Link Partner Ability Register    =  Error  \n\r");
        else
			printf("Link Partner Ability Register    = %04x\n\r",settings);

        if (!fec_mii_read(0, 6, &settings))
            printf("Auto-Neg. Expansion Register     =  Error  \n\r");
        else
			printf("Auto-Neg. Expansion Register     = %04x\n\r",settings);

        if (!fec_mii_read(0, 7, &settings))
            printf("Auto-Neg. Next Page Transmit     =  Error  \n\r");
        else
			printf("Auto-Neg. Next Page Transmit     = %04x\n\r",settings);

        if (!fec_mii_read(0, 16, &settings))
            printf("Interrupt Register Register      =  Error  \n\r");
        else
			printf("Interrupt Register Register      = %04x\n\r",settings);

        if (!fec_mii_read(0, 17, &settings))
            printf("Proprietary Status Register      =  Error  \n\r");
        else
			printf("Proprietary Status Register      = %04x\n\r",settings);

        if (!fec_mii_read(0, 18, &settings))
            printf("Proprietary Control Register     =  Error  \n\r");
        else
			printf("Proprietary Control Register     = %04x\n\r",settings);
        
        if (!fec_mii_read(0, 20, &settings))
            printf("Register 20  =  Error  \n\r");
        else
			printf("Register 20  = %04x\n\r",settings);

}
/********************************************************************/
/********************************************************************/
/*
 * Print the MII block counters registers
 *
 * Parameters:
 *  none
 */
void
fec_mii_blk_init(void)
{
MCF_FEC_RMON_T_DROP = 0;
MCF_FEC_RMON_T_PACKETS = 0;
MCF_FEC_RMON_T_BC_PKT = 0;
MCF_FEC_RMON_T_MC_PKT = 0;
MCF_FEC_RMON_T_CRC_ALIGN = 0; 
MCF_FEC_RMON_T_UNDERSIZE = 0;
MCF_FEC_RMON_T_OVERSIZE = 0;
MCF_FEC_RMON_T_FRAG = 0;
MCF_FEC_RMON_T_JAB = 0;   
MCF_FEC_RMON_T_COL = 0;    
MCF_FEC_RMON_T_P64 = 0;    
MCF_FEC_RMON_T_P65TO127 = 0;    
MCF_FEC_RMON_T_P128TO255 = 0;
MCF_FEC_RMON_T_P256TO511 = 0;
MCF_FEC_RMON_T_P512TO1023 = 0;
MCF_FEC_RMON_T_P1024TO2047 = 0;
MCF_FEC_RMON_T_P_GTE2048 = 0;
MCF_FEC_RMON_T_OCTETS = 0;
MCF_FEC_IEEE_T_DROP = 0; 
MCF_FEC_IEEE_T_FRAME_OK = 0;   
MCF_FEC_IEEE_T_1COL = 0;   
MCF_FEC_IEEE_T_MCOL = 0;   
MCF_FEC_IEEE_T_DEF = 0;   
MCF_FEC_IEEE_T_LCOL = 0;    
MCF_FEC_IEEE_T_EXCOL = 0;   
MCF_FEC_IEEE_T_MACERR = 0;  
MCF_FEC_IEEE_T_CSERR = 0; 
MCF_FEC_IEEE_T_SQE = 0;  
MCF_FEC_IEEE_T_FDXFC = 0;    
MCF_FEC_IEEE_T_OCTETS_OK = 0; 
(*(vuint32*)(&__IPSBAR[0x001278])) =0;
(*(vuint32*)(&__IPSBAR[0x00127C])) =0;
(*(vuint32*)(&__IPSBAR[0x001280])) =0;
MCF_FEC_RMON_R_PACKETS = 0;
MCF_FEC_RMON_R_BC_PKT = 0;
MCF_FEC_RMON_R_MC_PKT = 0;      
MCF_FEC_RMON_R_CRC_ALIGN = 0;   
MCF_FEC_RMON_R_UNDERSIZE = 0;   
MCF_FEC_RMON_R_OVERSIZE = 0;    
MCF_FEC_RMON_R_FRAG = 0;        
MCF_FEC_RMON_R_JAB = 0;         
MCF_FEC_RMON_R_RESVD_0 = 0;     
MCF_FEC_RMON_R_P64 = 0;         
MCF_FEC_RMON_R_P65TO127 = 0;    
MCF_FEC_RMON_R_P128TO255 = 0;   
MCF_FEC_RMON_R_P256TO511 = 0;   
MCF_FEC_RMON_R_512TO1023 = 0;   
MCF_FEC_RMON_R_P_GTE2048 = 0;   
MCF_FEC_RMON_R_1024TO2047 = 0;  
MCF_FEC_RMON_R_OCTETS = 0;      
MCF_FEC_IEEE_R_DROP = 0;        
MCF_FEC_IEEE_R_FRAME_OK = 0;    
MCF_FEC_IEEE_R_CRC = 0;         
MCF_FEC_IEEE_R_ALIGN = 0;       
MCF_FEC_IEEE_R_MACERR = 0;      
MCF_FEC_IEEE_R_FDXFC = 0;       
MCF_FEC_IEEE_R_OCTETS_OK = 0; 
}    
void
fec_mii_blk_printf(void)
{
    printf("\n\r");
	printf("MCF_FEC_RMON_T_DROP        = %08x\n\r",MCF_FEC_RMON_T_DROP);
	printf("MCF_FEC_RMON_T_PACKETS     = %08x\n\r",MCF_FEC_RMON_T_PACKETS);
	printf("MCF_FEC_RMON_T_BC_PKT      = %08x\n\r",MCF_FEC_RMON_T_BC_PKT);
	printf("MCF_FEC_RMON_T_MC_PKT      = %08x\n\r",MCF_FEC_RMON_T_MC_PKT);
	printf("MCF_FEC_RMON_T_CRC_ALIGN   = %08x\n\r",MCF_FEC_RMON_T_CRC_ALIGN);
	printf("MCF_FEC_RMON_T_UNDERSIZE   = %08x\n\r",MCF_FEC_RMON_T_UNDERSIZE);
	printf("MCF_FEC_RMON_T_FRAG        = %08x\n\r",MCF_FEC_RMON_T_FRAG);
	printf("MCF_FEC_RMON_T_JAB         = %08x\n\r",MCF_FEC_RMON_T_JAB);
	printf("MCF_FEC_RMON_T_COL         = %08x\n\r",MCF_FEC_RMON_T_COL);
	printf("MCF_FEC_RMON_T_P64         = %08x\n\r",MCF_FEC_RMON_T_P64);
	printf("MCF_FEC_RMON_T_P65TO127    = %08x\n\r",MCF_FEC_RMON_T_P65TO127);
	printf("MCF_FEC_RMON_T_P128TO255   = %08x\n\r",MCF_FEC_RMON_T_P128TO255);
	printf("MCF_FEC_RMON_T_P256TO511   = %08x\n\r",MCF_FEC_RMON_T_P256TO511);
	printf("MCF_FEC_RMON_T_P512TO1023  = %08x\n\r",MCF_FEC_RMON_T_P512TO1023);
	printf("MCF_FEC_RMON_T_P1024TO2047 = %08x\n\r",MCF_FEC_RMON_T_P1024TO2047);
	printf("MCF_FEC_RMON_T_P_GTE2048   = %08x\n\r",MCF_FEC_RMON_T_P_GTE2048);
	printf("MCF_FEC_RMON_T_OCTETS      = %08x\n\r",MCF_FEC_RMON_T_OCTETS);
	printf("MCF_FEC_IEEE_T_DROP        = %08x\n\r",MCF_FEC_IEEE_T_DROP);
	printf("MCF_FEC_IEEE_T_FRAME_OK    = %08x\n\r",MCF_FEC_IEEE_T_FRAME_OK);
	printf("MCF_FEC_IEEE_T_1COL        = %08x\n\r",MCF_FEC_IEEE_T_1COL);
	printf("MCF_FEC_IEEE_T_MCOL        = %08x\n\r",MCF_FEC_IEEE_T_MCOL);
	printf("MCF_FEC_IEEE_T_DEF         = %08x\n\r",MCF_FEC_IEEE_T_DEF);
	printf("MCF_FEC_IEEE_T_LCOL        = %08x\n\r",MCF_FEC_IEEE_T_LCOL);
	printf("MCF_FEC_IEEE_T_EXCOL       = %08x\n\r",MCF_FEC_IEEE_T_EXCOL);
	printf("MCF_FEC_IEEE_T_MACERR      = %08x\n\r",MCF_FEC_IEEE_T_MACERR);
	printf("MCF_FEC_IEEE_T_MACERR      = %08x\n\r",MCF_FEC_IEEE_T_MACERR);
	printf("MCF_FEC_IEEE_T_CSERR       = %08x\n\r",MCF_FEC_IEEE_T_CSERR);
	printf("MCF_FEC_IEEE_T_SQE         = %08x\n\r",MCF_FEC_IEEE_T_SQE);
	printf("MCF_FEC_IEEE_T_FDXFC       = %08x\n\r",MCF_FEC_IEEE_T_FDXFC);
	printf("MCF_FEC_IEEE_T_OCTETS_OK   = %08x\n\r",MCF_FEC_IEEE_T_OCTETS_OK);
	printf("MCF_FEC_RMON_R_PACKETS     = %08x\n\r",MCF_FEC_RMON_R_PACKETS);
	printf("MCF_FEC_RMON_R_BC_PKT      = %08x\n\r",MCF_FEC_RMON_R_BC_PKT);
	printf("MCF_FEC_RMON_R_CRC_ALIGN   = %08x\n\r",MCF_FEC_RMON_R_CRC_ALIGN);
	printf("MCF_FEC_RMON_R_UNDERSIZE   = %08x\n\r",MCF_FEC_RMON_R_UNDERSIZE);
	printf("MCF_FEC_RMON_R_OVERSIZE    = %08x\n\r",MCF_FEC_RMON_R_OVERSIZE);
	printf("MCF_FEC_RMON_R_FRAG        = %08x\n\r",MCF_FEC_RMON_R_FRAG);
	printf("MCF_FEC_RMON_R_JAB         = %08x\n\r",MCF_FEC_RMON_R_JAB);
	printf("MCF_FEC_RMON_R_RESVD_0     = %08x\n\r",MCF_FEC_RMON_R_RESVD_0);
	printf("MCF_FEC_RMON_R_P64         = %08x\n\r",MCF_FEC_RMON_R_P64);
	printf("MCF_FEC_RMON_R_P64         = %08x\n\r",MCF_FEC_RMON_R_P64);
	printf("MCF_FEC_RMON_R_P65TO127    = %08x\n\r",MCF_FEC_RMON_R_P65TO127);
	printf("MCF_FEC_RMON_R_P128TO255   = %08x\n\r",MCF_FEC_RMON_R_P128TO255);
	printf("MCF_FEC_RMON_R_P256TO511   = %08x\n\r",MCF_FEC_RMON_R_P256TO511);
	printf("MCF_FEC_RMON_R_512TO1023   = %08x\n\r",MCF_FEC_RMON_R_512TO1023);
	printf("MCF_FEC_RMON_R_P_GTE2048   = %08x\n\r",MCF_FEC_RMON_R_P_GTE2048);
	printf("MCF_FEC_RMON_R_1024TO2047  = %08x\n\r",MCF_FEC_RMON_R_1024TO2047);
	printf("MCF_FEC_RMON_R_OCTETS      = %08x\n\r",MCF_FEC_RMON_R_OCTETS);
	printf("MCF_FEC_IEEE_R_DROP        = %08x\n\r",MCF_FEC_IEEE_R_DROP);
	printf("MCF_FEC_IEEE_R_FRAME_OK    = %08x\n\r",MCF_FEC_IEEE_R_FRAME_OK);
	printf("MCF_FEC_IEEE_R_CRC         = %08x\n\r",MCF_FEC_IEEE_R_CRC);
	printf("MCF_FEC_IEEE_R_ALIGN       = %08x\n\r",MCF_FEC_IEEE_R_ALIGN);
	printf("MCF_FEC_IEEE_R_MACERR      = %08x\n\r",MCF_FEC_IEEE_R_MACERR);
	printf("MCF_FEC_IEEE_R_FDXFC       = %08x\n\r",MCF_FEC_IEEE_R_FDXFC);
	printf("MCF_FEC_IEEE_R_OCTETS_OK   = %08x\n\r",MCF_FEC_IEEE_R_OCTETS_OK);
}
void
fec_mii_blk_test(void)
{
	vuint32 *i;
	for (i = &MCF_FEC_RMON_T_DROP; i<&MCF_FEC_IEEE_R_OCTETS_OK; i++)
	{
		if(*i!=0)
		{
		fec_mii_blk_printf();
		}
	}
}