/*
 * File:		eth_stub.h
 * Purpose:		Wrapper functions betw. freescale fec, nif & nbuf code 
 *
 * IMPORTANT. Read the following Freescale Semiconductor Software License 
 * Agreement:
 * http://www.freescale.com/files/disclaimer/LA_OPT1.html
 * 
 */
#include "common.h"

#ifndef __INCLUDE_ETH_STUB_H__
#define __INCLUDE_ETH_STUB_H__


/* Buffer sizes	*/

#define	NETWORK_TX_BUFFER_SIZE	1400		/* Always keep odd*/ //jpw  Why?

/* Symbolic constants	*/

#define	ETH_ADDRESS_LEN	6

/* Protocol Constants	*/

#define	ETH_HEADER_LEN		14
#define ETH_CHIP_HEADER_LEN	4
#define	ETH_MTU				1500

#define	PROTOCOL_IP			0x0800			/**< IP over Ethernet	*/
#define PROTOCOL_IPv6		0x86DD			/**< IPv6 over Ethernet	*/
#define PROTOCOL_ARP		0x0806			/**< ARP over Ethernet 	*/

/* Buffer addresses */
#define	ARP_BUFFER	0x5F	/**< 256 byte Tx for ARP 		 */
#define	ICMP_BUF	0x4D	/**< 1536 byte Tx for ICMP	 */
#define	TCP_BUF		0x53	/**< 1536 byte Tx for TCP		 */
#define	UDP_BUF		0x59	/**< 1536 byte Tx for UDP		 */

/** \struct ethernet_frame 
 *	\brief Ethernet packet header fields
 *
 *	This structure holds information about the Ethernet packets. In
 *	addition to standard Ethernet header (destination HW address, source
 *	HW address, frame size and protocol), buff_index is added. This variable
 *	is used by higher level protocols (IP, ARP or other) to initialize
 *	reading of the Ethernet packet by invoking NETWORK_RECEIVE_INITIALIZE
 *	macro to initialize reading of the data carried in the Ethernet packet (
 *	not the Ethernet header itself!).
 */
struct ethernet_frame
{
	UINT8	destination[ETH_ADDRESS_LEN];	/**< destination hardware address
											 *	 as read from the received
											 *	 ethernet packet
											 */
	UINT8	source[ETH_ADDRESS_LEN];		/**< source hardware address
											 *	 as read from the received
											 *	 ethernet packet
											 */
	UINT16	frame_size;						/**< size of the received
											 *	 Ethernet packet
											 */	
	UINT16	protocol;						/**< protocol field of the
											 * 	 Ethernet header. For now we
											 *	 work with:
											 *		\li PROTOCOL_IP	- 0x0800
											 *		\li PROTOCOL_ARP - 0x0806
											 */
	UINT16	buf_index;						/**< Address in the Ethernet
											 * 	 controllers buffer where
											 *	 data can be read from
											 */

};

/* Frame we receive	*/
//extern struct ethernet_frame received_frame;

/* Frame we send	*/
//extern struct ethernet_frame send_frame;

/********************************************************************************
Function:		eth_stub_init

Parameters:		void

Return val:		void


Date:			22.1.2004

Desc:			Call this function on init before entering to main loop.
				Just initializes the eth_wrapper
*********************************************************************************/
void eth_stub_init(void);

/********************************************************************************
Function:		stack_fechandler

Parameters:		NIF *nif - network interface

Return val:		void


Date:			22.1.2004

Desc:			This function is called by nif_protocol_handler in order to
				supply received Ethernet packet. We just increase the number of
				unprocessed packets stack to poll.
*********************************************************************************/
void stack_fechandler(NIF *nif);

/********************************************************************************
Function:		eth_stub_isframe

Parameters:		NIF *nif - network interface

Return val:		INT16 : (<0) - Error
						(0) - No pending frames
						(>0) - There are pending frames


Date:			22.1.2004

Desc:			Checks do we have any pending frame
*********************************************************************************/
INT16 eth_stub_isframe(NIF *nif);

/********************************************************************************
Function:		eth_stub_dumpframe

Parameters:		NIF *nif - network interface

Return val:		INT16 : (<0) - Error
						(=>0) - OK, frame released


Date:			22.1.2004

Desc:			Releases the frame and decreases the number of pending packets
*********************************************************************************/
INT16 eth_stub_dumpframe(NIF *nif);

/********************************************************************************
Function:		eth_stub_seekread

Parameters:		NIF *nif - network interface
				UINT16 pos - position from the start of read buffer

Return val:		INT16 : (<0) - Error
						(=>0) - OK


Date:			22.1.2004

Desc:			Moves the read pointer to the given position from start
*********************************************************************************/
INT16 eth_stub_seekread(NIF *nif, UINT16 pos);

/********************************************************************************
Function:		eth_stub_read

Parameters:		NIF *nif - network interface

Return val:		UINT8 - read data


Date:			22.1.2004

Desc:			Return one byte of data from rx buffer and advances read pointer.
				TODO:More efficient to make this a MACRO
*********************************************************************************/
UINT8 eth_stub_read(NIF *nif);

/********************************************************************************
Function:		eth_stub_read_buf

Parameters:		NIF *nif - network interface
				UINT8* buf - buffer for data
				UINT16 buflen - number of bytes requested

Return val:		INT16: (-1) - Error
					   (>=0) - Number of bytes read


Date:			22.1.2004

Desc:			Reads multiple bytes from rx buffer and advances read pointer
*********************************************************************************/
INT16 eth_stub_read_buf(NIF *nif, UINT8* buf, UINT16 buflen);

/********************************************************************************
Function:		eth_stub_txalloc

Parameters:		NIF *nif - network interface

Return val:		INT16: (-1) - Error
					   (>=0) - OK, tx buffer allocated


Date:			22.1.2004

Desc:			Allocates tx buffer from nbuf
*********************************************************************************/
INT16 eth_stub_txalloc(NIF *nif);

/********************************************************************************
Function:		eth_stub_writeheader

Parameters:		NIF *nif - network interface
				struct otcp_ethframe* frame - ethernet header info

Return val:		INT16: (-1) - Error
					   (>=0) - OK


Date:			22.1.2004

Desc:			Writes ethernet header to tx buffer
*********************************************************************************/
INT16 eth_stub_writeheader(NIF *nif, struct ethernet_frame* frame);

/********************************************************************************
Function:		eth_stub_send

Parameters:		NIF *nif - network interface
				UINT16 len - length of packet without the eth header & CRR

Return val:		INT16: (-1) - Error
					   (>=0) - OK (number of bytes written)


Date:			22.1.2004

Desc:			Sends the ethernet frame
*********************************************************************************/
INT16 eth_stub_send(NIF *nif, UINT16 len);

/********************************************************************************
Function:		eth_stub_write

Parameters:		NIF *nif - network interface
				UINT8 dat - data byte to write

Return val:		INT16: (-1) - Error
					   (>=0) - OK (number of bytes written)


Date:			22.1.2004

Desc:			Writes single byte to tx buffer and increases write pointer
*********************************************************************************/
INT16 eth_stub_write(NIF *nif, UINT8 dat);

/********************************************************************************
Function:		eth_stub_write_buf

Parameters:		NIF *nif - network interface
				UINT8* buf - data to write
				UINT16 buflen - number of bytes to write

Return val:		INT16: (-1) - Error
					   (>=0) - OK (number of bytes written)


Date:			22.1.2004

Desc:			Writes multiple bytes to tx buffer and increases write pointer
*********************************************************************************/
INT16 eth_stub_write_buf(NIF *nif, UINT8* buf, UINT16 buflen);
#endif
/* EOF	*/
