/*
 * File:		ether_API.h
 * Purpose:		General definions, bindings and includes of stack.
				Processor depended stuff like enable/disable IRQs.
				Helper functions
 * IMPORTANT. Read the following Freescale Semiconductor Software License 
 * Agreement:
 * http://www.freescale.com/files/disclaimer/LA_OPT1.html
 * 
	Version Info:	01/24/2006 - First version
 */


/*********************************************************************************/

#ifndef __INCLUDE_ETHER_API_H__
#define __INCLUDE_ETHER_API_H__




/* System functions	*/

#define RESET_SYSTEM()			{}

#define DISABLE_INTERRUPTS()	{}
#define ENABLE_INTERRUPTS()		{}

#define DISABLE_TIMER_IRQ()		MCF_INTC0_IMRH |= MCF_INTC_IMRH_INT55
#define ENABLE_TIMER_IRQ()		MCF_INTC0_IMRH &= ~MCF_INTC_IMRH_INT55

extern UINT8 net_buf[NETWORK_TX_BUFFER_SIZE];	/* Network transmit buffer	*/



/* the network interface for freescale code	*/

extern NIF fec_nif;

/* and for OpenTCP internals	*/
extern struct netif localmachine;

/* rx & tx ethernet frames on OpenTCP	*/

extern struct ethernet_frame received_frame;	/* Frame we receive	*/
extern struct ethernet_frame send_frame;	/* Frame we send	*/

/* rx & tx IP frames 			*/

extern struct ip_frame received_ip_packet;
extern struct ip_frame send_ip_packet;

/* Bindings of OpenTCP network macros	*/

/** \def RECEIVE_NETWORK_B
 *	\brief Use this macro to read data from Ethernet controller
 *
 *	This macro should be used to read data from the Ethernet
 *	controller. Procedure for doing this would be as follows:
 *		\li Initialize reading of data from certain address in the
 *		Ethernet controller (usually you will do that based on buf_index
 *		value of ip_frame, udp_frame or tcp_frame type of variables; 
 *		in certain special situations you can also use buf_index from 
 *		ethernet_frame type of var.
 *		\li Keep invoking RECEIVE_NETWORK_B() to read one byte at a time from
 *		the ethernet controller. Take care not to read more data than
 *		actually received
 *		\li If needed, reinitialize reading of data again and start all
 *		over again
 *		\li When finished discard the current frame in the Ethernet
 *		controller by invoking NETWORK_RECEIVE_END() macro
 *
 */
#define RECEIVE_NETWORK_B()				eth_stub_read(&fec_nif)

 /** \def RECEIVE_NETWORK_W
 *	\brief Use this macro to read data from Ethernet controller
 *
 *	This macro should be used to read data from the Ethernet
 *	controller. Procedure for doing this would be as follows:
 *		\li Initialize reading of data from certain address in the
 *		Ethernet controller (usually you will do that based on buf_index
 *		value of ip_frame, udp_frame or tcp_frame type of variables; 
 *		in certain special situations you can also use buf_index from 
 *		ethernet_frame type of var.
 *		\li Keep invoking RECEIVE_NETWORK_W() to read one word at a time from
 *		the ethernet controller. Take care not to read more data than
 *		actually received
 *		\li If needed, reinitialize reading of data again and start all
 *		over again
 *		\li When finished discard the current frame in the Ethernet
 *		controller by invoking NETWORK_RECEIVE_END() macro
 *
 */
#define RECEIVE_NETWORK_W()				void()

/** \def RECEIVE_NETWORK_BUF
 *	\brief Use this macro to read data from Ethernet controller to a buffer
 *
 *	This macro should be used to read data from the Ethernet
 *	controller to a buffer in memory. Procedure for using this macro
 * is the same as for using RECEIVE_NETWORK_B() macro.
 *
 */
#define RECEIVE_NETWORK_BUF(c,d)		eth_stub_read_buf(&fec_nif, c, d)

/** \def SEND_NETWORK_B
 *	\brief Use this macro to write data to Ethernet controller
 *
 *	This macro should be used to write data to Ethernet
 *	controller. Procedure for doing this would be as follows:
 *		\li Initialize writing of data to certain address in the
 *		Ethernet controller. Buffer space in Ethernet controller is
 *		divided among the protocols in the following way:
 *			\li 256 byte Tx for ARP (see ARP_BUFFER )
 *			\li 1536 byte Tx for ICMP (see ICMP_BUF)
 *			\li 1536 byte Tx for TCP (see TCP_BUF)
 *			\li 1536 byte Tx for UDP (see UDP_BUF)
 *		\li Write the data by using SEND_NETWORK_B() macro
 *		\li When all of the data is written instruct the Ethernet controller
 *		to send the data by calling the NETWORK_COMPLETE_SEND() macro with
 *		number of bytes to send as a parameter
 *
 */
#define SEND_NETWORK_B(c)				eth_stub_write(&fec_nif, c)

/** \def SEND_NETWORK_W
 *	\brief Use this macro to write data to Ethernet controller
 *
 *	This macro should be used to write data to Ethernet
 *	controller. Procedure for doing this would be as follows:
 *		\li Initialize writing of data to certain address in the
 *		Ethernet controller. Buffer space in Ethernet controller is
 *		divided among the protocols in the following way:
 *			\li 256 byte Tx for ARP (see ARP_BUFFER )
 *			\li 1536 byte Tx for ICMP (see ICMP_BUF)
 *			\li 1536 byte Tx for TCP (see TCP_BUF)
 *			\li 1536 byte Tx for UDP (see UDP_BUF)
 *		\li Write the data by using SEND_NETWORK_W() macro
 *		\li When all of the data is written instruct the Ethernet controller
 *		to send the data by calling the NETWORK_COMPLETE_SEND() macro with
 *		number of bytes to send as a parameter
 *
 */
#define SEND_NETWORK_W(c) 				void(c)

/** \def SEND_NETWORK_BUF
 *	\brief Use this macro to write data from buffer to Ethernet controller
 *
 *	This macro should be used to write data from a buffer to Ethernet
 *	controller. Usage is the same as for the SEND_NETWORK_B() macro.
 */
#define SEND_NETWORK_BUF(c,d)			eth_stub_write_buf(&fec_nif, c, d)

/** \def NETWORK_CHECK_IF_RECEIVED
 *	\ingroup periodic_functions
 *	\brief Use this macro to check if there is recieved data in Ethernet controller
 *
 *	Invoke this macro periodically (see main_demo.c for example) to check
 *	if there is new data in the Ethernet controller.
 *
 *	If there is new data in the Ethernet controller, this macro (function
 *	that it points to that is) will return a value of TRUE and fill in
 *	the appropriate values in the received_frame variable. Otherwise it
 *	returns FALSE.
 */
#define NETWORK_CHECK_IF_RECEIVED()		eth_stub_isframe(&fec_nif)

/** \def NETWORK_RECEIVE_INITIALIZE
 *	\brief Initialize reading from a given address
 *
 *	This macro initializes reading of the received Ethernet frame from
 *	a given address in the Ethernet controller.
 */
#define NETWORK_RECEIVE_INITIALIZE(c)	eth_stub_seekread(&fec_nif, c)

/** \def NETWORK_RECEIVE_END
 *	\ingroup periodic_functions
 *	\brief Dump received packet in the Ethernet controller
 *
 *	Invoke this macro when the received Ethernet packet is not needed
 *	any more and can be discarded.
 */
#define NETWORK_RECEIVE_END()			eth_stub_dumpframe(&fec_nif)

/** \def NETWORK_COMPLETE_SEND
 *	\brief Send the Ethernet packet that was formed in the Ethernet controller
 *
 *	After the data has been written to the Ethernet controller, use this
 *	function to instruct the Ethernet controller that data is in it's 
 *	internal buffer and should be sent.
 */
#define	NETWORK_COMPLETE_SEND(c)		eth_stub_send(&fec_nif, c)

/** \def NETWORK_SEND_INITIALIZE
 *	\brief Initialize sending of Ethernet packet from a given address
 *
 *	Use this function to initialize sending (or creating) of an Ethernet
 *	packet from a given address in the Ethernet controller.
 */
#define NETWORK_SEND_INITIALIZE(c)		eth_stub_txalloc(&fec_nif)

/** \def NETWORK_ADD_DATALINK
 *	\brief	Add lower-level datalink information
 *
 *	This implementation adds Ethernet data-link information by
 *	invoking WriteEthernetHeader() function that writes Ethernet
 *	header based on information provided (destination and source ethernet
 *	address and protocol field).
 */
#define NETWORK_ADD_DATALINK(c)			eth_stub_writeheader(&fec_nif, c)


#endif		//guard


/* EOF	*/
