//*****************************************************************************
// Copyright (c) 2006, Freescale Semiconductor
// Freescale Confidential Proprietary
// For use on Freescale products only.
//
// File name :   	freescale_serial_flash.c
// Project name: 	emg_HTTP web server for Coldfire
// Author:			Eric Gregori   (847) 651 - 1971
//		   			eric.gregori@freescale.com
//
// Description : 	Provides a interface to a serial flash device.
//					
//
//*****************************************************************************
#include "m5223evb.h"
#include "common.h"
#include "menu.h"

#define QSPI_TX_RAM			0x00
#define QSPI_RX_RAM			0x10
#define QSPI_COMMAND_RAM	0x20
#define QSPI_RAM_END		0x30

// Field Description
// 15 CONT Continuous.
//			0 Chip selects return to inactive level defined by QWR[CSIV] when a single word transfer is complete.
//			1 Chip selects return to inactive level defined by QWR[CSIV] only after the transfer of the queue entries (max of 16
//				words).
//			Note: In order to keep the chip selects asserted for transfers beyond 16 words, the QWR[CSIV] bit must be set to
//			control the level that the chip selects return to after the first transfer.
//
// 14 BITSE Bits per transfer enable.
//			0 Eight bits
//			1 Number of bits set in QMR[BITS]
//
// 13 DT Delay after transfer enable.
//			0 Default reset value.
//			1 The QSPI provides a variable delay at the end of serial transfer to facilitate interfacing with peripherals that have
//			a latency requirement. The delay between transfers is determined by QDLYR[DTL].
//
// 12 DSCK Chip select to QSPI_CLK delay enable.
//			0 Chip select valid to QSPI_CLK transition is one-half QSPI_CLK period.
//			1 QDLYR[QCD] specifies the delay from QSPI_CS valid to QSPI_CLK.
// 11-8 QSPI_CS
//			Peripheral chip selects. Used to select an external device for serial data transfer. More than one chip select may be
//			active at once, and more than one device can be connected to each chip select. Bits 11-8 map directly to the
//			corresponding QSPI_CSn pins. If more than four chip selects are needed, then an external demultiplexor can be
//			used with the QSPI_CSn pins.
//			Note: Not all chip selects may be available on all device packages. See Chapter 2, Signal Descriptions,for details
//			on which chip selects are pinned-out.
// 7-0 Reserved, should be cleared.
#define SPI_COM_CONT	0x8000
#define SPI_COM_BITSE	0x4000
#define SPI_COM_DT		0x2000
#define SPI_COM_DSK		0x1000
#define SPI_COM_CS0		0x0800
#define SPI_COM_CS1		0x0400
#define SPI_COM_CS2		0x0200
#define SPI_COM_CS3		0x0100

#define SPI_COMMAND_CS0			SPI_COM_CONT



//*****************************************************************************
// High Level Initialization routine
// 
// Data rate is set to 8Mhz
// CPOL = CPHA = 0
// 8 bits
// 
//
// Author: Eric Gregori  (847) 651 - 1971
//		   eric.gregori@freescale.com
//*****************************************************************************
void init_serial_flash( void )
{
	// No delay, disable QSPI
	MCF_QSPI_QDLYR = 0;
	
	// QSPI interrupt disabled
	MCF_QSPI_QIR   = 0;
	

	// QMR[BAUD] = fsys/ / (2  [desired QSPI_CLK baud rate])
	// QMR[BAUD] = 60000000/(2*8000000)
	// Using 3 will yield a baud rate of 10Mhz
	MCF_QSPI_QMR = (0|	
					MCF_QSPI_QMR_MSTR|
					MCF_QSPI_QMR_BITS(8)|
					MCF_QSPI_QMR_BAUD(3)	
				   );

    // Pin assignments for port QS
    //     Pin QS6 : GPIO input
    //     Pin QS5 : GPIO input
    //     Pin QS4 : GPIO input
    //     Pin QS3 : QSPI chip select QSPI_CS0
    //     Pin QS2 : QSPI serial clock, QSPI_CLK
    //     Pin QS1 : QSPI serial data input, QSPI_DIN
    //     Pin QS0 : QSPI serial data output, QSPI_DOUT

    //     DDRQS[DDRQS6]   = 0 
    //     DDRQS[DDRQS5]   = 0 
    //     DDRQS[DDRQS4]   = 0 
    //     DDRQS[DDRQS3]   = 0 
    //     DDRQS[DDRQS2]   = 0 
    //     DDRQS[DDRQS1]   = 0 
    //     DDRQS[DDRQS0]   = 0 
    MCF_GPIO_DDRQS = 0;

    //     PQSPAR[PQSPAR6] = 0 
    //     PQSPAR[PQSPAR5] = 0 
    //     PQSPAR[PQSPAR4] = 0 
    //     PQSPAR[PQSPAR3] = %01
    //     PQSPAR[PQSPAR2] = %01
    //     PQSPAR[PQSPAR1] = %01
    //     PQSPAR[PQSPAR0] = %01
    MCF_GPIO_PQSPAR = MCF_GPIO_PQSPAR_PQSPAR3(0x1) |
                      MCF_GPIO_PQSPAR_PQSPAR2(0x1) |
                      MCF_GPIO_PQSPAR_PQSPAR1(0x1) |
                      MCF_GPIO_PQSPAR_PQSPAR0(0x1);
				   	
}


void write_to_qspi_ram( uint8 address, uint16 data )
{
	MCF_QSPI_QAR = address;
	MCF_QSPI_QDR = data;		
}


uint16 read_from_qspi_ram( uint8 address )
{
	MCF_QSPI_QAR = address;
	return( MCF_QSPI_QDR );	
}


void start_spi_xfer( uint8 bytes, uint8 csiv )
{
	MCF_QSPI_QIR = MCF_QSPI_QIR_SPIF;
	
	if( csiv == 1 )
		MCF_QSPI_QWR = MCF_QSPI_QWR_ENDQP(bytes-1)|MCF_QSPI_QWR_CSIV; 
	else
		MCF_QSPI_QWR = MCF_QSPI_QWR_ENDQP(bytes-1); 
	
	MCF_QSPI_QDLYR = MCF_QSPI_QDLYR_SPE; 						// Start Xfer

	// Trick to set CS high after transfer
	if( csiv == 2 )
			MCF_QSPI_QWR = MCF_QSPI_QWR_ENDQP(bytes-1)|MCF_QSPI_QWR_CSIV; 

	while( !(MCF_QSPI_QIR & MCF_QSPI_QIR_SPIF ))
	{
		// Spin here waiting for completion					
	};
}


void read_write_header( uint32 address, uint8 command )
{
	// bring CS low
	// byte 1 = command
	// byte 2 = (address & 0x00FF0000)
	// byte 3 = (address & 0x0000FF00)
	// byte 4 = (address & 0x000000FF)
	// There may be additional bytes depending on bytestoread
	write_to_qspi_ram( QSPI_TX_RAM, command );
	write_to_qspi_ram( QSPI_TX_RAM+1, (uint8)((address&0x00FF0000)>>16 ));
	write_to_qspi_ram( QSPI_TX_RAM+2, (uint8)((address&0x0000FF00)>>8 ));
	write_to_qspi_ram( QSPI_TX_RAM+3, (uint8)((address&0x000000FF) ));				

	start_spi_xfer( 4, 0 );
}


//*****************************************************************************
// Read Data from serial Flash.
// Send the flash a read command, and a start address
// Bring the CS line for the selected device low.
//
// device  		= device to read - for future expansion
// address 		= if <0xFFFFFFFF, address of first byte to read
//			 	  if 0xFFFFFFFF, read next byte
// buffer  		= pointer to buffer to copy data
// bytestoread 	= # of bytes to read, can be 0
// done_flag	= if >0, unselect CS after data is copied  
//
// Returns >= 0 on success, and <0 on failure
//
// Author: Eric Gregori  (847) 651 - 1971
//		   eric.gregori@freescale.com
//*****************************************************************************
uint32 read_serial_flash( uint32 device, uint32 address, uint8 *buffer, uint32 bytestoread, uint32 done_flag )
{
	uint8	tx_ram_index, rx_ram_index, btr, i;
	uint32	buffer_index;


	tx_ram_index 		= 0;
	rx_ram_index		= 0;
	buffer_index		= 0;
	switch( device )
	{
		case (uint32)'m25p':
		
			// Init Command RAM
			// We only need to do this once, since it does not change
			for( i=QSPI_COMMAND_RAM; i<QSPI_RAM_END; i++ ) 				
				write_to_qspi_ram( i, SPI_COMMAND_CS0 );

			if( address != 0xFFFFFFFF )
			{
				// We need to send a read command byte
				// Instruction Description     code          bytes dummy data
				// READ        Read Data Bytes 0000 0011 03h 3     0     1 to inf
				// The device is first selected by driving Chip Select (S) Low. 
				// The instruction code for the Read Data Bytes (READ) instruction 
				// is followed by a 3-byte address (A23-A0), each bit being
				// latched-in during the rising edge of Serial Clock (C). Then the
				// memory contents, at that address, is shifted out on Serial Data 
				// Output (Q), each bit being shifted out, at a maximum
				// frequency fR, during the falling edge of Serial Clock (C).
				read_write_header( address, 0x03 );
			}
			
			for( btr=bytestoread; btr; )
			{
				// Do read in 16 byte chunks
				i 	= 	btr&0x0F;
				btr -= 	i;

				start_spi_xfer( i, 0 );
				
				// Copy data from SPI RAM to buffer				
				// j = # of bytes read
				MCF_QSPI_QAR = QSPI_RX_RAM;				
				for( ; i; i-- )
					buffer[buffer_index++] = MCF_QSPI_QDR;	
			} // End of For()

			// Do dummy read, to get CS high			
			if( done_flag )
				start_spi_xfer( 1, 2 );

			break;
			
		default:
			return( (uint32)'bdev' );	
		
	} // End of Switch( device )
	
	return( 0 );
}


//*****************************************************************************
// Write To Serial Flash.
// Write to flash in 256 byte chunks.
//
// device  		= device to read - for future expansion
// address 		= Must be on a 256 byte boundary.
// buffer  		= pointer to buffer containing data
//
// Returns >= 0 on success, and <0 on failure
//
// Author: Eric Gregori  (847) 651 - 1971
//		   eric.gregori@freescale.com
//*****************************************************************************
int write_serial_flash( uint32 device, uint32 address, uint8 *buffer  )
{
	uint8	tx_ram_index, rx_ram_index, btr, i, j;
	uint16	status;
	uint32	buffer_index;


	tx_ram_index 		= 0;
	buffer_index		= 0;
	switch( device )
	{
		case (uint32)'m25p':
			if( !(address & 0x000000FF  ))
			{
				// Init Command RAM
				// We only need to do this once, since it does not change
				for( i=QSPI_COMMAND_RAM; i<QSPI_RAM_END; i++ ) 				
					write_to_qspi_ram( i, SPI_COMMAND_CS0 );
						
				// Send Write Enable
				// The Write Enable (WREN) instruction is entered by driving Chip Select (S) Low, sending the
				// instruction code, and then driving Chip Select (S) High.				
				write_to_qspi_ram( QSPI_TX_RAM, 0x06 );
				start_spi_xfer( 1, 1 );

				// Send Page Write command
				// The Page Program (PP) instruction is entered by driving Chip Select (S) Low, followed by
				// the instruction code, three address bytes and at least one data byte on Serial Data Input (D).
				// If the 8 least significant address bits (A7-A0) are not all zero, all transmitted data that goes
				// beyond the end of the current page are programmed from the start address of the same
				// page (from the address whose 8 least significant bits (A7-A0) are all zero). Chip Select (S)
				// must be driven Low for the entire duration of the sequence.				
				read_write_header( address, 0x02 );
				
				// Send Data in 16 bytes Chunks
				for( j=0; j<0x10; j++ )
				{
					tx_ram_index = 0;
					// Write Data to TX_RAM
					for( i=0; i<0x10; i++ )
						write_to_qspi_ram( (QSPI_TX_RAM+tx_ram_index++), buffer[buffer_index+i] );	

					start_spi_xfer( 0x0f, 0 );
					buffer_index += 0x10;
				}
				
				// We have loaded the part with 256 bytes of data
				// The flash started its programming sequence when the 
				
				// Send the read status register command
				write_to_qspi_ram( QSPI_TX_RAM, 0x05 );
				write_to_qspi_ram( QSPI_COMMAND_RAM, SPI_COMMAND_CS0 );
				write_to_qspi_ram( QSPI_COMMAND_RAM+1, SPI_COMMAND_CS0 );
				while( 1 )
				{
					start_spi_xfer( 2, 1 );
					status = read_from_qspi_ram(QSPI_RX_RAM+1);
//					printf( "\n%x", status );
					if( !(status & 0x0001 ) )
						break;		
				}
			}
			break;	
							
		default:
			return((uint32)'bdev' );
			break;	
		
	} // End of Switch( device )
	
	return( 0 );
}


//*****************************************************************************
// Bulk Erase Flash.
//
// device  		= device to erase - for future expansion
//
// Returns >= 0 on success, and <0 on failure
//
// Author: Eric Gregori  (847) 651 - 1971
//		   eric.gregori@freescale.com
//*****************************************************************************
int erase_serial_flash( uint32 device )
{
	uint8	rx_ram_index, btr, i, j;
	uint16	status;
	uint32	buffer_index;


	buffer_index		= 0;
	switch( device )
	{
		case (uint32)'m25p':
			// Send Write Enable
			// Send Bulk Erase 
			// Spin waiting while reading status 
			// Init Command RAM
			// We only need to do this once, since it does not change
			for( i=QSPI_COMMAND_RAM; i<QSPI_RAM_END; i++ ) 				
				write_to_qspi_ram( i, SPI_COMMAND_CS0 );
						
			// Send Write Enable
			// The Write Enable (WREN) instruction is entered by driving Chip Select (S) Low, sending the
			// instruction code, and then driving Chip Select (S) High.				
			write_to_qspi_ram( QSPI_TX_RAM, 0x06 );
			start_spi_xfer( 1, 1 );

			// Send Bulk Erase Command
			write_to_qspi_ram( QSPI_TX_RAM, 0xC7 );
			start_spi_xfer( 1, 1 );
				
			// Send the read status register command
			write_to_qspi_ram( QSPI_TX_RAM, 0x05 );
			write_to_qspi_ram( QSPI_COMMAND_RAM, SPI_COMMAND_CS0 );
			write_to_qspi_ram( QSPI_COMMAND_RAM+1, SPI_COMMAND_CS0 );
			while( 1 )
			{
				start_spi_xfer( 2, 1 );
				status = read_from_qspi_ram(QSPI_RX_RAM+1);
//				printf( "\n%x", status );
				if( !(status & 0x0001 ) )
					break;		
			}
			break;	
				
							
		default:
			return((uint32)'bdev' );
			break;	
		
	} // End of Switch( device )
	
	return( 0 );
}


int	sffs_erase( void *pio )
{
	return( erase_serial_flash( (uint32)'m25p' ) );	
}

int	sffs_read( void *pio )
{
	uint8 data[4], i;

	(void)read_serial_flash( (uint32)'m25p', 0, data, 4, 0 );
	printf( "\n%x%x%x%x", data[0], data[1], data[2], data[3] );
	for( i=0; i<16; i++ )
	{
		(void)read_serial_flash( (uint32)'m25p', 0xFFFFFFFF, data, 4, 0 );
		printf( "\n%x%x%x%x", data[0], data[1], data[2], data[3] );
	}

	(void)read_serial_flash( (uint32)'m25p', 0xFFFFFFFF, data, 4, 1 );
	printf( "\n%x%x%x%x", data[0], data[1], data[2], data[3] );
	
	return(0);
}

int	sffs_write( void *pio )
{
	uint8 i, data[256];
	
	for( i=0; i<0xFF; i++ )
		data[i] = i;

	(void)write_serial_flash( (uint32)'m25p', 0, data );
	
	return(0);

//	return( erase_serial_flash( (uint32)'m25p' ) );	
}