//*****************************************************************************
// Copyright (c) 2006, Freescale Semiconductor
// Freescale Confidential Proprietary
// For use on Freescale products only.
//
// File name :   	emg_http_server.c
// Project name: 	emg_HTTP web server for Coldfire
// Author:			Eric Gregori   (847) 651 - 1971
//		   			eric.gregori@freescale.com
//
// Description : 	HTTP/Web server with static anf dynamic FFS support.
//					Dynamic HTML is supported via tokens in the HTML.
//					A key is used to support run time ( dynamic ) web page uploads.
//					Without the correct key, the download is rejected.
//					The web server supports multiple sessions as defined by EMG_HTTP_SESSION.
//					The web server supports 'GET', 'POST','EMG'.
//					EMG is a unique non-standard HTTP command to request a upload.
//					Uploading is done using the same server that serves up the web pages.
//					Long filenames are supported, along with subdirectories.
//
//*****************************************************************************
#include "ipport.h"
#include "tcpapp.h"
#include "msring.h"
#include "menu.h"
#include OSPORT_H
#include "freescale_http_server.h"





extern	void freescale_http_process( int i );
extern	int  freescale_http_connection( M_SOCK so );
extern	void freescale_http_delete( void );
extern	void freescale_http_remove( M_SOCK so );
extern 	struct menu_op emg_ffs_dir_menu[];
extern void Flash_Init(unsigned long FlashClk);




//*****************************************************************************
// Global session structure array
//*****************************************************************************
EMG_HTTP_SESSION freescale_http_sessions[MAX_NUMBER_OF_SESSIONS];


//*****************************************************************************
// Global vars
//*****************************************************************************
int 				semaphore;
int					flash_ffs_lockout;


//*****************************************************************************
// Declare Task Object
//*****************************************************************************
TK_OBJECT(to_emghttpsrv);
TK_ENTRY(tk_emghttpsrv);
struct inet_taskinfo emg_http_task = {
      									&to_emghttpsrv,
      									"EMG HTTP server",
      									tk_emghttpsrv,
      									NET_PRIORITY,
      									APP_STACK_SIZE
									};
long emghttpsrv_wakes = 0;


//*****************************************************************************
// Declare a Socket structure and communications queue "msring"
//*****************************************************************************
struct sockaddr_in   		emg_http_sin;
M_SOCK	 					emg_http_server_socket = INVALID_SOCKET;
static struct msring 		emg_http_msring;
static M_SOCK 				emg_http_msring_buf[MAX_NUMBER_OF_SESSIONS + 10];


//*****************************************************************************
// emg_http_loop() - Written By Eric Gregori
//		   			 eric.gregori@freescale.com
//
// Although the server is a seperate task.
// Each session is run under the context of the server task.
// If we had more RAM, we could spawn a new process for each session.
//
// For now, run each session under the context of the server task.
// Poll through the sessions, executing the state machine for each one.
//*****************************************************************************
int freescale_http_loop()
{
	int			i;
	
	
	// Loop through connected sessions
	for( i=0; i<MAX_NUMBER_OF_SESSIONS; i++ )
	{
		if( freescale_http_sessions[i].valid == VALID_EMG_SESSION )
		{
			freescale_http_process( i );
			tk_sleep( 10 );
		}
	}
	
   	return SUCCESS;
}


//*****************************************************************************
// emg_http_cmdcb() - Written by Eric Gregori
//		   			  eric.gregori@freescale.com
//
// This is the mini-sockets callback function.
// We only uses to detect a connection to the socket.
// When a connection is made, this function is called by the stack.
// The connection message is sent to the application through a 
// msring queue.
//*****************************************************************************
int freescale_http_cmdcb(int code, M_SOCK so, void * data)
{
	int e = 0;

	switch(code)
	{
		// socket open complete
		case M_OPENOK:
			msring_add(&emg_http_msring, so);
			break;
      
		// socket has closed      
   		case M_CLOSED:  
   			while( semaphore ){};
   			semaphore = 1;
			freescale_http_remove( so );  		// EMG - 3/31/06   		        
			semaphore = 0;
     		break;								// let stale conn timer catch these 
      
		// passing received data
		// blocked transmit now ready
		case M_RXDATA:          				// received data packet, let recv() handle it 
		case M_TXDATA:          				// ready to send more, loop will do it
			e = -1;        						// return nonzero code to indicate we don't want it 
			break;
      
   		default:
      		dtrap();             				// not a legal case
      		return 0;
   }

   TK_WAKE(&to_emghttpsrv);    					// wake server task

   USE_VOID(data);
   return e;
}


//*****************************************************************************
// emg_http_init() - written by Eric Gregori
//		   			 eric.gregori@freescale.com
//
// Create and bind a socket to our listening port ( PORT_NUMBER ).
// Set the socket to listen and non-blocking.
//*****************************************************************************
int freescale_http_init()
{
	int e;
	
	
	semaphore 				= 0;
	flash_ffs_lockout 		= 0;
	
	// Init flash state machine clock
	flash_init_old( );
	Flash_Init(SYSTEM_CLOCK*1000/2);
	//jpw needs changed to Flash_Init
	
	// Init session array
	freescale_http_delete( );
	
	// Install Menu item 'DIR' for EMG FFS
	if( install_menu( emg_ffs_dir_menu ) )
		printf( "\nCould not install DIR menu item for EMG FFS" );
	
	// Init message queue for MINI_TCP socket interface
    msring_init(&emg_http_msring, emg_http_msring_buf, sizeof(emg_http_msring_buf) / sizeof(emg_http_msring_buf[0]));

	// Init a socket structure with our Port Number
	emg_http_sin.sin_addr.s_addr 	= (INADDR_ANY);
	emg_http_sin.sin_port        	= (PORT_NUMBER);

    emg_http_server_socket 			= m_listen(&emg_http_sin, freescale_http_cmdcb, &e);
    if (emg_http_server_socket == INVALID_SOCKET)
    {
    	dprintf("error %d starting listen on emg HTTP server\n", e);
    }

   	return SUCCESS ;
}


//*****************************************************************************
// emg_http_cleanup() - Written by Eric Gregori
//		   			    eric.gregori@freescale.com
//
// Delete all sessions.
//*****************************************************************************
void freescale_http_cleanup()
{
	freescale_http_delete( );
}


//*****************************************************************************
// emg_http_check() - Written by Eric Gregori
//		   			  eric.gregori@freescale.com
//
// Check msring for message from socket callback function.
// If we received a connect request, call the connection function.
// While we are waiting for a connection to complete, we need to
// continue running our loop.
// Make the socket non-blocking.
//
// Call the loop function to execute any pending sessions.
//*****************************************************************************
void freescale_http_check(void)
{
	M_SOCK 		so;


	if ( emg_http_server_socket == INVALID_SOCKET )
    	return ;

   	while(msring_del(&emg_http_msring, &so) == 0)
   	{
   	   	while( semaphore ){};
   		semaphore = 1;
      	while(freescale_http_connection(so) == 0)
      	{
      		semaphore = 0;
     		freescale_http_loop();
      	}
      	
      	semaphore = 0;
#if(MAX_NUMBER_OF_SESSIONS>0)      
      	m_ioctl(so, SO_NONBLOCK, NULL);   /* make socket non-blocking */
#endif
   	}

	freescale_http_loop();
}


//*****************************************************************************
// The application thread works on a "controlled polling" basis: 
// it wakes up periodically and polls for work.
//
// The task could aternativly be set up to use blocking sockets,
// in which case the loops below would only call the "xxx_check()"
// routines - suspending would be handled by the TCP code.
//
//
// FUNCTION: tk_emg_http_srv
// 
// PARAM1: n/a
//
// RETURNS: n/a
//
//*****************************************************************************
TK_ENTRY(tk_emghttpsrv)
{
   int err;

   while (!iniche_net_ready)
      TK_SLEEP(1);

   err = freescale_http_init();
   if( err == SUCCESS )
   {
      exit_hook(freescale_http_cleanup);
   }
   else
   {
      dtrap();    						// emghttp_init()  shouldn't ever fail
   }

   for (;;)
   {
      freescale_http_check();         	// will block on select
      tk_yield();          				// give up CPU in case it didn't block

		emghttpsrv_wakes++;				//

      if (net_system_exit)
         break;
   }
   TK_RETURN_OK();
}

