
#include "ipport.h"        /* from Interniche directory */
#include "osport.h"
#include "msock.h"		// For M_SOCK & M_CALLBACK macros

/**** MACROS ****/
#define IP_ANY				0		// Wildcard IP address
#define PORT_CONTROL		23		// Use port 23 as our control port for now
#define BRIDGE_STACK_SIZE	2048
#define MAX_RECV_SIZE		128		// Max Receive size

/**** Function Prototypes ****/
int ControlCallback(int code, M_SOCK so, void * data);
TK_ENTRY(BridgeSetup);
void PrintSocket( M_SOCK so );

/**** Variable Declarations ****/
char* mustang_help = \
"Possible Commands Are:\r\n \
help - Display this menu\r\n \
baud <uart> <baudrate> - Set the baud rate of <uart> to <baudrate>\r\n \
connect <uart> <port> - Connect a uart to an ethernet port\r\n";

M_SOCK				ctrl_listen = INVALID_SOCKET;
M_SOCK				ctrl_sock = INVALID_SOCKET;
struct sockaddr_in	ctrl_addr;
char	new_data = 0;

TK_OBJECT(obj_SEBridge);
struct inet_taskinfo SEBridge_task = {
      									&obj_SEBridge,
      									"FLE Serial/Ethernet Bridge",
      									BridgeSetup,
      									NET_PRIORITY,
      									BRIDGE_STACK_SIZE
									};


/*******************************************************************************
 * This is the function which is called when the tasking system starts to run.
 * It contains the main loop for this section of code.
 ******************************************************************************/
TK_ENTRY(BridgeSetup)
{
	int err;		// Return value
	int len=0;		// Data length
	char data_buf[MAX_RECV_SIZE];				// Received data

	while(!uart_flush(0)){};	// Clear up anything in the buffer before we start

	// Init a socket structure with our Port Number
	ctrl_addr.sin_addr.s_addr 	= IP_ANY;
	ctrl_addr.sin_port        	= PORT_CONTROL;
	ctrl_listen = m_listen( &ctrl_addr, ControlCallback, &err);
	if( INVALID_SOCKET == ctrl_listen )
	{
		printf("Error setting up listening socket for control port\r\n");
	}

	while( 1 )
	{
		if( new_data )	// The callback says there is data ready.
		{
			new_data = 0;
			
			if( INVALID_SOCKET == ctrl_sock )
				len = 0;
			else
				len = m_recv(ctrl_sock, data_buf, MAX_RECV_SIZE );
			
			if( 0 >= len )	// There is not any data, report error
			{
				if( EWOULDBLOCK == ctrl_sock->error )
				{
					printf("EWOULDBLOCK - no data available\r\n");
				}
				else if( ESHUTDOWN == ctrl_sock->error )
				{
					printf("ESHUTDOWN - socket is closing\r\n");
				}
				else
				{
					printf("No data.  Error = %d\r\n", ctrl_sock->error);
				}
			}
			else	// There is data, echo it
			{
				printf("Got %d bytes = %s\r\n", len, data_buf );
			}
		}
		
		while(!uart_flush(0)){};	// Make sure debug statments print
		tk_yield();
	}
}


/*******************************************************************************
 * This socket callback is called whenever a connection is recieved on the 
 * listening socket that was setup above.
 ******************************************************************************/
int ControlCallback(int code, M_SOCK so, void * data)
{
	unsigned int len;
	int ret;
	int e = 0;

//	ret = m_ioctl( so, SO_NONBLOCK, 0 );
	
	switch( code )
	{
		case M_OPENOK : 
			printf("Control Callback: Socket open complete\n");
			ctrl_sock = so;
			break;
		case M_OPENERR :
			printf("Control Callback: Active open failed\n");
			break;
		case M_CLOSED :
			printf("Control Callback: Socket has closed\n");
			ctrl_sock = INVALID_SOCKET;
			break;
		case M_RXDATA :
			new_data = 1;
			len = so->rcvdq.p_head->m_len;
			printf("Callback sees %d bytes\r\n", len );
			e = -1;        						// return nonzero code to indicate we don't want it 
			break;
		case M_TXDATA :
			break;
		default :
			printf("Control Callback: Something has gone horribly awry\n");
			break;
	}
	while(!uart_flush(0)){};
   return e;
}


/*******************************************************************************
 * This is a debug function used to print out some info about a socket.
 ******************************************************************************/
void PrintSocket( M_SOCK so )
{
	printf("Socket at 0x%0X\n", &so);
	printf("\tLocal IP:port:   0x%08X:%d\n", so->lhost, so->lport);
	printf("\tRemote IP:port:  0x%08X:%d\n", so->fhost, so->fport);
	printf("\tLast Error:      %d\n", so->error);
	printf("\tState Bitmast:   0x%08X\n", so->state);
	printf("\tOptions Bitmast: 0x%08X\n", so->so_options);
	printf("\tLinger Time:     %d\n", so->linger);
	printf("\tNumber of Receive bytes: %d\n", so->rcvdq.sb_cc);
	printf("\tNumber of Bytes to send: %d\n", so->sendq.sb_cc);
	while(!uart_flush(0)){};
}

