/*   tftputil.c
 *
 * Copyright 1997- 2006 By InterNiche Technologies Inc. All rights reserved
 *
 * Portions Copyright 1986 by Carnegie Mellon
 * Portions Copyright 1984 by the Massachusetts Institute of Technology
 *
 * Copyright (c) 1982, 1986, 1988 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation and other 
 * materials related to such distribution and use acknowledge that 
 * the software was developed by the University of California, Berkeley.
 * The name of the University may not be used to endorse or promote 
 * products derived from this software without specific prior written 
 * permission. THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 *
 * Rights, responsibilities and use of this software are controlled by
 * the agreement found in the "LICENSE.H" file distributed with this
 * source code.  "LICENSE.H" may not be removed from this distribution,
 * modified, enhanced nor references to it omitted.
 *
 * Utility routines common to TFTP client and server.
 */

#include "license.h"
#include "tftpport.h"

#include "tftp.h"

/* internal routine */
static void check_state(struct tfconn * cn);

struct tfconn * tftp_conns = NULL;   /* list of active connections */

int ntftps = 0;   /* number of connections */

/* Setup a TFTP connection block. */

struct tfconn * 
tfmkcn(void)
{
struct tfconn * cn;

   cn = (struct tfconn *)TFC_ALLOC(sizeof(struct tfconn));
   if(!cn)
      return NULL;

   ntftps++;
   cn->next = tftp_conns;   /* add this tfconn to list */
   tftp_conns = cn;

   cn->tf_srtt = cn->tf_rt = TPS/2;   /* init round trip timer with large-ish guess */
   if (cn->tf_rt > TF_MAXTMO)
      cn->tf_rt = TF_MAXTMO;
   if (cn->tf_rt < TF_MINTMO)
      cn->tf_rt = TF_MINTMO;
   cn->tf_NR_last = 1;   /* fake last packet Number retrys */

   return cn;
}


/* Cleanup routine called when done */

void
tfcleanup(struct tfconn *cn)
{
struct tfconn * tmp, * last;

   /* unlink cn from list */
   tmp = tftp_conns;
   last = NULL;
   while(tmp)   /* traverse list, looking of cn to free */
   {
      if(tmp == cn)
      {
         if(last)   /* unlink from list */
            last->next = tmp->next;
         else   /* was first in list, fix list pointer */
            tftp_conns = tmp->next;
         break;
      }
      last = tmp;
      tmp = tmp->next;
   }
   if(!tmp)   /* connection not in master list? */
   {
      dtrap();   /* serious prog error..... */
   }

   if(cn->tf_fd != NULL)
      vfclose(cn->tf_fd);
   if(cn->tf_outbuf.data)
      tftp_udpfree(cn);
   if(cn->tf_conn != 0)
      tftp_udpclose(cn->tf_conn);

   TFC_FREE(cn);
   ntftps--;
}


void 
tftptmo(struct tfconn * cn)
{
TFTPBUF p;

   cn->tf_tmo++;
   /* if we have more tries... */
   if((cn->tf_tries == 0) || (--cn->tf_tries))
   {
   int e;

      cn->tf_rsnd++;
      cn->tf_NR++;
      /* do the actual retry. Don't use tf_write() since it will re-init 
         the round trip & retry count values. */
      p = &cn->tf_outbuf;
      e = tftp_udpsend(cn, p->data, cn->tf_lastlen);
      if(e < 0)
      {
         if (cn->callback)
         {
            if (e == ENP_NOBUFFER)
               cn->callback(TFC_BUFFER, cn, "UDP alloc failed");
            else
               cn->callback(TFC_UDPSEND, cn, "UDP send failed");
         }
         tfkill(cn);    /* UDP send error, kill tftp session */
      }
      else  /* udp retry packet sent OK */
      {
         cn->tf_rt <<= 1;      /* double round trip est. */
         if (cn->tf_rt > TF_MAXTMO)
            cn->tf_rt = TF_MAXTMO;
         cn->tf_tick = cticks + cn->tf_rt;   /* set time to do next retry */
      }
   }
   else 
   {
      cn->tf_state = TIMEOUT;
   }
}

/* tf_good() - called to do timing calcs after good receives */

void
tf_good(struct tfconn * cn)
{
unsigned long trt;

   trt = cticks - cn->tf_sent;  /*  Measured round trip time  */
   /* set smoothed round trip time based on measured */
   if((cn->tf_srtt > trt) && (cn->tf_srtt > 1))
      cn->tf_srtt--;
   else if((cn->tf_srtt) < trt)
      cn->tf_srtt++;
   /* set the retry time: experimental algorithm: */
   trt = trt + cn->tf_srtt + 1;
   if (trt < TF_MINTMO)
      cn->tf_rt = TF_MINTMO;
   else if (trt > TF_MAXTMO)
      cn->tf_rt = TF_MAXTMO;
   else
      cn->tf_rt = trt;

   cn->tf_NR_last = cn->tf_NR;
   cn->tf_NR = 0;
}



/* tfdodata() - Process a data packet received for TFTP connection cn.
Handle out of sequence blocks and check on block length. If a block
is way to short (len < tftp header size) send back an error message
and abort the transfer. If the block is less than NORMLEN bytes long, shut 
down the transfer; we're done. Otherwise, just write it to disk (if necessary).

Returns TRUE if OK, FALSE if error.
*/

int /* BOOL */
tfdodata(struct tfconn * cn,
   TFTPBUF p,
   unsigned len)
{
char * data;
struct tfdata *ptfdat;

   if(len < 4) 
   {
      tfsnderr(cn, ERRTXT, "Bad len (too short)");
      if (cn->callback)
         cn->callback(TFC_BADLEN, cn, "short data from peer");
      tfkill(cn);
      return FALSE;
   }

   if(cn->tf_state != DATAWAIT)
   {
      tfsnderr(cn, ERRTXT, "Rcvd unexpected data block");
      return FALSE;
   }

   /* point to tftp header at front of */
   ptfdat = (struct tfdata *) p->data;
   len -= 4;   /* subtract length of header from data */

   if((ptfdat->tf_block) != cn->tf_expected)
   {

   /*   We got a retransmission of a packet we have already tried to
   ACK.  If we retransmit the ACK, and the old ACK finally gets through also,
   our correspondent will resend the next data block, and we will do the same
   thing for it, on through to the end of the file.  So we shouldn't retransmit
   the ACK until we are convinced that the first one was actually lost.  We will
   become convinced if our own timeout waiting for the next data packet
   expires.  */

      cn->tf_ous++;   
      return FALSE;
   }
   tf_good(cn);   /* adjust timer values */

   cn->tf_size += len;
   cn->tf_flen = len;

   /* write UDP data into file */
   data = ptfdat->tf_data;
   if((int)vfwrite(data, 1, len, cn->tf_fd) != (int)len)
   {
      tf_full(cn);
      return FALSE;
   }
   
   /* Send the ack & move to next state */
   tfsndack(cn);
   if(len == NORMLEN) 
      cn->tf_state = DATAWAIT;
   else    /* sub-normal len (less than NORMLEN) indicates end of file */
      cn->tf_state = RCVLASTDATA;

   cn->tf_expected++;
   return TRUE;
}

/*  Handle disk full condition by sending error packet and killing
    this connection.  */
void
tf_full(struct tfconn *cn)
{
   tfsnderr(cn, DISKFULL, " ");
   tfkill(cn);   
   if(cn->callback)
       cn->callback(TFC_DISKFULL, cn, tferrors[DISKFULL] );
}

/* Send an error packet. If the code is nonzero, put it in the packet.
   Otherwise, copy the text into the packet. */

char * tferrors[] = 
{
   "See text",
   "File not found",
   "Access violation",
   "Disk full",
   "Illegal TFTP operation",
   "Unknown transfer ID",
   "File already exists",
   "No such user"
};

void
tfsnderr(struct tfconn * cn,
   unsigned code,
   char *text)
{
unsigned len;
struct tferr *perr;

   len = sizeof(struct tferr)-2;

   tftp_udpbuffer(cn, sizeof(struct tfack) );
   if(cn->tf_outbuf.data == NULL)
      return;

   perr = (struct tferr *)(cn->tf_outbuf.data);
   perr->tf_op = (TF_ERROR);
   perr->tf_code = (code);

   if(code == ERRTXT)
   {
      strcpy(perr->tf_err, text);
      len += strlen(text)+1;
   }
   else
   {
      strcpy(perr->tf_err, tferrors[code]);
      len += strlen(tferrors[code]) + 1;
   }
   tftp_udpsend(cn, perr, len);
   tftp_udpfree(cn);    /* free now, there are no retrys on these */
}

/* Process an incoming error packet */

void
tfdoerr(struct tfconn *cn, TFTPBUF p, unsigned len)
{
struct tferr * perr;
int tferror;   /* tftp protocol error */
char * errtxt;   /* text of/for error */

   /* if connection is dead, don't worry about this... */
   if(cn->tf_state == DEAD || cn->tf_state == TERMINATED)
      return;

   tfkill(cn);

   perr = (struct tferr *) p->data;   /* overlay tftp error packet struct */
   tferror = (perr->tf_code);      /* get tftp error code (see rfc) */
   if(tferror > 0 && tferror < 8)      /* check for known range of errors */
      errtxt = tferrors[tferror];      /* get text from array if known error */
   else   /* not know error, try to extract text from tftp packet */
   {
      errtxt = perr->tf_err;
      if(*errtxt <= ' ' || *errtxt & 0x80 || (strlen(errtxt) > 100))   /* make sure it's printable */
         errtxt = "bogus tftp error text";
   }
   if(cn->callback)
       cn->callback(TFC_FOREIGN, cn, errtxt);
   USE_ARG(len);
}

/* tfsndack() - build and send ack for latest block */

int
tfsndack(struct tfconn *cn)
{
struct tfack * pack;
int e;

   tftp_udpbuffer(cn, sizeof(struct tfack) );
   if(cn->tf_outbuf.data == NULL)
      return ENP_NOBUFFER;
   pack = (struct tfack *)(cn->tf_outbuf.data);

   cn->tf_lastlen = sizeof(struct tfack);
   pack->tf_op = TF_ACK;   /* in local endian, tf_write will fix */
   pack->tf_block = cn->tf_expected;
   e = tf_write(cn, sizeof(struct tfack));   /* send the ack */
   return e;
}

/* tf_write() - write (send) a tftp packet to the other host */

int
tf_write(struct tfconn * cn, unsigned len)
{
struct tfack * packet;

   packet = (struct tfack *)(cn->tf_outbuf.data);
   if(packet == NULL)
   {
      dtrap(); /* prog error */
      return ENP_NOBUFFER;
   }
   if(packet->tf_op != TF_RRQ && packet->tf_op != TF_WRQ)
   {
      packet->tf_block = (packet->tf_block);
      cn->tf_tries = TFTPTRIES;
   }      
   else 
      cn->tf_tries = REQTRIES;

   packet->tf_op = (packet->tf_op);   /* convert op to net format */

   cn->tf_lastlen = len;
   cn->tf_snt++;
   cn->tf_sent = cticks;
   cn->tf_tick = cticks + cn->tf_rt;   /* set time to retry */
   cn->tf_NR = 1;

   return(tftp_udpsend(cn, cn->tf_outbuf.data, len));
}


/* Send a TFTP data block. */

int
tfsndata(struct tfconn *cn)
{
struct tfdata * tfdata;
int bytes = 0;   /* bytes read from file */
int err;

   tftp_udpbuffer(cn, sizeof(struct tfdata));
   if(cn->tf_outbuf.data == NULL)
      return ENP_NOBUFFER;
   tfdata = (struct tfdata *) (cn->tf_outbuf.data);
   tfdata->tf_op = TF_DATA;
   tfdata->tf_block = cn->tf_expected;

   /* load file data into tftp buffer */
   if(!cn->tf_NR)   /* if this is NOT a retry, read in new data */
   {
      bytes = vfread(tfdata->tf_data, 1, NORMLEN, cn->tf_fd);   /* read next block from file */
      if(bytes < NORMLEN)      /* end of file? */
      {
         if(vferror(cn->tf_fd))   /* see if it;'s an error */
         {
            if(cn->callback)
                cn->callback(TFC_FILEREAD, cn, "file read error");
            return FALSE;
         }
         /* else at End Of File; fall through to do last send */
      }
      cn->tf_flen = bytes;   /* bytes in last packet sent */
      cn->tf_size += bytes;   /* total bytes sent so far */
   }
   else
   {   dtrap();   /* can this happen? */
   }
   err = tf_write(cn, sizeof(struct tfdata)-NORMLEN+bytes); /* send the data block */
   if(err == 0)   /* if sent OK, wait for reply */
   {
      if(cn->tf_flen == NORMLEN)   /* this a full sized block? */
         cn->tf_state = ACKWAIT;      /* yes, normal wait for ack */
      else
         cn->tf_state = SENTLAST;   /* no, this is last block to send */
   }
   else   /* else kill connection */
   {
      if (cn->callback)
      {
         if (err == ENP_NOBUFFER)
            cn->callback(TFC_BUFFER, cn, "UDP alloc failed");
         else
            cn->callback(TFC_UDPSEND, cn, "UDP send failed");
      }
      tfkill(cn);
   }
   return err;
}

/* Handle an incoming ack. */

void
tfdoack(struct tfconn *cn)
{
struct tfack *ack;
TFTPBUF p = &cn->tf_inbuf;

   ack = (struct tfack *)(p->data);

   if((ack->tf_block) != cn->tf_expected) 
   {/*  We have received an ACK,
         but not for the data block we sent.  It must be for
         a duplicate, since we wouldn't have sent
         the current data block if we hadn't gotten an ACK for
         the previous one.  This duplicate ACK means either
         that the network resent a packet that it wasn't sure
         got through, or else the other end resent the ACK
         because our current data block is lost or late.
         In either case, we can safely ignore this extra ACK,
         and if the ACK we want doesn't come our own timer will
         get us started again.  It isn't safe
         to resend the current data block now unless we are
         absolutely certain that the other end won't reack
         it if the earlier send was just delayed.  */

      cn->tf_ous++;
   }
   else 
   {
      tf_good(cn);
      /* If ack was for last packet, we are done with this connection */
      if(cn->tf_state == SENTLAST)
      {
         cn->tf_state = TERMINATED;
      }
      else
         cn->tf_state = RCVACK;

      tftp_udpfree(cn);    /* free acked data bufffer */
      cn->tf_expected++;   /* ready to do next block */
   }
}


/* tftprcv() - Handle an incoming TFTP packet. 

   This is upcalled from the tftpudp.c code which  handles UDP semantics. 
Note that contents of cn->tf_inbuf are only good for the duration of this 
upcall.

*/

void
tftprcv(struct tfconn *cn, unshort fport)
{
TFTPBUF p;
struct tfdata * pdata;
unshort op;
unsigned len;

   p = &cn->tf_inbuf;   /* data is in connection's buffer */
   len = p->dlen;      /* size of received data in buffer */

   /* If tf_fport is still not set, this should be ack for first block */
   if(cn->tf_fport == TFTPPORT)
   {
      struct tfack * ack = (struct tfack *)(p->data);
      if((ack->tf_block) <= 1)   /* only do this on first block */
         cn->tf_fport = fport;   /* set port from upcalled value */
   }
   cn->tf_rcv++;
   pdata = (struct tfdata *) (p->data);
   op = (pdata->tf_op);
   pdata->tf_op = op;

   switch(op)
   {
      case TF_RRQ:   /* retry of original req? */
      case TF_WRQ:
         break;      /* silently ignore... */
      case TF_DATA:
         tfdodata(cn, p, len);
         break;
      case TF_ACK:
         tfdoack(cn);
         break;
      case TF_ERROR:
         tfdoerr(cn, p, len);
         break;
      default:
         dtrap(); /* printf("TFTPRCV: Got bad opcode %u.\n", op); */
         tfsnderr(cn, ILLTFTP, " ");
         break;
   }
   check_state(cn);   /* see if we need to send something */
   return;
}


void
tfkill(struct tfconn * cn)
{
   cn->tf_state = DEAD;   /* will get cleaned up later */
   cn->tf_tick = cticks + TPS;      /* clean up after one second */
}

/* the tftp task/superloop handler. This scans all tftp connections to
see if any states have timed out. This should be called on every
received tftp packet and 1 second timer. 
*/

static int in_tftp_tick = 0;  /* re-entry guard */

void
tftp_tick(void)
{
struct tfconn * cn, * nextcn;

   if(in_tftp_tick)   /* avoid re-entring */
      return;
   in_tftp_tick++;

   for(cn = tftp_conns; cn; cn = nextcn)
   {
      nextcn = cn->next;   /* in case cn is deleted in check_state */
      if(cn->tf_tick > cticks)   /* time for state check? */
         continue;
      check_state(cn);
   }
   in_tftp_tick--;
}

void
check_state(struct tfconn * cn)
{
char * msg;

   /* prevent clock ticks from re-entering. Do not exit
   this routine without decrementing this counter! */
   in_tftp_tick++;

   switch (cn->tf_state)
   {
   case DATAWAIT:    /* waiting for data? */
   case ACKWAIT:     /* waiting for ack? */
   case SENTLAST:    /* sent last data block? */
      if(cn->tf_tick <= cticks)   /* timeout now? */
         tftptmo(cn);
      break;
   case TIMEOUT:
      tfsnderr(cn, ERRTXT, "Retry limit exceeded, giving up");
      if (cn->callback)
         cn->callback(TFC_TIMEOUT, cn, "retry limit exceeded");
      tfkill(cn);
      break;
   case RCVLASTDATA:   /* receive last incoming block of file */
   case TERMINATED:   /* sent last outgoing block of file */
      if(cn->callback)
      {
         if(cn->tf_size == 0L)
            cn->callback(TFC_ZEROFILE, cn, "zero length file");
         else
         {
         unsigned secs, tenths, elapsed;

            msg = cn->tf_inbuf.data;
            elapsed = (unsigned)(cticks - cn->tf_start);   /* elapsed ticks for transfer */
            secs = elapsed/TPS;      /* round down to number of whole seconds */
            tenths = ((unsigned)(cticks - cn->tf_start) - (secs*TPS))/2;   /* approximate 10ths of a second */
            sprintf(msg, "Transferred %lu bytes in %u.%u seconds", 
               cn->tf_size, secs, tenths);
            cn->callback(TFC_SUCCESS, cn, msg);
         }
      }
      cn->tf_state = DEAD;
      break;
   case RCVACK:   /* got an ack */
      tfsndata(cn);      /* send more data */
      break;
   case RCVDATA:
      tfsndack(cn);
      break;
   case DEAD:
      tfcleanup(cn);
      break;
   default:
      dtrap();   /* bad state */
      if (cn->callback)
         cn->callback(TFC_BADSTATE, cn, " ");
      tfkill(cn);
      break;
   }
   in_tftp_tick--;
}


