/*
 * FILENAME: dnsclnt.c
 *
 * Copyright 1997- 2006 By InterNiche Technologies Inc. All rights reserved
 *
 * Portions Copyright 1986 by Carnegie Mellon
 * Portions Copyright 1984 by the Massachusetts Institute of Technology
 *
 * Copyright (c) 1982, 1986, 1988 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation and other 
 * materials related to such distribution and use acknowledge that 
 * the software was developed by the University of California, Berkeley.
 * The name of the University may not be used to endorse or promote 
 * products derived from this software without specific prior written 
 * permission. THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 *
 * Rights, responsibilities and use of this software are controlled by
 * the agreement found in the "LICENSE.H" file distributed with this
 * source code.  "LICENSE.H" may not be removed from this distribution,
 * modified, enhanced nor references to it omitted.
 *
 * DNS Client code for IP stack
 *
 * MODULE: INET
 *
 * ROUTINES: dns_query(), dnc_sendreq(), dnc_copyout(), 
 * ROUTINES: dnc_copyin(), dnc_del(), dnc_new(), dns_check(), 
 * ROUTINES: badreply(), dns_upcall(), dnc_lookup(), dns_lookup(), 
 * ROUTINES: gethostbyname(), dns_state(), 
 *
 * PORTABLE: yes
 */

#include "license.h"
#include "ipport.h"

#ifdef DNS_CLIENT    /* ifdef out whole file */


#include "q.h"
#include "netbuf.h"
#include "net.h"
#include "ip.h"
#include "udp.h"
#include "dns.h"

#ifdef IP_V6
#include "ip6.h"
#endif

/* List for querie/database records. */
struct dns_querys * dns_qs;

unshort  dnsids   =  0x1234;  /* seed for unique request IDs */

/* DNS client statistics: */
ulong      dnsc_errors = 0;      /* protocol/implementation runtime errors */
ulong      dnsc_requests = 0;    /* requests sent */
ulong      dnsc_replies = 0;     /* replys received */
ulong      dnsc_good = 0;        /* usable replys received */
ulong      dnsc_tmos = 0;        /* timeouts */
ulong      dnsc_retry = 0;       /* total retries */

#ifdef DNS_CLIENT_UPDT
ulong      dnsc_updates = 0;     /* Dynamic DNS Updates sent */
#endif /* DNS_CLIENT_UPDT */

/* pending requests, can be used as a flag to spin dnsc_check() task */
unsigned   dnsc_active = 0;

/* retry info, can be overwritten by application code */
unsigned dns_firsttry = 4; /* time to first retry, in seconds */
unsigned dns_trys = 5;     /* max number of retrys */

ip_addr dns_servers[MAXDNSSERVERS]; /* need to be set from NV parameters */

#ifdef  DNS_CLIENT_UPDT
char    soa_mname[MAXDNSNAME];

/* internal routines */
static struct hostent* getsoa(char *);
#endif  /* DNS_CLIENT_UPDT */

/* internal routines */
static int  dnc_sendreq(struct dns_querys *);
static int  dnc_copyout(char * dest, char * src);
static int  dnc_copyin(char * dest, char * src, struct dns_hdr * dns);
static void dnc_del(struct dns_querys *);
static struct dns_querys * dnc_new(void);
static int  badreply(struct dns_hdr * dns, char * text);
static void dnc_setaddr(struct dns_querys *, unshort type, char * cp);
static int  dnc_lookup(char * name, struct dns_querys ** cacheentry);


/* FUNCTION: dns_query()
 *
 * dns_query() - starts the process of sending out a DNS request for 
 * a domain name. This is hardwired for the basic question "what is 
 * the IP address of named host?". It creates the control structures 
 * and calls dnc_sendreq() to send the actual packet. Caller should 
 * poll dns_lookup for results. 
 *
 * Old version, for backward API compatibility. This is implemented
 * as a wrapper for the new function
 *
 *
 * PARAM1: char * name
 * PARAM2: ip_addr * ip
 *
 * RETURNS: Returns 0 if IP address was filled in 
 * from cache, ENP_SEND_PENDING if query was launched OK, or one of 
 * the ENP error codes. 
 */

int
dns_query(char * name, ip_addr * ip_ptr)
{
   int      err;
   struct dns_querys * dns_entry;

   /* Do query for an "A" record (DNS_TYPE_IPADDR) */
   err = dns_query_type(name, DNS_TYPE_IPADDR, &dns_entry);
   if(!err)
      *ip_ptr = dns_entry->ipaddr_list[0];

   return err;
}


/* FUNCTION: dns_query_type()
 *
 * New (for v2.0) version of dns_query()
 * This creates the control structures and calls dnc_sendreq() to
 * send the DNS query packet. Caller should 
 * poll dns_lookup for results. 
 *
 *
 * PARAM1: char * name - IN - host name to look up
 * PARAM2: char type - IN - type of query (V4, V6, SERVER, etc.)
 * PARAM3: dns_querys ** dnsptr - OUT - dns entry 
 *
 * RETURNS: Returns 0 if IP address was filled in 
 * from cache, ENP_SEND_PENDING if query was launched OK, or one of 
 * the ENP error codes. 
 */

int
dns_query_type(char * name, char type, struct dns_querys ** dns_ptr)
{
   struct dns_querys *  dns_entry;
   int   e;

   /* see if we already have an entry for this name in the cache */
   e = dnc_lookup(name, dns_ptr);
   if (e == 0) /* no error */
      return 0;
   else if(e == ENP_SEND_PENDING)   /* waiting for previous query */
      return e;

   /* else find a free entry so we can start a query */
   dns_entry = dnc_new();
   if (dns_entry == NULL)
      return ENP_RESOURCE;

   /* prepare entry for use and copy in name. The name for the query will
    * be replaced by the reply, and other strings (alias, server name) may
    * be added to the buffer if room permits.
    */
   strncpy(dns_entry->dns_names, name, MAXDNSNAME-1);  /* copy in name */

   dns_entry->tries = 0;      /* no retries yet */
   dns_entry->id = dnsids++;  /* set ID for this transaction */
   dns_entry->ipaddr_list[0] = 0L;  /* non-zero when it succeeds */
   if (dnsids == 0)              /* don't allow ID of 0 */
      dnsids++;

   /* get UDP port for packet, keep for ID */
   dns_entry->lport = udp_socket();
   dnsc_active++;       /* maintain global dns pending count */
   dnsc_requests++;     /* count this request */

   dns_entry->type = type;    /* set type of DNS query */
   *dns_ptr = dns_entry;      /* return DNS entry */

   e = dnc_sendreq(dns_entry);
   if (e == 0) /* first packet sent OK */
      return ENP_SEND_PENDING;
   else  /* some kind of error */
      return e;
}



/* FUNCTION: dnc_sendreq()
 *
 * dnc_sendreq() - Sends out a DNS request packet based on the query 
 * info in dns_entry. This is intended to be called from 
 * dns_query() or dns_check(). 
 *
 * 
 * PARAM1: int entry
 *
 * RETURNS: Returns 0 if OK, or one of the ENP error codes. 
 */

static int
dnc_sendreq(struct dns_querys * dns_entry)
{      
   PACKET pkt;
   struct dns_hdr *  dns;  /* outgoing dns header */
   char *   cp;   /* scratch pointer for building question field */
   int   server;  /* index of server to use */
   
   
   /* figure out which server to try */
   for (server = 0; server < MAXDNSSERVERS; server++)
   {
      if (dns_servers[server] == 0L)
         break;
   }
   if (server == 0)  /* no servers set? */
   {
      dtrap();
      return ENP_LOGIC;
   }
   server = dns_entry->tries % server;

   /* allocate packet for DNS request */
   if ((pkt = udp_alloc(MAXDNSUDP, 0)) == NULL)
      return ENP_RESOURCE;

   /* fill in DNS header. Most fields are cleared to 0s */
   dns = (struct dns_hdr *)pkt->nb_prot;
   MEMSET(dns, 0, sizeof(struct dns_hdr));
   dns->id = dns_entry->id;

#ifdef DNS_CLIENT_UPDT
   /* If this is an UPDATE packet, format the DNS packet differently */
   
   if (dns_entry->type == DNS_UPDT)
   {
      dns->qdcount = (1);      /* Set zone count to 1 */
      dns->nscount = (1);      /* Set update count to 1 */
      dns->flags = (0x2800);   /* Set opcode field to 5 */

      /* format zone name into UPDATE packet */
      cp = (char*)(dns+1);      /* point at next byte past header */
      cp += dnc_copyout(cp, dns_entry->he.h_z_name);

      /* finish off zone section. We write these two 16 bit words a
       * byte at a time since cp may be on an odd address and some
       * machines
       */
      *cp++ = 0;  /* high byte of type */
      *cp++ = 6;  /* type 6 = soa */
      *cp++ = 0;  /* high byte of class */
      *cp++ = 1;  /* class 1 == internet */
      cp += dnc_copyout(cp, dns_entry->dns_names);

      /* Put in NAME and TYPE */
      *cp++ = 0;  /* high byte of type */

      /* If ttl value is 0, this is a delete operation. Set type to ANY */
      if (dns_entry->he.h_ttl == 0)
         *cp++ = (unsigned char)255;
      else
         *cp++ = 1;  /* type 1 == host address, type 6 = soa */
      *cp++ = 0;     /* high byte of class */
      *cp++ = 1;     /* class 1 == internet */

      /* Put in TTL value */
      *cp++ = (unsigned char)(dns_entry->he.h_ttl >> 24);
      *cp++ = (unsigned char)(dns_entry->he.h_ttl >> 16);
      *cp++ = (unsigned char)(dns_entry->he.h_ttl >> 8);
      *cp++ = (unsigned char)dns_entry->he.h_ttl;

      /* Put in RDLENGTH which is length of ip address ie 4 */
      *cp++ = 0;  /* low byte of length */

      /* If ttl value is 0, this is a delete operation. Set RDLENGTH to 0 */
      if (dns_entry->he.h_ttl == 0)
         *cp++ = 0;
      else
         *cp++ = 4;
      /* Put in IP address */
      MEMCPY(cp, &dns_entry->he.h_add_ipaddr, 4);
      cp += 4;
      dnsc_updates++;
   }
   else
   {
#endif  /* DNS_CLIENT_UPDT */


      dns->qdcount = (1);      /* 1 question */
      dns->flags = (DNSF_RD);  /* Recursion Desired */

      /* format name into question field after header */
      cp = (char*)(dns + 1);  /* point at next byte past header */
      cp += dnc_copyout(cp, dns_entry->dns_names);

      /* finish off question field. We write these two 16 bit words a 
       * byte at a time since cp may be on an odd address and some 
       * machines 
       */
      *cp++ = 0;  /* high byte of type */
      *cp++ = dns_entry->type;  /* type 1 == host address, type 6 = soa */
      *cp++ = 0;  /* high byte of class */
      *cp++ = 1;  /* class 1 == internet */

#ifdef DNS_CLIENT_UPDT
   }
#endif /* DNS_CLIENT_UPDT */
   pkt->fhost = dns_servers[server];
   pkt->nb_plen = cp - (char*)dns;     /* length of packet */
   dns_entry->send_time = cticks;   /* note time we send packet */

   return(udp_send(DNS_PORT, dns_entry->lport, pkt));
}


/* FUNCTION: dnc_copyout()
 *
 * dnc_copyout() - copy a domain name from user "dot" text format to 
 * DNS header format. dest buffer is assumed to be large enough, 
 * which means at least as big as src string.
 * 
 * PARAM1: char * dest
 * PARAM2: char * src
 *
 * RETURNS:  Returns length of string copied. 
 */

static int
dnc_copyout(char * dest, char * src)
{
   int   namelen; /* length of name segment */
   char *   cp;   /* pointer to dest string we're building */
   char *   fld, *   next; /* pointers into src */

   cp = dest;
   next = src; /* will be assigned to "this" on first pass */

   while (next)   /* 1 loop per field in dot-delimited string */
   {
      fld = next;
      next = strchr(fld, '.');   /* find next dot in string */
      if (next)   /* if not last field... */
      {
         namelen = next - fld;   /* length of this field */
         next++;  /* bump next pointer past dot */
      }
      else  /* no dot means end of string */
         namelen = strlen(fld);

      *cp++ = (char)(namelen & 0xFF);  /* put length in dest buffer */
      MEMCPY(cp, fld, namelen);     /* follow with string data */
      cp += namelen;    /* point past name to next length byte */
   }
   *cp++ = 0;  /* null terminate string */
   return(cp - dest);
}


/* FUNCTION: dnc_copyin()
 *
 * dnc_copyin() - the inverse of copyout above - it copies a dns 
 * format domain name to "dot" formatting. 
 *
 * 
 * PARAM1: char * dest
 * PARAM2: char * src
 * PARAM3: struct dns_hdr * dns
 *
 * RETURNS: Returns length of string copied, 0 if error. 
 */

int
dnc_copyin(char * dest, char * src, struct dns_hdr * dns)
{
   unshort   namelen; /* length of name segment */
   char *   cp;   /* pointer to dest string we're building */
   char *   fld, *   next; /* pointers into src */
   int   donelen;    /* number of bytes moved */
   unshort offset;

   cp = dest;
   next = src; /* will be assigned to "this" on first pass */
   donelen = 0;

   while (next)   /* 1 loop per field in dot-delimited string */
   {
      fld = next;
      namelen = *fld++;
      if (namelen == 0)
      break;   /* done */
      if ((namelen & 0xC0) == 0xC0)
      {
         fld--;
         offset = (unshort)*fld; /* get first byte of offset code */
         fld++;
         offset &= 0x3f;             /* mask our high two bits */
         offset <<= 8;               /* make it high byte of word */
         offset += (unshort)*fld; /* add low byte of word */
         fld = offset + (char *)dns;  /* ptr into domain name */
         namelen = *fld++;
      }
      if (namelen + donelen > MAXDNSNAME) /* check for overflow */
         return 0;   /* error */
      MEMCPY(cp, fld, namelen);
      donelen += (namelen+1); /* allow for dot/len byte */
      cp += namelen;
      *cp++ = '.';
      next = fld + namelen;
   }
   *(cp-1) = 0;   /* null terminate string (over last dot) */
   return donelen + 1;  /* include null in count */
}



/* FUNCTION: dnc_del()
 *
 * Delete a DNS entry
 *
 * PARAM1: int entry
 *
 * RETURNS: void
 */

static void
dnc_del(struct dns_querys * entry)
{
   struct dns_querys * tmp;
   struct dns_querys * last;

   /* find passed dns entrery in list */
   last = NULL;
   for(tmp = dns_qs; tmp; tmp = tmp->next)
   {
      if(tmp == entry)  /* found entry in list */
      {
         if(last)          /* unlink */
            last->next = tmp->next;
         else
            dns_qs = tmp->next;
         break;
      }
      last = tmp;
   }

   DNC_FREE(entry);  /* fre the memory */
   dnsc_active--;    /* one less active entry */
}

/* FUNCTION: dnc_ageout()
 *
 * See if we can age out any of the DNS entries.
 * 
 * PARAM1: void
 *
 * RETURNS: void
 */

static void
dnc_ageout()
{
   struct dns_querys * dns_entry;

   LOCK_NET_RESOURCE(NET_RESID);

   /* See if we can expire any old entries */
   for(dns_entry = dns_qs; dns_entry; dns_entry = dns_entry->next)
   {
      /* don't use entries that haven't resolved yet */
#ifdef IP_V6AND4
      if((dns_entry->ipaddrs == 0) && (dns_entry->ip6addrs == 0))
         continue;
#elif IP_V4
      if (dns_entry->ipaddrs == 0)
         continue;
#elif MINI_IP
      if (dns_entry->ipaddrs == 0)
         continue;
#else /* IPv6 */
      if(dns_entry->ip6addrs == 0)
         continue;
#endif

      /* If entry has expired then delete it. */
      if (dns_entry->expire_time < cticks)
         dnc_del(dns_entry);
   }
   UNLOCK_NET_RESOURCE(NET_RESID);
}


/* FUNCTION: dnc_new()
 *
 * Tries to make an empty entry use in the cache. 
 *
 * 
 * PARAM1: void
 *
 * RETURNS: void
 */

static struct dns_querys *
dnc_new(void)
{
   struct dns_querys * dns_entry;

   dns_entry = DNC_ALLOC(sizeof (struct dns_querys));  /* get new entry */

   /* set respose code to an illegal value. Response will overwrite */
   dns_entry->rcode = DNSRC_UNSET;

   /* set legacy hostent alias pointer to dns_entry alias list */
   dns_entry->he.h_aliases = &dns_entry->alist[0];

   /* Put new entry in the DNS database/request list */
   LOCK_NET_RESOURCE(NET_RESID);
   dns_entry->next = dns_qs;
   dns_qs = dns_entry;
   UNLOCK_NET_RESOURCE(NET_RESID);

   return dns_entry;
}


/* FUNCTION: dns_check()
 *
 * dns_second() - poll once a second to drive DNS timeouts and 
 * retrys. Can be skipped when dnsc_active == 0 
 * 
 * PARAM1: 
 *
 * RETURNS: void
 */

void
dns_check(void)
{
   struct dns_querys * dns_entry;
   int   trysec;  /* seconds to wait for next try */

   dnc_ageout();     /* age out expired entries */

   for(dns_entry = dns_qs; dns_entry; dns_entry = dns_entry->next)
   {
      if (dns_entry->id == 0) /* skip non-active request entrys */
         continue;

      /* If we already have a reply we like, don't send */
      if(dns_entry->rcode == DNSRC_OK)
         continue;

      /* If it's a name error then punt the request */
      if(dns_entry->rcode == DNSRC_ENAME)
      {
         /* only if it's over 10 seconds old */
         if(((cticks - dns_entry->send_time)/TPS) > 10)
            goto timeout;
      }


      /* active request, see if it's time to retry */
      trysec = dns_firsttry << dns_entry->tries;
      if ((dns_entry->send_time + (TPS*(unsigned long)trysec)) < cticks)
      {
         if (dns_entry->tries >= dns_trys)   /* retried out */
         {
timeout:
            dnc_del(dns_entry);
            dnsc_tmos++;      /* count timeouts */
            /* After a timeout we return because the list is now altered.
             * We'll process the rest of it on the next time tick.
             */
            return;
         }
         dnsc_retry++;           /* count total retries */
         dns_entry->tries++;  /* count entry retries */
         dnc_sendreq(dns_entry);
      }
   }
}


/* FUNCTION: badreply()
 *
 * dns_badreply() - per-port handler for less than ideal DNS replys. 
 * 
 * 
 * PARAM1: struct dns_hdr * dns
 * PARAM2: char * text
 *
 * RETURNS: Returns ENP code if problem should kill transaction, else 0 
 */

static int 
badreply(struct dns_hdr * dns, char * text)
{
   dnsc_errors++;
#ifdef NPDEBUG
   dprintf("DNS error: %s; flags %x\n", text, (dns->flags));
#endif   /* NPDEBUG */
   /* don't kill request, let it retry with another server */
   USE_ARG(dns);
   USE_ARG(text);
   return 0;
}

static
char *
getshort(char * cp, unshort * val)
{
   *val = (unshort)(*cp++ << 8);
   *val += (unshort)(*cp++);
   return (cp);
}

static
char *
getlong(char * cp, u_long * val)
{
   *val = (u_long)((u_char)*cp++) << 24;
   *val += (u_long)((u_char)*cp++) << 16;
   *val += (u_long)((u_char)*cp++) << 8;
   *val += (u_long)((u_char)*cp++);
   return (cp);
}

static char *
getoffset(char * cp, char * dns, unshort * offsetp)
{
   unshort  offset;

   /* bump past name field in answer, keeping offset to text */
   if ((*cp & 0xC0) == 0xC0)  /* is it an offset? */
   {
      offset = (unshort)(*cp++); /* get first byte of offset code */
      offset &= 0x3f;   /* mask our high two bits */
      offset <<= 8;     /* make it high byte of word */
      offset += (unshort)(*cp++);   /* add low byte of word */
   }
   else  /* text for name is right here */
   {
      offset = (unshort)(cp - dns);   /* save offset */
      while (*cp++ != 0);  /* scan past name */
   }
   *offsetp = offset;
   return cp;
}

static void
dnc_setaddr(struct dns_querys * dns_entry, unshort type, char * cp)
{
   int   addrx;      /* index into address lists */

   /* save reply IP addresses in array of IP addresses so long
    * as there is room for them.
    */
#ifdef IP_V4
   if(type == DNS_TYPE_IPADDR)
   {
      if (dns_entry->ipaddrs < MAXDNSADDRS)
      {
         addrx = dns_entry->ipaddrs++;      /* set index */
         MEMCPY(&dns_entry->ipaddr_list[addrx], cp, 4);
      }
      return;
   }
#endif   /* IP_V4 */

#ifdef IP_V6
   if(type == DNS_TYPE_AAAA)     /* IPv6 address */
   {
      if (dns_entry->ip6addrs < MAXDNSADDRS)
      {
         addrx = dns_entry->ip6addrs++;      /* set index */
         MEMCPY(&dns_entry->ip6_list[addrx], cp, 16);
      }
      return;
   }
#endif   /* IP_V6 */
}


/* FUNCTION: dnc_savename()
 *
 * Save a passed name in the passed dns_query structure. name is given
 * via an offset into the dns packet.
 *
 * 
 * PARAM1: dns_query structure to add name to
 * PARAM2: pointer to dns header
 * PARAM3: offset into dns header for name info
 * PARAM3: TRUE if name is an alias
 *
 * RETURNS: void
 */

void
dnc_savename(struct dns_querys *  dns_entry,
            struct dns_hdr * dns, 
            int offset,
            int aliasflag)
{
   char * cp;     /* pointer to dns_names[] name buffer */

   /* find next available space in name buffer */
   cp = dns_entry->dns_names;
   while(*cp)
   {
      if(*cp)
         cp += (strlen(cp) + 1); /* end of name (if any) */

      /* check for buffer overflow */
      if(cp >= &dns_entry->dns_names[MAXDNSNAME])
      {
         dtrap();    /* ran out of buffer space for names */
         return;
      }
      if(*cp == 0)
         break;
   }

   /* copy dns-format name into buffer as text */
   dnc_copyin(cp, offset + (char*)(dns), dns);

   /* Set pointer(s) in dns structures to new name */
   if(aliasflag)
   {
      int alias;     /* alias name index */
      /* Set the next alias pointer. First we have to find out
       * how many aliases are already in dns_entry.
       */
      for(alias = 0; alias < MAXDNSALIAS; alias++)
      {
         if(dns_entry->alist[alias] == NULL)
            break;
      }
      if(alias >= MAXDNSALIAS)
          return;
      /* set alias pointer to alias name in dns_names[] buffer */
      dns_entry->alist[alias] = cp;
   }
   else  /* set the name pointer(s) */
   {
      /* The hostent name field always points to dns_name */
      dns_entry->he.h_name = cp;
   }
   return;
}

/* FUNCTION: dns_upcall()
 *
 * dns_upcall() - called from the UDP layer whenever we receive a DNS 
 * packet (one addressed to port 53). p_nb_prot points to UDP data, 
 * which should be DNS header. Return 0 if OK, or ENP code. 
 *
 * 
 * PARAM1: PACKET pkt
 * PARAM2: unshort lport
 *
 * RETURNS: 
 */

int
dns_upcall(PACKET pkt, unshort lport)
{
   int      i;          /* record index */
   char *   cp;         /* scratch pointer for building question field */
   unshort  flags;      /* dns->flags in local endian */
   unshort  rcode;      /* response code */
   unshort  queries;    /* number of question records in reply */
   unshort  answers;    /* number of answer records in reply */
   unshort  records;    /* total records in reply */
   unshort  type;       /* various fields from the reply */
   unshort  netclass;   /* class of net (1 == inet) */
   u_long   ttl;        /* records time to live */
   unshort  rdlength;   /* length of record data */
   unshort  offset;     /* scratch offset to domain name text */
   unshort  nameoffset = 0;   /* offset to name in query */
   unshort  authoffset = 0;   /* offset to first auth server name */
   struct dns_hdr *  dns;     /* incoming dns header */
   struct dns_querys *  dns_entry;

   dnsc_replies++;
   dns = (struct dns_hdr *)pkt->nb_prot;

   for (dns_entry = dns_qs; dns_entry; dns_entry = dns_entry->next)
   {
      if ((dns_entry->id == dns->id) &&
          (dns_entry->lport == lport))
      {
         break;
      }
   }

   /* make sure we found our entry in the table */
   if (dns_entry == NULL)
      return ENP_NOT_MINE;         /* do not free pkt here */

   dns_entry->replies++;           /* count total replies */

   /* If we already have a reply we liked then punt this one */
   if(dns_entry->rcode == DNSRC_OK)
      return 0;
   
   flags = (dns->flags);      /* extract data fromheader */
   rcode = flags & DNSF_RCMASK;    /* Response code */
   queries = (dns->qdcount);    /* number of questions */
   answers = (dns->ancount);  /* number of answers */
   records = queries + answers + 
      (dns->nscount) + (dns->arcount);

   /* check DNS flags for good response value */
   if (!(flags & DNSF_QR))
      return(badreply(dns, "not reply"));

   /* looks like we got an answer to the query */
   dns_entry->rcode = rcode;

#ifdef DNS_CLIENT_UPDT
 {
   int opcode = (flags & DNSF_OPMASK) >> 11;   /* Op code */

   /* If UPDATE opcode set the ip address field to non zero and return */
   if (opcode == DNS_UPDT)
   {
      dns_entry->ipaddr_list[0] = 0xff;
      return 0;
   }
 }
#endif /* DNS_CLIENT_UPDT */

   if (rcode != DNSRC_OK) 
      return(badreply(dns, "bad response code"));

   if (answers < 1)
      return(badreply(dns, "no answers"));

   cp = (char*)(dns+1);    /* start after DNS entry */

   /* Since we're going to store new DNS info, clean out the old info */
   MEMSET(dns_entry->dns_names, 0, MAXDNSNAME);
   MEMSET(dns_entry->alist, 0, sizeof(dns_entry->alist) );
   dns_entry->ipaddrs = 0;

#ifdef IP_V6
   dns_entry->ip6addrs = 0;
   MEMSET(dns_entry->ip6_list, 0, sizeof(dns_entry->ip6_list) );
   MEMSET(dns_entry->auths_ip6, 0, sizeof(dns_entry->auths_ip6) );
#endif   /* IP_V6 */

   /* loop through remaining records - answers, auth, and RRs */
   for (i = 0; i < records; i++)
   {
      /* Get offset to record's name */
      cp = getoffset(cp, (char *)dns, &offset);

      /* get records type and class from packet bytes */
      cp = getshort(cp, &type);
      cp = getshort(cp, &netclass);
      if (netclass != 1)
         return badreply(dns, "class not internet");

      if(i < queries)      /* just skip of echos question record */
         continue;

      /* Get the Time and data-length */
      cp = getlong(cp, &ttl);       /* 4 byte time to live field */
      cp = getshort(cp, &rdlength); /* length of record data */

      switch(type)
      {
      case DNS_TYPE_IPADDR:   /* IPv4 address for a name */
#ifdef IP_V6
      case DNS_TYPE_AAAA:     /* IPv6 address */
#endif   /* IP_V6 */

         if ((type == DNS_TYPE_IPADDR) && (rdlength != 4))
            return(badreply(dns, "IPADDR len not 4"));

         dnsc_good++;
         if(i < (queries + answers))   /* If record is an answer... */
         {
            if(nameoffset == 0)
            {
               /* save first answer name over request name */
               nameoffset = offset;

               /* save the name in the local DNS structure */
               dnc_savename(dns_entry, dns, offset, 0);
            }

            dnc_setaddr(dns_entry, type, cp);    /* save address */
            dns_entry->expire_time = (cticks/TPS) + (ttl);  /* save ttl */
         }
         else  /* auth record or additional record */
         {
            if(offset == nameoffset)   /* another addr for query? */
            {
               dnc_setaddr(dns_entry, type, cp);
            }
            else if(offset == authoffset) /* auth server IP address */
            {
               if(type == DNS_TYPE_IPADDR)
                  MEMCPY(&dns_entry->auths_ip, cp, 4);
#ifdef IP_V6
               if(type == DNS_TYPE_AAAA)
                  MEMCPY(&dns_entry->auths_ip6, cp, 16);
#endif   /* IP_V6 */

            }
         }
         break;

      case DNS_TYPE_AUTHNS: /* authoritative name server */
         /* What we really want for the name server is it's IP address,
          * however this record only contains a name, ttl, etc. We save 
          * the offset to the name, hoping that one of the additional 
          * records will have the IP address matching this name.
          */
         if(authoffset == 0)  /* only save first one */
            authoffset = cp - (char*)dns;
         break;

      case DNS_TYPE_ALIAS:  /* alias */
         /* save name in dns rec as an alias */
         dnc_savename(dns_entry, dns, offset, 1);
         break;

      default:       /* unhandled record type, ignore it. */
         break;
      }
      cp += rdlength;   /* bump past trailing data to next record */
   }
   return 0;
}



/* FUNCTION: dnc_lookup()
 *
 * dnc_lookup() - check cache for a pending or completed DNS client 
 * request. Passed entry index is filled in if DNS query was resolved 
 * without error, else this returns non-zero ENP_ error code. 
 *
 * 
 * PARAM1: char * name
 * PARAM2: int * cacheindex
 *
 * RETURNS: ENP_SEND_PENDING if request is still on the net, or ENP_TIMEOUT
 * id request timed out (or was never made), or 0 (SUCCESS) if passed 
 * name found in cache, or ENP_PARAM if 
 */

static  int
dnc_lookup(char * name, struct dns_querys ** cacheentry)
{
   struct dns_querys * dns_entry;
   int      alias;

   /* look through the cache for the passed name */
   for(dns_entry = dns_qs; dns_entry; dns_entry = dns_entry->next)
   {
      if(strcmp(dns_entry->dns_names, name) == 0)
         break;

      if((dns_entry->he.h_name) &&
         (strcmp(dns_entry->he.h_name, name) == 0))
      {
         break;
      }
      for(alias = 0; alias < MAXDNSALIAS; alias++)
      {
         if((dns_entry->alist[alias]) &&
            (strcmp(dns_entry->alist[alias], name) == 0))
               goto found_name;
      }
   }

found_name:

   /* if not found, return error */
   if (dns_entry == NULL)
      return ENP_PARAM;

   /* else, prepare to return entry index */
   if (cacheentry != NULL)
      *cacheentry = dns_entry;

   /* if completed request, return success */
   if (dns_entry->rcode != DNSRC_UNSET)
      return 0;

   /* incomplete request -- return pending or timed-out status */
   if (dns_entry->tries < dns_trys) /* still trying? */
      return ENP_SEND_PENDING;
   else
      return ENP_TIMEOUT;
}



/* FUNCTION: dns_lookup()
 *
 * dns_lookup() - check state of a DNS client request previously 
 * launched with dns_query(). Passed IP address is filled in (in net 
 * endian) if DNS query was resolved without error, else this clears 
 * passed Ip address and returns non-zero ENP_ code. 
 *
 * 
 * PARAM1: ip_addr * ip
 * PARAM2: char * name
 *
 * RETURNS: ENP_SEND_PENDING if request is still on the net, or ENP_TIMEOUT
 * if request timed out (or was never made), or 0 (SUCCESS) if passed IP 
 * is filled in and everything is peachy, or other ENP errors if 
 * detected. 
 */

int   
dns_lookup(ip_addr * ip, char * name)
{
   struct dns_querys * dns_entry;
   int      err;

   *ip = 0L;
   err = dnc_lookup(name, &dns_entry);
   if (err)
      return err;
   *ip = dns_entry->ipaddr_list[0];    /* return IP address */
   return 0;
}


/* static hostent structure for gethostbyname to return when it's
 * been passed a "name" which is just a dot notation IP address.
 */
static struct hostent dnc_hostent;
static ip_addr dnc_addr;         /* actual output in hostent */
static char *  dnc_addrptrs[2];  /* pointer array for hostent */

/* FUNCTION: gethostbyname()
 *
 * "standard" Unixey version of gethostbyname() returns pointer to a
 * hostent structure if 
 * successful, NULL if not successful. 
 *
 * PARAM1: char *name
 *
 * RETURNS:  pointer to a hostent structure if successful, NULL if not.
 */

struct hostent *
gethostbyname(char * name)
{
   int      err;
   int      i;
   char *   cp;
   unsigned long tmo;
   unsigned int snbits;
   struct dns_querys * dns_entry;
   struct hostent * ho;

   /* first see if the passed in name looks like a dot notation IP address */
   cp = parse_ipad(&dnc_addr, &snbits, name);

   /* if name looks like an dot notation just return the address */
   if (!cp)    
   {
      /* fill in the static hostent structure and return it */
      dnc_hostent.h_name = name;
      dnc_hostent.h_aliases = NULL;  /* we don't do the aliases */
      dnc_hostent.h_addrtype = AF_INET; /* AF_INET */
      dnc_hostent.h_length = 4;   /* length of IP address */
      dnc_hostent.h_addr_list = &dnc_addrptrs[0];
      dnc_addrptrs[0] = (char*)&dnc_addr;
      dnc_addrptrs[1] = NULL;
      return &dnc_hostent;
   }

   /* we will call dns_query_type() until it either succeeds, fails or
    * we time out trying.
    */
   tmo = cticks + (5 * TPS);
   do
   {
      err = dns_query_type(name, DNS_TYPE_IPADDR, &dns_entry);
      if(err != ENP_SEND_PENDING)
         break;
      tk_yield();
   } while (tmo > cticks);

   if(err)
      return NULL;

   /* Set up the hostent fields for IPv4. */
   ho = &dns_entry->he;
   ho->h_addrtype = AF_INET;
   ho->h_length = 4;   /* length of IP address */

   /* fill in the address pointer list in hostent structure */
   for(i = 0; i < dns_entry->ipaddrs; i++)
      dns_entry->addrptrs[i] = (char*)&dns_entry->ipaddr_list[i];

   dns_entry->addrptrs[i] = NULL;      /* last one gets a NULL */
   ho->h_addr_list = &dns_entry->addrptrs[0];

   return &dns_entry->he;
}


#ifdef IP_V6

struct hostent *
gethostbyname2(char * name, int af)
{
   int      err;
   int      i;
   struct hostent * ho;
   struct dns_querys * dns_entry;

   /* Invoke the plain v4 version, which will also get v6 replies if they 
    * are available. If Address Family is AF_INET (v4) then just return 
    * the result.
    */
   ho = gethostbyname(name);
   if((af == AF_INET) ||
      (ho == NULL))     /* also return failures at this point */
   {
      return ho;
   }

   /* fall to here if called wanted a v6 query and got an answer of 
    * some kind.
    */
   err = dnc_lookup(name, &dns_entry);
   if(err)
   {
      dtrap();    /* shouldn't happen, since we got a non-null "ho" */
      return NULL;      /* fail gracefully */
   }

   if(dns_entry->ip6addrs == 0)     /* no IPv6 address? */
      return NULL;

   /* Rework the hostent fields for IPv6 */
   ho = &dns_entry->he;
   ho->h_addrtype = AF_INET6;
   ho->h_length = 16;   /* length of IP address */

   /* fill in the address pointer list in hostent structure */
   for(i = 0; i < dns_entry->ip6addrs; i++)
      dns_entry->addrptrs[i] = &dns_entry->ip6_list[i][0];

   dns_entry->addrptrs[i] = NULL;      /* last one gets a NULL */
   ho->h_addr_list = &dns_entry->addrptrs[0];

   return ho;
}
#endif /* IP_V6 */



#ifdef DNS_CLIENT_UPDT

/* FUNCTION:  unsigned dns_update()
 *
 * dns_update- sends a DNS UPDATE packet to the authoritative server
 * of the specified domain name.  This routine uses the getsoa call
 * to get the authoritative server of the domain name.  It then sends
 * the UPDATE packet to the authoritative server.
 *
 * 
 * PARAM1: char * soa_mname     domain name
 * PARAM2: char * dname         host name
 * PARAM3: ip_addr ipaddress   ip address
 * PARAM4: unsigned long ttl   ttl value
 * PARAM5: IO handle for output (NULL is OK)
 *
 * RETURNS: 0 if successful
 *          negative ENP error if internal error occurs (eg timeout)
 *          else one of the DNSRC_ errors (all positive).
 */




int
dns_update(char * soa_mname, char * dname, ip_addr ipaddr,
            unsigned long ttl, void * pio)
{
   int      err;
   u_long   tmo;
   struct dns_querys * dns_entry;
   unsigned char * ucp;
   ip_addr save_dns_srvr;




   /* get authoritative name server for specified domain. We will
    * call dns_query_type() until it either succeeds, fails or
    * we time out trying.
    */
   tmo = cticks + (5 * TPS);

     err = dns_query_type(dname, DNS_TYPE_AUTHNS, &dns_entry);	  

   do
   {	   
      if ((err = dnc_lookup(dname, &dns_entry)) == 0)
             break;	
      tk_yield();	  
   } while  (cticks < tmo);
   if(err)
      return err;

   /* Get here if we received a hostent structure. Send the update packet
    * to the received IP address (first in list) by swapping the address
    * into the list of DNS servers.
    */

   if (dns_entry->he.h_addr_list)
   {
      ucp = (unsigned char *) *(dns_entry->he.h_addr_list);
      MEMCPY(&save_dns_srvr, dns_servers, 4);   /* save 1st DNS server */
      MEMCPY(dns_servers, ucp, 4);              /* swap in AUTHS address */

      ns_printf(pio," sending update packet to %u.%u.%u.%u\n",
         *ucp, *(ucp + 1), *(ucp + 2), *(ucp + 3));
   }

   /* load up input parameters for DNS UPDATE request */
   dns_entry->he.h_z_name = soa_mname;
   dns_entry->he.h_add_ipaddr = ipaddr;
   dns_entry->he.h_ttl = ttl;

   tmo = cticks + (5 * TPS);
      err = dns_query_type(dname, DNS_TYPE_ALIAS, &dns_entry);

	  
   while (cticks < tmo)
   {
      if ((err = dnc_lookup(dname, &dns_entry)) == 0)
             break;     	  
	  tk_yield();   
   }

   MEMCPY(&dns_servers[0], &save_dns_srvr, 4);
   if(err)
      return err;
   else
      return dns_entry->rcode; /* Return response code */
}

#endif  /* DNS_CLIENT_UPDT */


#ifdef NET_STATS


/* FUNCTION: dns_state()
 * 
 * PARAM1: void * pio
 *
 * RETURNS: 
 */

int
dns_state(void * pio)
{
   struct dns_querys * dns_entry;
   int   i;

   ns_printf(pio,"DNS servers:");
   for (i = 0; i < MAXDNSSERVERS; i++)
      ns_printf(pio,"%u.%u.%u.%u ", PUSH_IPADDR(dns_servers[i]) );

   ns_printf(pio,"\nDNS cache:\n");
   for(dns_entry = dns_qs; dns_entry; dns_entry = dns_entry->next)
   {
      ns_printf(pio,"name: %s, IP: %u.%u.%u.%u, ",
       dns_entry->he.h_name, PUSH_IPADDR(dns_entry->ipaddr_list[0]) );
      ns_printf(pio,"retry:%d, ID:%d, rcode:%d, err:%d\n",
       dns_entry->tries, dns_entry->id, dns_entry->rcode, dns_entry->err);
   }
   ns_printf(pio,"protocol/implementation runtime errors:%lu\n", dnsc_errors);
   ns_printf(pio,"requests sent:%lu\n", dnsc_requests);
#ifdef DNS_CLIENT_UPDT
   ns_printf(pio, "Updates sent:%lu\n", dnsc_updates);
#endif /* DNS_CLIENT_UPDT */
   ns_printf(pio,"replies received:%lu\n", dnsc_replies);
   ns_printf(pio,"usable replies:%lu\n", dnsc_good);
   ns_printf(pio,"total retries:%lu\n", dnsc_retry);
   ns_printf(pio,"timeouts:%lu\n", dnsc_tmos);

   return 0;
}


int
dnc_database(void * pio)
{
   struct dns_querys * dns_entry;
   struct hostent *  p;
   char ** cpp;
   int   i;

   if(dns_qs == NULL)
   {
      ns_printf(pio, "No DNS entries.\n");
      return 0;
   }

   /* look through the cache for the passed name */
   for(dns_entry = dns_qs; dns_entry; dns_entry = dns_entry->next)
   {
      if(dns_entry->rcode != DNSRC_OK)
      {
         ns_printf(pio, "Query for %s: ", dns_entry->dns_names);
         if(dns_entry->rcode == DNSRC_UNSET)
            ns_printf(pio, "no reply\n");
         else
            ns_printf(pio, "bad reply code was %d.\n", dns_entry->rcode);
         continue;
      }
      if(dns_entry->he.h_name)
      {
         ns_printf(pio, "name: %s, ", dns_entry->he.h_name);
      }
      p= &dns_entry->he;
      if(*p->h_aliases)
      {
         ns_printf(pio, "\n  Aliases: ");
         for(cpp = p->h_aliases; *cpp; cpp++)
         {
            ns_printf(pio, "%s, ", *cpp);
         }
      }
#ifdef IP_V4
      if(dns_entry->ipaddrs)
      {
         ns_printf(pio, "\n  IPv4 addrs: ");
         for(i = 0; i < dns_entry->ipaddrs; i++)
         {
            ns_printf(pio, "%u.%u.%u.%u, ", 
               PUSH_IPADDR(dns_entry->ipaddr_list[i]) );
         }
      }
#endif   /* IP_V4 */

#ifdef IP_V6
      if(dns_entry->ip6addrs)
      {
         ns_printf(pio, "\n  IPv6 addrs: ");
         for(i = 0; i < dns_entry->ip6addrs; i++)
         {
            char ip6buf[40];     /* tmp buffer for ipv6 address text */
            ns_printf(pio, "%s ", 
               print_ip6((struct in6_addr *)(&dns_entry->ip6_list[i][0]), ip6buf) );
         }
      }
#endif   /* IP_V6 */

      ns_printf(pio, "\n  Age (in seconds): %lu, Expires in %lu seconds", 
         (cticks - dns_entry->send_time)/TPS,
         (dns_entry->expire_time - cticks)/TPS );

      ns_printf(pio, "\n");
   }
   return 0;
}


#endif   /* NET_STATS */

#endif   /* DNS_CLIENT */
/* end of file dnsclnt.c */


