/*
 * FILENAME: parseip.c
 *
 * Copyright 1997- 2006 By InterNiche Technologies Inc. All rights reserved
 *
 * Portions Copyright 1986 by Carnegie Mellon
 * Portions Copyright 1984 by the Massachusetts Institute of Technology
 *
 * Copyright (c) 1982, 1986, 1988 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation and other 
 * materials related to such distribution and use acknowledge that 
 * the software was developed by the University of California, Berkeley.
 * The name of the University may not be used to endorse or promote 
 * products derived from this software without specific prior written 
 * permission. THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 *
 * Rights, responsibilities and use of this software are controlled by
 * the agreement found in the "LICENSE.H" file distributed with this
 * source code.  "LICENSE.H" may not be removed from this distribution,
 * modified, enhanced nor references to it omitted.
 *
 * parse_ipad() function and wrapper function inet_addr() for IP applications
 *
 * MODULE: MISCLIB
 *
 * ROUTINES: parse_ipad(), inet_addr(), print46_addr(), 
 * ROUTINES: inet46_addr(), inet_setport(), 
 *
 * PORTABLE: yes
 */

#include "license.h"
#include "ipport.h"
#include "in_utils.h"
#ifndef MINI_IP
#include "q.h"
#include "netbuf.h"
#include "net.h"
#include "nvparms.h"
#include "ip.h"
#include "nptcp.h"
#include "socket.h"
#ifdef IP_V6
#include "socket6.h"
#endif
#endif /* !MINI_IP */


/* FUNCTION: parse_ipad()
 *
 * parse_ipad(long_out, string_in); Looks for an IP address in 
 * string_in buffer, makes an IP address (in big endian) in long_out. 
 * returns NULL if OK, else returns a pointer to a string describing 
 * problem. 
 *
 * 
 * PARAM1: ip_addr * ipout
 * PARAM2: unsigned * sbits
 * PARAM3: char * stringin
 *
 * RETURNS: 
 */

char *   
parse_ipad(ip_addr * ipout,   /* pointer to IP address to set */
   unsigned *  sbits,      /* default subnet bit number */
   char *   stringin)      /* buffer with ascii to parse */
{
   char *   cp;
   int   dots  =  0; /* periods imbedded in input string */
   int   number;
   union   
   {
      u_char   c[4];
      u_long   l;
   } retval;
   char *   toobig   = "each number must be less than 255";

   cp = stringin;
   while (*cp)
   {
      if (*cp > '9' || *cp < '.' || *cp == '/')
         return("all chars must be digits (0-9) or dots (.)");
      if (*cp == '.')dots++;
         cp++;
   }

   if ( dots < 1 || dots > 3 )
      return("string must contain 1 - 3 dots (.)");

   cp = stringin;
   if ((number = atoi(cp)) > 255)   /* set net number */
      return(toobig);

   retval.c[0] = (u_char)number;

   while (*cp != '.')cp++; /* find dot (end of number) */
      cp++;             /* point past dot */

   if (dots == 1 || dots == 2) retval.c[1] = 0;
      else
   {
      number = atoi(cp);
      while (*cp != '.')cp++; /* find dot (end of number) */
         cp++;             /* point past dot */
      if (number > 255) return(toobig);
         retval.c[1] = (u_char)number;
   }

   if (dots == 1) retval.c[2] = 0;
      else
   {
      number = atoi(cp);
      while (*cp != '.')cp++; /* find dot (end of number) */
         cp++;             /* point past dot */
      if (number > 255) return(toobig);
         retval.c[2] = (u_char)number;
   }

   if ((number = atoi(cp)) > 255)
      return(toobig);
   retval.c[3] = (u_char)number;

   if (retval.c[0] < 128) *sbits = 8;
      else if(retval.c[0] < 192) *sbits = 16;
      else *sbits = 24;

      *ipout = retval.l;      /* everything went OK, return number */
   return(NULL);        /* return OK code (no error string) */
}


/* FUNCTION: inet_addr()
 * 
 * PARAM1: char * str
 *
 * RETURNS: u_long ipaddr
 */

u_long 
inet_addr(char * str)
{
   u_long   ipaddr;
   unsigned bits ;    
   static char nearBuf[30];

   /* Copy string into local ram area (portability code) */
   strcpy((char *)nearBuf,str);

   if ( parse_ipad(&ipaddr,&bits,nearBuf) == NULL )
   {
      return ipaddr ;
   }
   else
   {
      return (u_long)NULL ;
   }
}

#ifndef MINI_IP

/* FUNCTION: print46_addr()
 * 
 * Print the address into a string and return a pointer to it.
 *
 * PARAM1: struct v3_host *host
 *
 * RETURNS: Pointer to string with the address in readable format.
 */

char * print46_addr(struct sockaddr *ipaddr)
{
   if (ipaddr->sa_family == AF_INET)
   {
      struct sockaddr_in * addr = (struct sockaddr_in *)ipaddr;
      return print_ipad(addr->sin_addr.s_addr);
   }
#ifdef IP_V6
   else
   {
      struct sockaddr_in6 * addr = (struct sockaddr_in6 *)ipaddr;
      static char namebuf[46];  /* max len of IPv6 addr */
      return (char *)inet_ntop(AF_INET6,&addr->sin6_addr, namebuf, sizeof(namebuf));
   }
#endif

   return NULL;
}

/* FUNCTION: inet46_addr()
 * 
 * PARAM1: char* str
 * PARAM2: struct sockaddr *addr
 *
 * inet_pton() is available only if IP_V6 is defined.
 * Also, it expects us to pass the family !!
 * So, it can't be used for more generic needs.
 *
 * inet46_addr() 
 * 1. Detects the address family 
 * 2. Appropriately parses the IPv4/IPv6 address.
 * 3. If IP_V6 is disabled, it doesn't parse IPv6 adddresses.
 *
 * RETURNS: Parsed IP address (IPv4/IPv6) in "address".
 * Returns 0 on success, else error.
 *
 */

int 
inet46_addr(char * str, struct sockaddr *address)
{
   /* Read the IPv4/IPv6 address */
   address->sa_family = AF_INET; /* assume IPv4 address by default */

   if ((str[1] == '.') || (str[2] == '.') || (str[3] == '.'))
   {
      struct sockaddr_in *addr = (struct sockaddr_in *)address;
      addr->sin_addr.s_addr = inet_addr(str);
      addr->sin_family = AF_INET;
   }
#ifdef IP_V6
   else
   {
      struct sockaddr_in6 *addr = (struct sockaddr_in6 *)address;
      inet_pton(AF_INET6, str, &addr->sin6_addr);
      addr->sin6_family = AF_INET6;
   }
#endif

   return 0;
}


/* FUNCTION: inet_setport()
 *
 * Utility function to set the destination port in the sockaddr structure.
 * Sets the port for both IPv4 and IPv6 addresses.
 *
 * PARAM1: struct sockaddr *addr
 * PARAM2: int port
 *
 * RETURNS: -
 */

void inet_setport(struct sockaddr *addr,int port) 
{
   if (addr->sa_family == AF_INET)
   {
      struct sockaddr_in *si = (struct sockaddr_in *)addr;
      si->sin_port = (port);  
   }
#ifdef IP_V6
   else
   {
      /* addr->sa_family = AF_INET6 */
      struct sockaddr_in6 *si = (struct sockaddr_in6 *)addr;
      si->sin6_port = (port);
   }
#endif

}

/*
 *Function Name:convert_ip()
 *
 *Parameters:
 *
 *Description:
 *
 *Returns:
 *
 */

unsigned long convert_ip(const char *p)
{
   const char *cp = p;
   unsigned long dw;
   unsigned char *lpb = (unsigned char *) &dw;
   int n = 0;
   unsigned short v = 0;
   dw = 0;
   while(*cp)
   {
      if( *cp == '.')
      {
         lpb[n] = (unsigned char) v;
         v = 0;
         n++;
         if(n > 3)
         {
            return dw;
         }
      }
      else if(((*cp >= '0') && (*cp <= '9')))
      {
         v = (v * 10) + (*cp - '0');
      }
      cp++;
   }
   lpb[n] = (unsigned char) v;
   return dw;
}  /* convert_ip() */

#endif /* !MINI_IP */
