/*
; timer.c
;
; Demonstracny program pre pouzitie casovaca a vystupu na port
;
; Opis:
;
;       Program ma vysielat na pin P3.4 (pripojene LED na EVB doske) obdlznikovy
;       signal s periodou 1ms. Perioda je urcena casovacom T0.
;       Casovac pracuje v rezime 1 t.j. ako 16 bitovy.
;
;       Nastavenie T0:
;
;       TMOD: GATE  C/T  M1  M0  GATE  C/T  M1  M0
;              X    X    X   X    0     0   0   1       mod 1 casovaca T0
;
;       TCON:  TF1  TR1  TF0  TR0  IE1  IT1   IE0  IT0
;               x    x   x    0/1   x    x     x    x   zapnutie/vypnutie casovaca T0
;
;       IE:     EA  ET2  ES  ET1  EX1  ET0  EX0
;               1    x   x    x    x    1    x          povolenie prerusenia od T0
;
;
;       V preruseni sa zmeni (ivertuje) stav pinu p3.4 .
;
;       Zakladna perioda casovaca je 500us, co pri frekvenii osc. 12MHz
;       vychadza na predvolbu T0: 65536-500=65036D
;
;       timer0 - podprogram obsluhy prerusenia casovaca 0
;
;       main  - hlavny program, ktory inicializuje casovac,
;           po inicializacii v nekonecnej slucke realizuje
;
;       Pristup k premennej pocitadlo je realizovany z hlavneho 
;       programu a zaroven aj v obsluhe prerusenia timer0()
;       Uvedeny (tzv. asynchrony pristup)  je osetreny  oznacenim 
;       premennej ako volatile:
;
;                 volatile int pocitadlo; 
;
;
*/

#include <reg51.h>

#define PERIOD  (65536-500)     // casovac bude pocitat od hodnoty PERIOD do 0xffff = 65536
#define STOP_CASOVACA (100)     // zastavenie pocitadla v hlavnom programe

sbit LED = P3 ^ 4;              // definovanie symbolickho mena pre vyvod portu P3.4

volatile int pocitadlo;         // pristup k premennej je aj v preruseni aj
                                // v hlavnom programe (stacil by aj typ char,
                                // resp. unsigned char)

unsigned long cnt;              // "demo" pocitadlo v main programe

void timer0 (void) interrupt 1  // obsluha prerusenia casovaca T0 (vektor na adrese 0x000B)
{
  TR0=0;                        // zastavenie casovaca T0
  TL0=PERIOD%256;               // definovanie periody (spodnych 8 bitov cisla PERIOD)
  TH0=PERIOD/256;               // definovanie periody (hornych 8 bitov cisla PERIOD)
  TR0=1;                        // spustenie casovaca T0
  LED = ~LED;                   // negovanie signalu pre LEDku
  pocitadlo++;                  // inkrementacia (softveroveho) pocitadla
}


//----------------------------------------------
// HLAVNY PROGRAM
//----------------------------------------------

void main (void)
{
/* inicializacia casovaca T0 a povolenie prislusneho zdroja prerusenia */
  TMOD=0x01;                    // mod 1 casovaca T0
  IE=0x82;                      // povolenie prerusenia od T0 a golbalneho prerusenia
  TL0=PERIOD%256;               // definovanie periody prerusenia
  TH0=PERIOD/256;
  TR0=1;                        // spustenie casovaca

  pocitadlo = 0;                // inicializacia pocitadiel
  cnt = 0;

  while(1) {                    // casovac T0 je "obsluhovany" na pozadi

     if( pocitadlo == STOP_CASOVACA) {
              TR0=0;
              while( 1 )        // ukoncenie po zastaveni
                  ;
     }
     cnt++;                     // mozno sledovat aktivitu v simulatore

 }
}
