//*****************************************************************************
// Copyright (c) 2006, Freescale Semiconductor
// Freescale Confidential Proprietary
// For use on Freescale products only.
//
// File name :   	freescale_http.c
// Project name: 	emg_HTTP web server for Coldfire
// Author:			Eric Gregori   (847) 651 - 1971
//		   			eric.gregori@freescale.com
//
// Description : 	HTTP/Web server with static anf dynamic FFS support.
//					Dynamic HTML is supported via tokens in the HTML.
//					A key is used to support run time ( dynamic ) web page uploads.
//					Without the correct key, the download is rejected.
//					The web server supports multiple sessions as defined by EMG_HTTP_SESSION.
//					The web server supports 'GET', 'POST','EMG'.
//					EMG is a unique non-standard HTTP command to request a upload.
//					Uploading is done using the same server that serves up the web pages.
//					Long filenames are supported, along with subdirectories.
//
//*****************************************************************************
#include "ipport.h"
#include "tcpapp.h"
#include "msring.h"
#include "freescale_http_server.h"


//*****************************************************************************
// External Declarations
//*****************************************************************************
extern EMG_HTTP_SESSION 	freescale_http_sessions[];
extern const unsigned char 	*emg_static_ffs_ptrs[];
extern const unsigned short emg_static_ffs_len[];
extern const unsigned char 	emg_static_ffs_nof;
extern const unsigned long 	emg_static_ffs_type[];
extern const char 			*emg_static_ffs_filenames[];
extern void 				replace_with_sensor_data( char *body_buff, unsigned long length );
extern void 				collect_sensor_data( void );
extern volatile void 		flash_write( unsigned long *address, unsigned long data );
extern int					flash_ffs_lockout;
extern void					form_led_function( char *);
extern void					form_serial_function( char *);


//*****************************************************************************
// Local declarations
//*****************************************************************************
void emg_http_process( int i );
int  emg_http_connection( M_SOCK so );
void emg_http_delete( void );
void emg_http_remove( M_SOCK so );
void emg_http_read_header( int i, char *buffer );
void emg_process_header( int i, char *filename, int bytesread );
void emg_http_send_file( int session );
void emg_http_upload_file( int session, unsigned long *buffer );
void emg_http_erase_flash( int session, unsigned long *buffer );


//*****************************************************************************
// Ram used for sensor variable replacement
//*****************************************************************************
char token_replacement_ram[MAX_DYNAMIC_STRING_SIZE];


//*****************************************************************************
// Global Variables
//*****************************************************************************
unsigned char		emg_quick_fix_for_show;


//*****************************************************************************
// Constant table defines acceptable HTTP commands
//*****************************************************************************
const HEADER_STRUCTURE header_table[] =
{
										{"GET",TYPE_GET},
										{"EMG",TYPE_UPLOAD},
										{"POST",TYPE_POST} 
};


//*****************************************************************************
// Upload password/key
//*****************************************************************************
const char key_string[] = "joshua";


//*****************************************************************************
// Keep Alive string for persistent connection
//*****************************************************************************
const char keep_alive_string[] = "Keep-Alive";


//*****************************************************************************
// Form Strings
//*****************************************************************************
const FORM_STRUCTURE forms[] = 
								{
									{"led", 	form_led_function},		
									{"serial",	form_serial_function},
									{"",		form_led_function}
								};

//*****************************************************************************
// File Not Found Web Page
//*****************************************************************************
const char page_not_found[] = "HTTP/1.0 200 OK\r\n\
Last-modified: Wed, 15 Mar 2006 10:42:00 GMT\r\n\
Server: EMG_HTTP 1.0.0\nContent-type: text/html\r\n\
\r\n\
<HEAD>\
<TITLE>Dynamic HTTP Server by Eric Gregori</TITLE></HEAD>\
<BODY>\
<H2>HTTP 1.0 404 Error. File Not Found</H2>\
The requested URL was not found on this server.\
<HR>\
<BR>\
<I>\r\n\
Freescale Semiconductor 2006\
<BR>\
Web Server for Embedded Applications\
</I>\r\n\
<BR>\
</BODY>";


//*****************************************************************************
// int freescale_http_connection( M_SOCK s )   Written By Eric Gregori
//		   										eric.gregori@freescale.com
//
// This function os called when a connection is requested by the client.
// The socket parameter is the communication socket we should use to
// communicate with the client.
//
// returns 0 on success, !0 on failure
//*****************************************************************************
int freescale_http_connection( M_SOCK s )
{
	int		i;
	int		found;
	
	// Walk the SESSION array looking for a invalid entry
	for( i=0, found=0; i<MAX_NUMBER_OF_SESSIONS; i++ )
	{
		if( freescale_http_sessions[i].valid != VALID_EMG_SESSION )
		{
			// Init session
			freescale_http_sessions[i].state  		= EMG_HTTP_STATE_WAIT_FOR_HEADER;
			freescale_http_sessions[i].socket 		= s;
			freescale_http_sessions[i].valid  		= VALID_EMG_SESSION;
			freescale_http_sessions[i].keep_alive  	= HTTP_KEEP_ALIVE_TIME;			
			found 									= 1;
#if HTTP_VERBOSE>2
			printf( "\nhttp %d started", i );
#endif			
			break;	
		}
	}
	
	return( found );
}


//*****************************************************************************
// freescale_http_delete() Written by Eric Gregori
//		   										eric.gregori@freescale.com
//
// Delete all sessions.  Only used for catastrophic failure and init
//*****************************************************************************
void freescale_http_delete( void )
{
	int		i;
	
	for( i=0; i<MAX_NUMBER_OF_SESSIONS; i++ )
		freescale_http_sessions[i].valid = INVALID_EMG_SESSION;
}


//*****************************************************************************
// Remove a session based on socket ID.
// Also, close the socket.
//
// Author: Eric Gregori  (847) 651 - 1971
//		   eric.gregori@freescale.com
//*****************************************************************************
void freescale_http_remove( M_SOCK so )
{
	int		i,j;

	for( i=0; i<MAX_NUMBER_OF_SESSIONS; i++ )
	{
		if( freescale_http_sessions[i].valid == VALID_EMG_SESSION )
		{
			if( freescale_http_sessions[i].socket == so )
			{
				freescale_http_sessions[i].valid = INVALID_EMG_SESSION;
				j = m_close( so );
				if( j>0 )
					printf( "\nFailure Closing Socket" );
#if HTTP_VERBOSE>2
				printf( "\nhttp closed %d", i );
#endif
			}
		}
	}	
}


//*****************************************************************************
// Process a single form assignment string.   vvvvvvvvv=ddddddddd0
//
// Author: Eric Gregori  (847) 651 - 1971
//		   eric.gregori@freescale.com
//*****************************************************************************
void process_form( char *form_string ) 
{
	int		i, index;
	
	for( i=0, index=0; (forms[index].form_name[0] != 0); i++ )
	{
		if( (form_string[i] == '=') && (forms[index].form_name[i] == 0) )
		{
			// Call function, passing the function a pointer to &form_string[i+1]
			(forms[index].func)(&form_string[i+1]);		
			break;
		}
			
		if( forms[index].form_name[i] != form_string[i] )
		{
			i=0;
			index++;
			continue;
		}
	}
}


//*****************************************************************************
// Pre-process the filename string to act on form entries
// Form entries are identified with a '?'
//
// Author: Eric Gregori  (847) 651 - 1971
//		   eric.gregori@freescale.com
//*****************************************************************************
void pre_process_filename( char *buffer )
{
	int		i, start, stop, j, k, index, next;
	
	// If no '?' is found, just return
	for( i=0, start=0; i<RECV_BUFFER_SIZE; i++ )
	{
		if( !buffer[i] )
			break;
	
		if( buffer[i] == '?' )
		{
			start = i;
			break;
		}
	}
	
	if( !start )
		return;
	
	// From start to buffer[i] == 0 is the full form string, it may contain multiple forms
	for( i=start, stop=0, start=0; i<RECV_BUFFER_SIZE; i++ )
	{
		if(buffer[i] == '?')
		{
			buffer[i] = 0;
		
			if( !start )
				start = i+1;
			else
			{
				stop = i;
				next = i+1;
			}
			
			continue;	
		}
		
		if( buffer[i] == 0 )
		{
			stop  = i;
			next  = 0;
		}
		
		if( stop )
		{			
			// Call Function
			process_form( &buffer[start]);
			
			if( !next )
				break; 
			else
			{
				start = 0;
				stop  = 0;
				i++;
			}
		}
	}
}


//*****************************************************************************
// Process the HTTP header.
//
// We currently support GET and EMG.
// GET is used to send a Web Page, EMG is used to upload runtime web pages.
//
// The default web page is the first web page in the file system.
// If a valid FFS is found in the dynamic region of flash, the dynamic
// region takes precendance over the static region.
//
// Long filenames are supported, and limited only by the FFS.
// A FFS revision number is supported for future FFS expansion/compatability.
// 
// For uploading, a key is used to protect the flash from non-authorized users.
// If the key is not valid, the upload is rejected.
//
// Author: Eric Gregori  (847) 651 - 1971
//		   eric.gregori@freescale.com
//*****************************************************************************
void freescale_process_header( int session, char *buffer, int bytesread )
{
	int							i, k, j, found;
	int							charcount, dump;
	unsigned long				size;
	unsigned char				ch;
	volatile unsigned long		*fat_file_sys;
	volatile unsigned char		*fat_file_names;
	HEADER_TYPES 				ht;
	
	
	ht = header_table[session].header_type;
	for( i=0, j=bytesread; j<RECV_BUFFER_SIZE; i++, j++ )
	{
		if( buffer[i] == ' ' )
		{
			buffer[i] = 0;
			break;
		}
	}

#if HTTP_VERBOSE>1	
	printf( "\nFile[%d] = %s", session, buffer );
#endif

	if( freescale_http_sessions[session].headertype == TYPE_GET )
	{
		collect_sensor_data( );
	
		fat_file_sys = (unsigned long *)FAT_FILE_BASE_ADDR;

		// Find the file
		if( buffer[0] == 0 )
		{
			if( (fat_file_sys[0] == (unsigned long)'EMG1' ) && !flash_ffs_lockout )
			{
				fat_file_sys = (unsigned long *)((unsigned long)fat_file_sys[3] + (unsigned long)FAT_FILE_BASE_ADDR);				

				// Index into descriptors, K is index
				// [0] = file pointer
				// [1] = file size
				// [2] = file type
				fat_file_sys = &fat_file_sys[0];
#if HTTP_VERBOSE>1
				printf( "\nLoading default File - First file in dynamic FFS" );
#endif	
				freescale_http_sessions[session].file_type		= fat_file_sys[2];
				freescale_http_sessions[session].file_size 		= fat_file_sys[1];
				freescale_http_sessions[session].file_pointer	= (void *)((unsigned long)fat_file_sys[0] + (unsigned long)FAT_FILE_BASE_ADDR);
				freescale_http_sessions[session].file_index 	= 0;
				freescale_http_sessions[session].state			= EMG_HTTP_STATE_SEND_FILE;			
			}
			else
			{
#if HTTP_VERBOSE>1
				printf( "\nLoading default File - First file in static FFS" );
#endif	
				// Default file = first file in memory
				freescale_http_sessions[session].file_size 		= emg_static_ffs_len[0];
				freescale_http_sessions[session].file_pointer	= (void *)emg_static_ffs_ptrs[0];
				freescale_http_sessions[session].file_index 	= 0;
				freescale_http_sessions[session].state			= EMG_HTTP_STATE_SEND_FILE;			
			}
		}
		else
		{
			pre_process_filename( buffer );
		
			found = 0;
			if( (fat_file_sys[0] == (unsigned long)'EMG1' ) && !flash_ffs_lockout )
			{
				// Use downloadable FFS
				// EMG1 ffs structures
				// typedef struct	{
				//						unsigned long	fatID;
				//						unsigned long	total_size_in_bytes;
				//						unsigned long	number_of_files;
				//						unsigned long	fat_pointer;
				//					}		FAT_HEADER;
				//
				// typedef struct	{
				//						unsigned long	file_pointer;
				//						unsigned long	file_size_in_bytes;
				//						unsigned long	file_type;
				//					}		FAT_FILE_DESCRIPTOR; 
				// 
				// After the header is a list of filenames
				fat_file_names = (unsigned char *)&fat_file_sys[4];
				
				// First char is 0				
				// buffer[0] = filaname
				for( k=0, j=0, found=0; ;k++ )
				{
					// Scan to start of string
					// 2 consecutive NULL equals end
					for( ; fat_file_names[j]; j++ )
					{};
					if( fat_file_names[j+1] == 0 )
						break;
					
					for( i=0, j++; buffer[i]; i++ )
					{
						if( fat_file_names[j+i] != buffer[i])
							break;
					}
					
					if( buffer[i] == 0 )
					{
						found = 1;
						break;
					}
				}
				
				// If found, K = index into FILE DESCRIPTORS
				// Set fat_file_sys to start of descriptors
				fat_file_sys = (unsigned long *)((unsigned long)fat_file_sys[3] + (unsigned long)FAT_FILE_BASE_ADDR);				

				// Index into descriptors, K is index
				// [0] = file pointer
				// [1] = file size
				// [2] = file type
				fat_file_sys = &fat_file_sys[k*3];
#if HTTP_VERBOSE>1
				printf( "\nFile Found - %d", k );
#endif	

				if( found )
				{
					freescale_http_sessions[session].file_type		= fat_file_sys[2];
					freescale_http_sessions[session].file_size 		= fat_file_sys[1];
					freescale_http_sessions[session].file_pointer	= (void *)((unsigned long)fat_file_sys[0] + (unsigned long)FAT_FILE_BASE_ADDR);
					freescale_http_sessions[session].file_index 	= 0;
					freescale_http_sessions[session].state			= EMG_HTTP_STATE_SEND_FILE;			
				}
				else
				{			
					freescale_http_sessions[session].file_type		= 0;
					freescale_http_sessions[session].file_size 		= sizeof(page_not_found);
					freescale_http_sessions[session].file_pointer	= (void *)page_not_found;
					freescale_http_sessions[session].file_index 	= 0;
					freescale_http_sessions[session].state			= EMG_HTTP_STATE_SEND_FILE;			
				}

			}

			if( !found )
			{
				// Use compiled FFS
				// Scan through the filenames, and find ours
				for( i=0, found=0; i<emg_static_ffs_nof; i++ )
				{
					for( j=0, k=0; buffer[j]; j++ )
					{
						if( buffer[j] == emg_static_ffs_filenames[i][k] )
							k++;		
					}
			
					if( emg_static_ffs_filenames[i][k] == 0 )
					{
						// Filename found
						found = 1;
#if HTTP_VERBOSE>1
						printf( "\nFile Found - %d", emg_static_ffs_len[i] );
#endif	
						break;
					}
				}
		
				if( found )
				{
					freescale_http_sessions[session].file_type		= (int)emg_static_ffs_type[i];
					freescale_http_sessions[session].file_size 		= emg_static_ffs_len[i];
					freescale_http_sessions[session].file_pointer	= (void *)emg_static_ffs_ptrs[i];
					freescale_http_sessions[session].file_index 	= 0;
					freescale_http_sessions[session].state			= EMG_HTTP_STATE_SEND_FILE;			
				}
				else
				{			
					freescale_http_sessions[session].file_type		= 0;
					freescale_http_sessions[session].file_size 		= sizeof(page_not_found);
					freescale_http_sessions[session].file_pointer	= (void *)page_not_found;
					freescale_http_sessions[session].file_index 	= 0;
					freescale_http_sessions[session].state			= EMG_HTTP_STATE_SEND_FILE;			
				}
			}
		}
	} // end of if TYPE_GET

	if( freescale_http_sessions[session].headertype == TYPE_UPLOAD )
	{		
		// Use filename as key.  Unless filename matches key, do not allow
		// upload
		for( j=0; buffer[j]; j++ )
		{
			if( buffer[j] != key_string[j] )
				break;
		}
		
		if( buffer[j] )
		{
			printf( "\nInvalid Key" );
			freescale_http_sessions[session].state = EMG_HTTP_STATE_CLOSE;
			return;	
		}
		
		// Use file size to indicate the # of byte to download
		// After the filename, the next 4 bytes are the size
		i++;
		// i now points to the MSB byte
		ch = buffer[i];
		size = ch;
		size <<= 8;
		ch = buffer[i+1];
		size += ch;
		size <<= 8;
		ch = buffer[i+2];
		size += ch;
		size <<= 8;
		ch = buffer[i+3];
		size += ch;
		 
		freescale_http_sessions[session].file_size  = size;
		freescale_http_sessions[session].file_index = FLASH_START_ADDRESS;
		freescale_http_sessions[session].state 		= EMG_HTTP_STATE_ERASE_FLASH;	
	}
	else
	{
		// Pitch the rest of the header.
		// The header is complete when a blank line 0D0A0D0A is sent
		dump 	= RECV_BUFFER_SIZE-bytesread;
		k    	= 0;
		found	= 0;
		do
		{
			for( charcount=0;i<dump; i++ )
			{
#if HTTP_VERBOSE>1
				printf( "%c", buffer[i] );
#endif			
				if( (buffer[i]==0x0D) || (buffer[i]==0x0A) )
					charcount++;
				else
					charcount = 0;
		
				if( charcount == 4 )
					return;
				
				if( buffer[i] == keep_alive_string[k] )
				{
					k++;
					if( keep_alive_string[k] == 0 )
					{
						found = 1;
#if HTTP_VERBOSE>0
						printf( "\nKeep-Alive flag set" );
#endif
						k = 0;
					}					
				}
				else
					k = 0;
			}
			
			dump = m_recv( freescale_http_sessions[session].socket, (char *)buffer, RECV_BUFFER_SIZE );
			i = 0;
		}		
		while( dump > 0	);		

		if( found )
			freescale_http_sessions[session].keep_alive = HTTP_KEEP_ALIVE_TIME;
		else
			freescale_http_sessions[session].keep_alive = 0;
	}
}


//*****************************************************************************
// HTTP read header.
//
// The HTTP header must be smaller then a packet of MTU size.
// Process header up to filename.
//
// Author: Eric Gregori  (847) 651 - 1971
//		   eric.gregori@freescale.com
//*****************************************************************************
void freescale_http_read_header( int session, char *buffer )
{
	HEADER_TYPES	i;
	int				length, j, k;


	if( freescale_http_sessions[session].keep_alive  )
		freescale_http_sessions[session].keep_alive--;

	// PARAM1: M_SOCK socket,
	// PARAM2: char * buffer
	// PARAM3: unsigned length
	//
	// RETURNS: number of bytes actually read, or -1 if error. 
	// int m_recv(M_SOCK so, char * buf, unsigned buflen)
	length = m_recv( freescale_http_sessions[session].socket, (char *)buffer, RECV_BUFFER_SIZE );
	
	if( length < 0 )
	{
		if( !freescale_http_sessions[session].keep_alive )
			freescale_http_sessions[session].state = EMG_HTTP_STATE_CLOSE;	
		return;
	}
	
	// Scan through the buffer looking for 
	for( i=NO_HEADER_FOUND; header_table[i].header_string[0]; i++ )
	{
		for( j=0, k=0; j<length; j++ )
		{
			if( buffer[j] == header_table[i].header_string[k] )
				k++;
		
			if( header_table[i].header_string[k] == 0 )
			{
#if HTTP_VERBOSE>3
				printf( "\n header[%d] = %s", session, &header_table[i].header_string[0] );
#endif				
				freescale_http_sessions[session].headertype = header_table[i].header_type;

				// Scan to /
				for( ; j<length; j++ )
				{
					if( buffer[j] == '/' )
					{
						freescale_process_header( session , &buffer[++j], j );						
						return;
					}
				}
				
				// If we get here, we did not find the '/'
				// Just send the default file
				buffer[0] = ' ';
				freescale_process_header( session , buffer, 0 );						
				return;								
			}
		}
	}
	
	// If we get here, then 
	//					We do not support the method 'GET', 'POST', 'EMG'
	// The only thing we can do is close the socket
	freescale_http_sessions[session].state = EMG_HTTP_STATE_CLOSE;	
}


//*****************************************************************************
// This function sends the file in chunks of MAX_BYTES_TO_SEND.
// It is executed by the state machine.
// Currently a sleep of 10ms is required after every packet is sent, this
// is due to a bug in the Interniche TCP/IP stack that causes the stack
// to hang is it runs out of packet buffers.
// 
// Use a sliding window to process dynamic HTML tokens.
// Normally packet processing is done using Zero-Copy, only when a section
// contains a token, is it copied to temporary RAM. 
//
// Author: Eric Gregori  (847) 651 - 1971
//		   eric.gregori@freescale.com
//*****************************************************************************
void freescale_http_send_file( int session )
{
	unsigned long		length, i, token_end;
	int					bytes_sent;
	char				*data;

	// Send MAX_BYTES_TO_SEND number of bytes
	length = freescale_http_sessions[session].file_size - freescale_http_sessions[session].file_index;
	if( length > MAX_BYTES_TO_SEND )
		length = MAX_BYTES_TO_SEND;
	
	data = freescale_http_sessions[session].file_pointer;
	data = &data[freescale_http_sessions[session].file_index];
		
	if( length )
	{		
		// if filetype == html, and
		// if data contains any dynamic content, process it
		// Scan through looking for token.
		// If token is first byte, then pass MAX_DYNAMIC_STRING_SIZE
		// bytes to the token processor.
		// else, if token is later in memory, send up to token
		if( (freescale_http_sessions[session].file_type == FILE_TYPE_DYANMIC1 ) ||
		    (freescale_http_sessions[session].file_type == FILE_TYPE_DYANMIC2) )
		{
			// Scan through length in data looking for token
			for( i=0; i<length; i++ )
			{
				if((data[i] == DYNAMIC_REPLACE_TOKEN) ||
				   (data[i] == DYNAMIC_COMPARE_TOKEN) )
				{
					break;
				}
			}
			
			if( !i )
			{
				token_end = i;
				
				// token found at start of data
				// Copy data to our token_replacement_buffer
				// We only want to copy complete token strings
				// A token string ends with a DYNAMIC_EOS_TOKEN
				for( i=0; i<length; i++ )
				{
					if( i<(MAX_DYNAMIC_STRING_SIZE-1) )
					{
						token_replacement_ram[i] = data[i];
						if( data[i] == DYNAMIC_EOS_TOKEN )
						{
							token_end = i+1;	
						}						
					}
					else
						break;
				}
				
				data 	= token_replacement_ram;
				length 	= token_end;
				
				replace_with_sensor_data( data, length );
			}
			else
			{
				// no token found, or token further along
				// just send up to token
				length = i;
			}			
		}
				
		bytes_sent = m_send( freescale_http_sessions[session].socket, data, length );
#if HTTP_VERBOSE>4
		printf( "\nses %d, bytes %d ", session, bytes_sent ); 
#endif

		tk_yield( );

		if( bytes_sent >= 0)
		{
			freescale_http_sessions[session].file_index += bytes_sent;
			emg_quick_fix_for_show = 0;
		}
		else
		{
			if( freescale_http_sessions[session].socket->error == EWOULDBLOCK )
			{
				emg_quick_fix_for_show++;
				if( emg_quick_fix_for_show > 80 )
				{
					freescale_http_sessions[session].state = EMG_HTTP_STATE_CLOSE;					
					printf( "\n\nStack failed due to blocking" );
				}
			}
			
			// If socket reported a error, or would block,
			// Give some extra time to sleep.
			tk_sleep( 10 );
		}		
	}
	else	
	{
#if HTTP_VERBOSE>2
		printf( "\n%d bytes sent[%d]", 	freescale_http_sessions[session].file_index, session );
#endif
		if( freescale_http_sessions[session].keep_alive  )
			freescale_http_sessions[session].state = EMG_HTTP_STATE_WAIT_FOR_HEADER;
		else
			freescale_http_sessions[session].state = EMG_HTTP_STATE_CLOSE;	
	}	
}


//*****************************************************************************
// Process a UPLOAD file.
// As packets are received they are sent to the flash writer.
// Packets are processed on a 32bit boundary.
// At the completion of a upload, a "complete" packet is sent for handshaking.
//
// Author: Eric Gregori  (847) 651 - 1971
//		   eric.gregori@freescale.com
//*****************************************************************************
void freescale_http_upload_file( int session, unsigned long *buffer )
{
	int				length, j;

	// PARAM1: M_SOCK socket,
	// PARAM2: char * buffer
	// PARAM3: unsigned length
	//
	// RETURNS: number of bytes actually read, or -1 if error. 
	// int m_recv(M_SOCK so, char * buf, unsigned buflen)
	length = m_recv( freescale_http_sessions[session].socket, (char *)buffer, RECV_BUFFER_SIZE );
	
	if( length < 0 )
		return;
	
#if HTTP_VERBOSE>1
	printf( "\nData size: 0x%x", length );
#endif
		
	// length is in bytes, we need to write longs
	for( j=0;j<(length/4); j++ )
	{
		flash_write( (unsigned long *)freescale_http_sessions[session].file_index, buffer[j] );	
		freescale_http_sessions[session].file_index += 4;
		freescale_http_sessions[session].file_size  -= 4;
	}
	
#if HTTP_VERBOSE>1
	printf( "  Remaining: 0x%x", freescale_http_sessions[session].file_size );
#endif
		
	if( freescale_http_sessions[session].file_size < 4 )
	{
		flash_write( (unsigned long *)freescale_http_sessions[session].file_index, buffer[j] );	
		freescale_http_sessions[session].file_size = 0;
	}	
	
	if( (freescale_http_sessions[session].file_size == 0) ||
		(freescale_http_sessions[session].file_size & 0xF0000000) )
	{
		flash_ffs_lockout = 0;
	
		sprintf( (char *)buffer, "Upload Complete" );

#if HTTP_VERBOSE>1
		printf( "\n%s", buffer );		
		printf( "\nHit enter to return to prompt" );
#endif		
		
		j = strlen((char *)buffer)+1;
		for( length = 0; length < j; )
		{
			length = m_send( freescale_http_sessions[session].socket, (char *)buffer, j );
			tk_sleep( 10 );
		}
	
		freescale_http_sessions[session].state = EMG_HTTP_STATE_CLOSE;		
	}
}


//*****************************************************************************
// Process erase flash state.
// Flash is erased in small chunks to avoid letting the TCP/IP connection idleing.
// Acks are sent after each chunk is erased.
//
// Author: Eric Gregori  (847) 651 - 1971
//		   eric.gregori@freescale.com
//*****************************************************************************
void freescale_http_erase_flash( int session, unsigned long *buffer )
{
	unsigned long	address;
	unsigned long	end_address;
	unsigned long	length, j;

	
	if( freescale_http_sessions[session].file_size  >
  		(FLASH_END_ADDRESS-FLASH_START_ADDRESS))
	{
		sprintf( (char *)buffer, "\n\nFile Too Big" );	
		printf( "%s", buffer );
		freescale_http_sessions[session].state = EMG_HTTP_STATE_CLOSE;
		j = strlen((char *)buffer)+1;
		for( length = 0; length < j; )
		{	
			length = m_send( freescale_http_sessions[session].socket, (char *)buffer, j );
			tk_sleep( 10 );
		}

		return;				
	}
		 	
	address 	= freescale_http_sessions[session].file_index;
	end_address = address + (PAGES_PER_SESSION*FLASH_PAGE_SIZE);
	if (end_address>=FLASH_END_ADDRESS)
		{
			end_address = FLASH_END_ADDRESS;
		}

	flash_ffs_lockout = 1;

	sprintf( (char *)buffer, "Erasing Flash 0x%x block address", address&0x000FFFFF );
	for( ; address < end_address; address += FLASH_PAGE_SIZE )
	{
		flash_page_erase( (unsigned long *)address, 0x00000000 );
	}


#if HTTP_VERBOSE>1
	printf( "\n%s", buffer );
#endif

	j = strlen((char *)buffer)+1;
	for( length = 0; length < j; )
	{
		length = m_send( freescale_http_sessions[session].socket, (char *)buffer, j );
		tk_sleep( 10 );
	}

	if( address < FLASH_END_ADDRESS )
		freescale_http_sessions[session].file_index = address;		
	else
	{
		freescale_http_sessions[session].file_index = FLASH_START_ADDRESS;
		freescale_http_sessions[session].state 		= EMG_HTTP_STATE_UPLOAD_FILE;	
		sprintf( (char *)buffer, "Erase Complete" );

#if HTTP_VERBOSE>1
		printf( "\n%s", buffer );
#endif		
		
		j = strlen((char *)buffer)+1;
		for( length = 0; length < j; )
		{
			length = m_send( freescale_http_sessions[session].socket, (char *)buffer, j );
			tk_sleep( 10 );
		}
	}
}


//*****************************************************************************
// The HTTP server state machine.
// We allocate a RAM buffer here, and pass it down to conserver RAM.
// The buffer is long aligned to facilitate uploading, do not change.
// 
// Author: Eric Gregori  (847) 651 - 1971
//		   eric.gregori@freescale.com
//*****************************************************************************
void freescale_http_process( int session )
{
	// Pass buffer to preserve RAM
	unsigned long	buffer[RECV_BUFFER_SIZE/4];


	if( freescale_http_sessions[session].valid == VALID_EMG_SESSION )
	{
		switch( freescale_http_sessions[session].state )
		{
			case EMG_HTTP_STATE_WAIT_FOR_HEADER:
				freescale_http_read_header( session, (char *)buffer );
				break;
				
			case EMG_HTTP_STATE_SEND_FILE:
				freescale_http_send_file( session );
				break;
				
			case EMG_HTTP_STATE_CLOSE:
				freescale_http_remove( freescale_http_sessions[session].socket );
				break;
				
			case EMG_HTTP_STATE_UPLOAD_FILE:
				freescale_http_upload_file( session, buffer );
				break;
				
			case EMG_HTTP_STATE_ERASE_FLASH:
				freescale_http_erase_flash( session, buffer );
				break;
			
			default:
				// This cannot be good
				dprintf( "\nIllegal HTTP state" );
				freescale_http_sessions[session].valid = INVALID_EMG_SESSION;
				break;	
		}
	}	
}


//*****************************************************************************
// Dump the HTTP session array
// 
// Author: Eric Gregori  (847) 651 - 1971
//		   eric.gregori@freescale.com
//*****************************************************************************
int	emg_http_sessions( void *pio  )
{
	unsigned long	i, fp, so; 
	

	ns_printf(pio, "\n\nHTTP sessions array Dump" );
	
	ns_printf( pio, "\n\nSTATE     VALID     KEEP_ALIVE     FILE_POINTER     SOCKET" );
	
	for( i=0; i<MAX_NUMBER_OF_SESSIONS; i++ )
	{
		switch( freescale_http_sessions[i].state )
		{
			case EMG_HTTP_STATE_WAIT_FOR_HEADER:
				ns_printf( pio, "\nWait for header     " );
				break;
				
			case EMG_HTTP_STATE_SEND_FILE:
				ns_printf( pio, "\nSend     " );
				break;
				
			case EMG_HTTP_STATE_CLOSE:
				ns_printf( pio, "\nClose     " );
				break;
				
			case EMG_HTTP_STATE_UPLOAD_FILE:
				ns_printf( pio, "\nUpload     " );
				break;
				
			case EMG_HTTP_STATE_ERASE_FLASH:
				ns_printf( pio, "\nErase     " );
				break;
			
			default:
				// This cannot be good
				ns_printf( pio, "\nInvalid     " );
				break;	
		}
	
		if( freescale_http_sessions[i].valid )
			ns_printf( pio, "Valid     " );
		else
			ns_printf( pio, "Not Valid     " );
		
		ns_printf( pio, "%d     ", freescale_http_sessions[i].keep_alive );
		
		fp = (unsigned long)freescale_http_sessions[i].file_pointer;
		so = (unsigned long)freescale_http_sessions[i].socket;
			
		ns_printf( pio, "0x%x     ", fp );
		ns_printf( pio, "0x%x     ", so );
	}
	
	ns_printf( pio, "\n\n" );
	
	return(0);
}

