/*
 * FILENAME: memdev.c
 *
 * Copyright 1997- 2006 By InterNiche Technologies Inc. All rights reserved
 *
 * Portions Copyright 1986 by Carnegie Mellon
 * Portions Copyright 1984 by the Massachusetts Institute of Technology
 *
 * Copyright (c) 1982, 1986, 1988 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation and other 
 * materials related to such distribution and use acknowledge that 
 * the software was developed by the University of California, Berkeley.
 * The name of the University may not be used to endorse or promote 
 * products derived from this software without specific prior written 
 * permission. THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 *
 * Rights, responsibilities and use of this software are controlled by
 * the agreement found in the "LICENSE.H" file distributed with this
 * source code.  "LICENSE.H" may not be removed from this distribution,
 * modified, enhanced nor references to it omitted.
 *
 * Memory device for VFS on diskless target systems. 
 * Currently only supports ftp read of memory image, and read/ write 
 * to "null" device. 
 *
 *
 * MODULE: MISCLIB
 *
 * ROUTINES: init_memdev(), md_fopen(), md_fclose(), md_fread(), 
 * ROUTINES: md_fwrite(), md_fseek(), md_ftell(), md_fgetc(), md_unlink(), 
 *
 * PORTABLE: yes
 */


#include "license.h"
#include "ipport.h"

#ifdef MEMDEV_SIZE   /* whole file can be ifdeffed */

#ifndef VFS_FILES /* These devices require VFS */
#error MEMDEV devices require VFS system
#endif
#ifndef HT_EXTDEV /* ...also VFS external devices feature... */
#error MEMDEV devices require VFS external devices
#endif

#include "vfsfiles.h"

/* The memory device IO routines */
VFILE* md_fopen(char * name, char * mode);
void   md_fclose(VFILE * vfd);
int    md_fread(char * buf, unsigned size, unsigned items, VFILE * vfd);
int    md_fwrite(char * buf, unsigned size, unsigned items, VFILE * vfd);
int    md_fseek(VFILE * vfd, long offset, int mode);
long   md_ftell(VFILE * vfd);
int    md_fgetc(VFILE * vfd);
int    md_unlink(char*);

/* The memory device VFS call structure. This is registered with the 
 * VFS system at init time. Thereafter calls to the memory device 
 * file 
 */

struct vfroutines mdio  =
{
   NULL,       /* list link */
   md_fopen, 
   md_fclose, 
   md_fread, 
   md_fwrite, 
   md_fseek,
   md_ftell, 
   md_fgetc, 
   md_unlink, 
};



#ifndef MEMDEV_BASE
#error memdev needs a base address defined in ipport.h
#endif   /* not MEMDEV_BASE */


struct vfs_file   mdlist[] = 
{
   {  NULL,          /* list link */
      "mem2M",         /* name of file */
      0,             /* flags (compression, etc) */
      NULL,          /* name of data array */
      MEMDEV_SIZE,   /* length of original file data */
      MEMDEV_SIZE,   /* length of compressed file data */
      0,             /* block size */
#ifdef WEBPORT
      NULL,          /* SSI data routine */
      NULL,          /* CGI routine */
#ifdef SERVER_PUSH
      NULL,          /* push callback */
#endif
#endif
      &mdio,
   },
   {  &mdlist[0],    /* list link */
      "mem1M",         /* name of file */
      0,             /* flags (compression, etc) */
      NULL,          /* name of data array */
      MEMDEV_SIZE/2,   /* length of original file data */
      MEMDEV_SIZE/2,   /* length of compressed file data */
      0,             /* block size */
#ifdef WEBPORT
      NULL,          /* SSI data routine */
      NULL,          /* CGI routine */
#ifdef SERVER_PUSH
      NULL,          /* push callback */
#endif
#endif
      &mdio,
   },
   {  &mdlist[1],          /* list link */
      "mem512K",         /* name of file */
      0,             /* flags (compression, etc) */
      NULL,          /* name of data array */
      MEMDEV_SIZE/4,   /* length of original file data */
      MEMDEV_SIZE/4,   /* length of compressed file data */
      0,             /* block size */
#ifdef WEBPORT
      NULL,          /* SSI data routine */
      NULL,          /* CGI routine */
#ifdef SERVER_PUSH
      NULL,          /* push callback */
#endif
#endif
      &mdio,
   },
   {  &mdlist[2],    /* list link */
      "null",        /* name of file */
      VF_WRITE,      /* flags (compression, etc) */
      NULL,          /* name of data array */
      MEMDEV_SIZE,   /* length of original file data */
      MEMDEV_SIZE,   /* length of compressed file data */
      0,             /* block size */
#ifdef WEBPORT
      NULL,          /* SSI data routine */
      NULL,          /* CGI routine */
#ifdef SERVER_PUSH
      NULL,          /* push callback */
#endif
#endif
      &mdio,
   },
};

int mdlist_size = sizeof(mdlist)/sizeof(struct vfs_file);


/* FUNCTION: init_memdev()
 * 
 * PARAM1: 
 *
 * RETURNS: 
 */

int
init_memdev(void)
{
   /* add our IO pointer to master list */
   mdio.next = vfsystems;
   vfsystems = &mdio;

   /* add the memory device files to vfs list */
   mdlist[0].next = vfsfiles;
   vfsfiles = &mdlist[3];

   return 0;
}


/* FUNCTION: md_fopen()
 * 
 * PARAM1: char * name
 * PARAM2: char * mode
 *
 * RETURNS: 
 */

VFILE* 
md_fopen(char * name, char * mode)
{
   USE_ARG(mode);
   USE_ARG(name);
   return NULL;
}



/* FUNCTION: md_fclose()
 * 
 * PARAM1: VFILE * vfd
 *
 * RETURNS: 
 */

void   
md_fclose(VFILE * vfd)
{
   USE_ARG(vfd);
}



/* FUNCTION: md_fread()
 * 
 * PARAM1: char * buf
 * PARAM2: unsigned size
 * PARAM3: unsigned items
 * PARAM4: VFILE * vfd
 *
 * RETURNS: 
 */

int    
md_fread(char * buf, unsigned size, unsigned items, VFILE * vfd)
{
   u_long   bcount;     /* number of bytes put in caller's buffer */
   u_long   location;   /* current offset into file */
   unsigned long file_size = MEMDEV_SIZE;
   if(vfd && vfd->file)
   {
      file_size = vfd->file->real_size; 
/*      printf("md_fread(): name = %s, file_size = %lu\n", vfd->file->name, file_size);*/
   }
/*
   else
   {
      printf("md_fread(): No name: file_size = %lu\n", vfd->file->name, file_size);
   }*/
   location = (u_long)(vfd->cmploc - vfd->file->data);

   bcount = (items * (u_long)size);     /* number of bytes to transfer */

   /* if near end of memory, trim read count accordingly */
   if ((location + bcount) > file_size)
      bcount = ((u_long)file_size - location);

   /* trap bogus size items and end-of-x86 memory conditions */
   if((location >= file_size) ||
      (bcount  & 0xFFFF0000) ||
      (bcount == 0))
   {
      return 0;
   }

   if (vfd->file->name[0] == 'm')   /* memory device */
      MEMCPY(buf, vfd->cmploc + MEMDEV_BASE, (unsigned)bcount);

   /* else for null device use whatever garbage is in the buffer */

   /* before returning, adjust file position ptr for next read */
   vfd->cmploc += bcount;  /* adjust location */
   
   return ((int)bcount/size);
}



/* FUNCTION: md_fwrite()
 * 
 * PARAM1: char * buf
 * PARAM2: unsigned size
 * PARAM3: unsigned items
 * PARAM4: VFILE * vfd
 *
 * RETURNS: 
 */

int    
md_fwrite(char * buf, unsigned size, unsigned items, VFILE * vfd)
{
   if (vfd->file->name[0] == 'm')   /* memory device */
      return 0;   /* not writable device */

   vfd->cmploc += (items * size);   /* adjust location */

   USE_ARG(buf);     /* supress compiler warnings */

   return (items);
}



/* FUNCTION: md_fseek()
 * 
 * PARAM1: VFILE * vfd
 * PARAM2: long offset
 * PARAM3: int mode
 *
 * RETURNS: 
 */

int
md_fseek(VFILE * vfd, long offset, int mode)
{
   USE_ARG(vfd);     /* supress compiler warnings */
   USE_ARG(offset);
   USE_ARG(mode);
   return 0;
}



/* FUNCTION: md_ftell()
 * 
 * PARAM1: VFILE * vfd
 *
 * RETURNS: 
 */

long   
md_ftell(VFILE * vfd)
{
   USE_ARG(vfd);     /* supress compiler warnings */
   return MEMDEV_SIZE;
}



/* FUNCTION: md_fgetc()
 * 
 * PARAM1: VFILE * vfd
 *
 * RETURNS: 
 */

int    
md_fgetc(VFILE * vfd)
{
   unsigned location;   /* current offset infile */
   int   retval   =  0;

   location = vfd->cmploc - vfd->file->data;
   if (location >= vfd->file->real_size)     /* at end of file? */
      return EOF;

   if (vfd->file->name[0] == 'm')   /* memory device */
      retval = (int)(*vfd->cmploc) & 0x00FF ;
   /* else for null device, use whatever is in retval */

   vfd->cmploc++;    /* adjust location */
   return retval;
}



/* FUNCTION: md_unlink()
 * 
 * PARAM1: char * filename
 *
 * RETURNS: 
 */

int    
md_unlink(char * filename)
{
   USE_ARG(filename);     /* supress compiler warnings */
   return 0;
}

#endif   /* MEMDEV_SIZE - whole file can be ifdeffed out */


