/*
 * FILENAME: menulib.c
 *
 * Copyright 1997- 2006 By InterNiche Technologies Inc. All rights reserved
 *
 * Portions Copyright 1986 by Carnegie Mellon
 * Portions Copyright 1984 by the Massachusetts Institute of Technology
 *
 * Copyright (c) 1982, 1986, 1988 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation and other 
 * materials related to such distribution and use acknowledge that 
 * the software was developed by the University of California, Berkeley.
 * The name of the University may not be used to endorse or promote 
 * products derived from this software without specific prior written 
 * permission. THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 *
 * Rights, responsibilities and use of this software are controlled by
 * the agreement found in the "LICENSE.H" file distributed with this
 * source code.  "LICENSE.H" may not be removed from this distribution,
 * modified, enhanced nor references to it omitted.
 *
 * Misc utility routines for Interniche menuing system
 *
 * MODULE: MISCLIB
 *
 * ROUTINES: setdebug(), set_upc(), snmp_stat(), iface_stats(), 
 * ROUTINES: ifstats(), linkstats(), settrapsize(), if_configbyname()
 * ROUTINES: if_netbytext
 *
 * PORTABLE: yes
 */


#include "license.h"
#include "ipport.h"
#include "in_utils.h"

#include "q.h"
#include "netbuf.h"
#include "net.h"
#include "ip.h"
#include "icmp.h"
#include "ether.h"
#include "arp.h"

#include "menu.h"

#include "nvparms.h"

extern uint8 ephy_isr;
extern void iuart_set_baud( int dev, int baud );	

int ifstats(void * pio, struct net * ifp);
char * print_uptime(unsigned long timetick);

unsigned atoh(char *);

#ifdef IP_V6
#include "ip6.h"
#endif

/* FUNCTION: setdebug()
 *
 * setdebug() - toggle NDEBUG protocol stack debugging options
 * 
 * PARAM1: void * pio
 *
 * RETURNS: 0
 */

int
setdebug(void * pio)
{
#ifdef NPDEBUG
     char *   cp;

   /* get an optional parameter */
   cp = nextarg(((GEN_IO)pio)->inbuf);

   /* if no parameter was entered */
   if (!*cp)
   {
      if (NDEBUG)
      { NDEBUG &= UPCTRACE;
         ns_printf(pio,"IP stack debug tracing off\n");
      }
      else
      { NDEBUG |= INFOMSG|IPTRACE|PROTERR|TPTRACE;
         ns_printf(pio,"IP stack debug tracing on\n");
      }
   }
   else
   {
      /* '?' means display NDEBUG value, else set NDEBUG flag to parameter value */
      if (*cp != '?')
      {
#ifdef IN_MENUS
         NDEBUG = atoh(cp);
#endif
      }
      ns_printf(pio,"NDEBUG is now 0x%x\n",NDEBUG);
   }
#else
   ns_printf(pio,"not a DEBUG compile.\n");
#endif
   return 0;
}



/* FUNCTION: set_upc()
 *
 * toggle state of upcall variable
 *
 * PARAM1: void * pio
 *
 * RETURNS: 0
 */

int
set_upc(void * pio)
{
   if (NDEBUG & UPCTRACE)
      NDEBUG &= ~UPCTRACE;
   else
      NDEBUG |= UPCTRACE;

   ns_printf(pio,"Upcall debugging %sabled\n", 
    (NDEBUG&UPCTRACE)?"en":"dis");

   return 0;
}


/* FUNCTION: if_configbyname(NET ifp)
 *
 * Configure the passed net interface from a set of NV parameters with
 * same name. This allows the NV system to specify parameters for
 * dynamic interfaces, and also frees the NV file from having to
 * specifiy the nets in the order they come up.
 *
 *
 * PARAM1: NET to configure - should have ifp->name set up.
 *
 * RETURNS: Returns 0 if OK, -1 if name not found
 */

#ifdef   INCLUDE_NVPARMS   /* only if using NV parms */
int
if_configbyname(NET ifp)
{
   int   i;

   /* search the NV parameters "nets" array for for our name */
   for(i = 0; i < MAXNETS; i++)
   {
      if(strncmp(inet_nvparms.ifs[i].name, ifp->name, IF_NAMELEN - 1) == 0)
      {
         /* found a matching name, set our IP congif from NV entry */
         ifp->n_ipaddr = inet_nvparms.ifs[i].ipaddr;
         ifp->snmask = inet_nvparms.ifs[i].subnet;
         ifp->n_defgw = inet_nvparms.ifs[i].gateway;
         return 0;
      }
   }
   return -1;
}
#endif   /* INCLUDE_NVPARMS */

/* FUNCTION: if_netbytext()
 *
 * get a net pointer based on a user text string. The string may be a ones
 * based number index (eg "1", "2") or an iface name ("pp0"). Prints error
 * message to passed output device if text does not map to an interface.
 * 
 *
 * PARAM1: void * pio   - output stream for error reporting
 * PARAM2: char * iftext - text of interface.
 *
 * RETURNS: pointer to iface if OK, else NULL.
 */


NET
if_netbytext(void * pio, char * iftext)
{
   unsigned iface;
   NET      ifp;

   if((strlen(iftext) == 1) && (isdigit(*iftext)))
   {
      iface = (unsigned)(*iftext - '0') - 1;
      if((iface >= ifNumber) ||
         ((ifp = if_getbynum((int)iface)) == NULL))
      {
           ns_printf(pio,"interface number must be 1 to %d.\n", ifNumber);
           return NULL;
      }
      else
         return (ifp);
   }
   else     /* not a single digit, look up as name */
   {
      for(ifp = (NET)(netlist.q_head); ifp; ifp = ifp->n_next)
      {
         if(strcmp(iftext, ifp->name) == 0)
            return ifp;
      }
      if(!ifp)
           ns_printf(pio,"no interface named \"%s\".\n", iftext);
   }
   return NULL;
}


/* FUNCTION: iface_stats()
 * 
 * menu routine to get iface number and invoke stats
 *
 * PARAM1: void * pio - output stream
 *
 * RETURNS: 0
 */

int
iface_stats(void * pio)
{
//   char *   cp;
   struct net * ifp;

   ifp = (NET)(netlist.q_head);   /* default to first net */

#if 0  // We only support 1 interface
   cp = nextarg(((GEN_IO)pio)->inbuf);    /* get console arg */
   if(cp && *cp)     /* arg is present on command line */
   {
      ifp = if_netbytext(pio, cp);
      if(ifp == NULL)      /* error parsing iface name/number text? */
         return -1;
   }
#endif

   ifstats(pio, ifp);
   return 0;
}


//*****************************************************************************
// int SoftEthernetNegotiation( int seconds )   Written By Eric Gregori
//
// Work-around for bug in hardware autonegotiation.
// Attempt to connect at 100Mbps - Half Duplexe
// Wait for seconds
// Attempt to connect at 10Mbps - Half Duplexe
// 
// Returns 10, or 100 on success, 0 on failure
//*****************************************************************************
int SoftEthernetNegotiation( int seconds )
{
	uint16		reg0, reg1, tick;

	// Force ePHY to manual, 100mbps, Half Duplexe
	(void)fec_mii_read(0, 0, &reg0);
	reg0 |= 0x2000;								// 100Mbps
	reg0 &= ~0x0100;							// Half Duplexe
	reg0 &= ~0x1000;							// Manual Mode	
	(void)fec_mii_write( 0, 0, reg0 );
	(void)fec_mii_write( 0, 0, (reg0|0x0200) ); // Force re-negotiate 
	
	for( tick=0, ephy_isr=0; tick<100; tick++ )
	{
		tk_sleep( seconds*10 );
		(void)fec_mii_read(0, 1, &reg1);
		if( reg1 & 0x0004 )
			return(100);
	}
	
	// Force ePHY to manual, 10mbps, Half Duplexe
	(void)fec_mii_read(0, 0, &reg0);
	reg0 &= ~0x2000;							// 10Mbps
	reg0 &= ~0x0100;							// Half Duplexe
	reg0 &= ~0x1000;							// Manual Mode	
	(void)fec_mii_write( 0, 0, reg0 );
	(void)fec_mii_write( 0, 0, (reg0|0x0200) ); // Force re-negotiate 
	
	for( tick=0, ephy_isr=0; tick<100; tick++ )
	{
		tk_sleep( seconds*10 );
		(void)fec_mii_read(0, 1, &reg1);
		if( reg1 & 0x0004 )
			return(10);
	}
	
	return(0);
}


/* FUNCTION: ifstats()
 *
 * per-iface portion of iface_stats()
 *
 * PARAM1: void * pio   - output stream
 * PARAM2: int ifc      - interface to dump
 *
 * EMG - Overloaded function with autonegotiation configuration and soft negotiation
 *
 * RETURNS: 0
 */
int
ifstats(void * pio, struct net * ifp)
{
	uint16 		reg0, reg4, settings;
	uint8		datarate, duplexe, manual, autodone;
	char 		*cp;


	datarate = 0;
	duplexe  = 0;
	manual   = 0;
	autodone = 0;
	
	/* get an optional parameter */
	cp = nextarg(((GEN_IO)pio)->inbuf);

	if( *cp )
	{
		if( strcmp( cp, "soft") == 0 )
		{
			ns_printf(pio, "\nPerforming Soft Ethernet Negotiation" );
				
			if( SoftEthernetNegotiation( 1 ) )
				ns_printf(pio, "\nSuccessfully Negotiated" );	
			else
			{
				ns_printf(pio, "\nTry Manual Negotiation" );
			}
			
			ns_printf(pio,"\n\n" );

			return(0);
		}

		// We have a parameter
		if( strstr(cp, "10") )
			datarate = 10;
		
		if( strstr(cp, "100") )
			datarate = 100;
		
		if( strstr(cp, "manual") )
			manual = 'm';
		
		if( strstr(cp, "auto") )
			manual = 'a';
		
		cp = nextarg(((GEN_IO)pio)->inbuf);
		
		if( *cp )
		{
			if( strstr(cp, "half") )
				duplexe = 'h';
		
			if( strstr(cp, "full") )
				duplexe = 'f';			
		}		
	}
	
	
	if( datarate || duplexe || manual )
	{
		ns_printf(pio,"\nSetting ePHY to");
	
		(void)fec_mii_read(0, 0, &reg0);
		(void)fec_mii_read(0, 4, &reg4);
		if( datarate == 100 )
		{
			ns_printf(pio," - 100 Mbps");
			reg0 |= 0x2000;
		}
		else
		{
			reg0 &= ~0x2000;			
			ns_printf(pio," - 10 Mbps");
		}
		
		if( duplexe == 'f' )
		{
			ns_printf(pio," - Full Duplex");
			reg0 |= 0x0100;	
		}
		else
		{
			reg0 &= ~0x0100;	
			ns_printf(pio," - Half Duplex");
		}
		
		if( manual == 'm' )
		{
			ns_printf(pio," - manual");
			reg0 &= ~0x1000;	
		}

		if( manual == 'a' )
		{
			ns_printf(pio," - auto");
			reg0 |= 0x1000;	
		}
		
		if( (datarate == 100) && (duplexe == 'f') )
		{
			ns_printf(pio," - Advertise 100FD");		
			reg4 |= 0x0100;
		}
		else
		{
			ns_printf(pio," - do NOT Advertise 100FD");
			reg4 &= ~0x0100;
		}

		if( (datarate == 100) && (duplexe == 'h') )
		{
			ns_printf(pio," - Advertise 100HD");		
			reg4 |= 0x0080;
		}
		else
		{
			ns_printf(pio," - do NOT Advertise 100HD");
			reg4 &= ~0x0080;
		}

		if( (datarate == 10) && (duplexe == 'f') )
		{
			ns_printf(pio," - Advertise 10FD");		
			reg4 |= 0x0040;
		}
		else
		{
			ns_printf(pio," - do NOT Advertise 10FD");
			reg4 &= ~0x0040;
		}

		if( (datarate == 10) && (duplexe == 'h') )
		{
			ns_printf(pio," - Advertise 10HD");		
			reg4 |= 0x0020;
		}
		else
		{
			ns_printf(pio," - do NOT Advertise 10HD");
			reg4 &= ~0x0020;
		}
		
		(void)fec_mii_write( 0, 4, reg4 );
		(void)fec_mii_write( 0, 0, reg0 );
		(void)fec_mii_write( 0, 0, (reg0|0x0200) ); // Force re-negotiate 
		
		ns_printf(pio,"\n\n" );
		
		return 0;	
	}
	
	
   ns_printf(pio, "Interface %s - %s \n", 
      ifp->name, ifp->n_mib->ifDescr);
#ifdef IP_V4
   ns_printf(pio,"IPv4 address: %s, " , print_ipad(ifp->n_ipaddr));
   ns_printf(pio,"subnet mask: %s, ", print_ipad(ifp->snmask));
   ns_printf(pio,"gateway: %s\n"    , print_ipad(ifp->n_defgw));
#endif   /* IP_V4 */
#ifdef IP_V6
   if(ifp->n_flags & NF_IPV6)
   {
      int i;
      char ip6buf[46];     /* tmp buffer for ipv6 address text */

      for(i = 0; i < MAX_V6_ADDRS; i++)
      {
         if(ifp->v6addrs[i])
         {
            ns_printf(pio,"IPv6 %6s addr: %s", v6types[i], 
               print_ip6(&(ifp->v6addrs[i]->addr), ip6buf));
            ns_printf(pio," - %s\n", 
               ip6addrstates[ifp->v6addrs[i]->flags & IA_STATEMASK]);
         }
      }
   }
   else
      ns_printf(pio,"No IPv6 addresses\n");
#endif   /* IP_V6 */
   ns_printf(pio,"Status; Admin:%s Oper:%s for: %s\n", 
      ifp->n_mib->ifAdminStatus==1?"up":"down",
      ifp->n_mib->ifOperStatus==1?"up":"down",
      print_uptime(sysuptime() - (ifp->n_mib->ifLastChange)));
   ns_printf(pio,"rcvd: errors:%lu   dropped:%lu   station:%lu   bcast:%lu   bytes:%lu\n",
      ifp->n_mib->ifInErrors, ifp->n_mib->ifInDiscards,
      ifp->n_mib->ifInUcastPkts, ifp->n_mib->ifInNUcastPkts,
      ifp->n_mib->ifInOctets);
   ns_printf(pio,"sent: errors:%lu   dropped:%lu   station:%lu   bcast:%lu   bytes:%lu\n",
      ifp->n_mib->ifOutErrors, ifp->n_mib->ifOutDiscards,
      ifp->n_mib->ifOutUcastPkts, ifp->n_mib->ifOutNUcastPkts,
      ifp->n_mib->ifOutOctets);
   ns_printf(pio,"MAC address: ");
   hexdump(pio,ifp->n_mib->ifPhysAddress, 6);
   ns_printf(pio," \n");

#ifdef IP_MULTICAST
   /* Print any multicast addresses assign to this iface */
   if(ifp->mc_list)
   {
      struct in_multi * imp;

      ns_printf(pio, "   Multicast addresses: \n");
      for (imp = ifp->mc_list; imp; imp = imp->inm_next)
      {
#ifdef IP_V6
         if(imp->inm_addr == 0)  /* not v4 means it;s v6 */
         {
            char ip6buf[40];     /* tmp buffer for ipv6 address text */
            ns_printf(pio, "   %s\n", print_ip6(&(imp->ip6addr), ip6buf));
            continue;
         }
#endif   /* IP_V6 */
         ns_printf(pio, "   %s\n",  print_ipad(imp->inm_addr) );
      }
   }
#endif   /* IP_MULTICAST */

	// EMG - Added code to facilitate software negotiation testing

	if (!fec_mii_read(0, 0, &settings))
    	ns_printf(pio, "\n\nControl Register                 =  Error  \n");
    else
    {
		ns_printf(pio, "\n\nControl Register                 = %04x\n",settings);
		
		// 	DATARATE  Speed Selection  0x2000
		//	The link speed will be selected either through the auto-negotiation process or by manual speed
		//	selection. ANE allows manual speed selection while it is set to 0. While auto-negotiation is enabled,
		//	DATARATE can be read or written but its value is not required to reflect speed of the link.
		//	1 = While auto-negotiation is disabled, selects 100 Mbps operation
		//	0 = While auto-negotiation is disabled, selects 10 Mbps operation
		if( settings & 0x2000 )
			ns_printf( pio, "\nDATARATE = 100Mbps" );
		else
			ns_printf( pio, "\nDATARATE = 10Mbps" );
			

		//	ANE  Auto-Negotiation Enable  0x1000
		//	The ANE bit determines whether the A/N process is enabled. When auto-negotiation is disabled,
		//	DATARATE and DPLX determine the link configuration. While auto-negotiation is enabled, bits
		//	DATARATE and DPLX do not affect the link.
		//	1 = Enables auto-negotiation
		//	0 = Disables auto-negotiation
		if( settings & 0x1000 )
			ns_printf( pio, "\nANE = Autonegotiation Enabled" );
		else
			ns_printf( pio, "\nANE = Autonegotiation Disabled" );

		//	DPLX  Duplex Mode  0x0100
		//	This mode can be selected by either the auto-negotiation process or manual duplex selection. Manual
		//	duplex selection is allowed only while the auto-negotiation process is disabled (ANE=0). While the
		//	auto-negotiation process is enabled (ANE = 1), the state of DPLX has no effect on the link
		//	configuration. While loopback mode is asserted (LOOPBACK =1), the value of DPLX will have no
		//	effect on the PHY.
		//	1 = Indicates full-duplex mode
		//	0 = Indicates half-duplex mode
		if( settings & 0x0100 )
			ns_printf( pio, "\nDPLX = Full Duplex" );
		else
			ns_printf( pio, "\nDPLX = Half Duplex" );
    }

	ns_printf(pio, "\n\nThis register advertises the capabilities of the port to the MII\n" );
    if (!fec_mii_read(0, 1, &settings))
    	ns_printf(pio, "Status Register                  =  Error  \n");
    else
    {
		ns_printf(pio, "Status Register                  = %04x\n",settings);

		// 100XFD 100BASE-TX Full-Duplex  0x4000
		// 1 = Indicates PHY supports 100BASE-TX full-duplex mode
		// 0 = Indicates PHY does not support 100BASE-TX full-duplex mode
		if( settings & 0x4000 )
			ns_printf( pio, "\nIndicates the PHY supports 100BASE-TX full-duplex mode");
		else
			ns_printf( pio, "\nIndicates the PHY does not support 100BASE-TX full-duplex mode");

		// 100XHD 100BASE-TX Half-Duplex  0x2000
		// 1 = Indicates the PHY supports 100BASE-TX half-duplex mode
		// 0 = Indicates the PHY does not support 100BASE-TX half-duplex mode
		if( settings & 0x2000 )
			ns_printf( pio, "\nIndicates the PHY supports 100BASE-TX half-duplex mode");
		else
			ns_printf( pio, "\nIndicates the PHY does not support 100BASE-TX half-duplex mode");

		// 10TFD 10BASE-T Full-Duplex	0x1000
		// 1 = Indicates the PHY supports 10BASE-T full-duplex mode
		// 0 = Indicates the PHY does not support 10BASE-T full-duplex mode
		if( settings & 0x1000 )
			ns_printf( pio, "\nIndicates the PHY supports 10BASE-T full-duplex mode");
		else
			ns_printf( pio, "\nIndicates the PHY does not support 10BASE-T full-duplex mode");

		// 10THD 10BASE-T Half-Duplex	0x08000
		// 1 = Indicates the PHY supports 10BASE-T half-duplex mode
		// 0 = Indicates the PHY does not support 10BASE-T half-duplex mode
		if( settings & 0x0800 )
			ns_printf( pio, "\nIndicates the PHY supports 10BASE-T half-duplex mode");
		else
			ns_printf( pio, "\nIndicates the PHY does not support 10BASE-T half-duplex mode");
		
		//	REMFLT  Remote Fault	0x0010
		//	Possible remote faults (RF)
		//	a) The link partner transmits the RF bit (5.13=1)
		//	b) Link partner protocol is not 00001 (5.4:0)
		//	c) Link partner advertises only T4 capability (5.9:5)
		//	d) No common operation mode found between PHY and the link partner.
		//		After it is set, REMFLT is cleared each time register 1 is read via the management interface. REMFLT
		//		is also cleared by a PHY reset.
		//	1 = Indicates that a remote fault condition has been detected.
		//	0 = No fault detected
		if( settings & 0x0010 )
			ns_printf( pio, "\na remote fault condition has been detected");
		else
			ns_printf( pio, "\nNo fault detected");
		
		//	ANABL  Auto-Negotiation Ability	0x0008
		//	1 = Indicates that PHY has auto-negotiation ability
		//	0 = Indicates that PHY does not have auto-negotiation ability
		if( settings & 0x0008 )
			ns_printf( pio, "\nPHY has auto-negotiation ability");
		else
			ns_printf( pio, "\nPHY does not have auto-negotiation ability");

		//	LNKSTST  Link Status	0x0004
		//	The PHY sets this bit when it determines that a valid link has been established. The occurrence of a
		//	link failure will cause LNKSTST to be cleared. After it has been cleared, it remains cleared until it is
		//	read via the management interface.
		//	1 = Indicates a valid link has been established
		//	0 = Indicates a valid link has NOT been established		
		if( settings & 0x0004 )
			ns_printf( pio, "\nvalid link has been established");
		else
			ns_printf( pio, "\nvalid link has NOT been established");
		
		//	ANCOMP Auto-Negotiation Complete  0x0020
		//	To inform the management interface (MI) that it has completed processing, ANCOMP is set by the
		//	A/N process. After it has been started, the auto-negotiation process uses link code words to exchange
		//	capability information and establish the highest common denominator (HCD) for link transactions.
		//	1 = Indicates that the auto-negotiation process has completed and that the contents of registers 4
		//		through 7 are valid.
		//	0 = Indicates that the auto-negotiation process has not completed and that the contents of registers
		//		4 through 7 are not valid
		if( settings & 0x0020 )
		{
			ns_printf( pio, "\nAutoNegotiation complete - Data is Valid");
			autodone = 1;
		}
		else
			ns_printf( pio, "\nAutoNegotiation NOT complete - Data is NOT Valid");
    }
	
	if( autodone )
	{
    	if (!fec_mii_read(0, 2, &settings))
    		ns_printf(pio, "\n\nPHY Identifier 1 Register        =  Error  \n");
    	else
			ns_printf(pio, "\n\nPHY Identifier 1 Register        = %04x\n",settings);

    	if (!fec_mii_read(0, 3, &settings))
    		ns_printf(pio, "\n\nPHY Identifier 2 Register        =  Error  \n");
    	else
			ns_printf(pio, "\n\nPHY Identifier 2 Register        = %04x\n",settings);
	}
	
    	if (!fec_mii_read(0, 4, &settings))
    		ns_printf(pio, "\n\nAuto-Neg. Advertisement Register =  Error  \n");
    	else
    	{
			ns_printf(pio, "\n\nAuto-Neg. Advertisement Register = %04x\n",settings);

			// TAF100FD  100BASE-TX Full-Duplex  0x0100
			// 1 = 100BASE-TX full -duplex capable
			// 0 = Not 100BASE-TX full-duplex capable
			if( settings & 0x0100 )
				ns_printf( pio, "\n100BASE-TX full -duplex capable");
			else
				ns_printf( pio, "\nNot 100BASE-TX full-duplex capable");

			// TAF100HD  100BASE-TX Half-Duplex	0x0080
			// 1 = 100BASE-TX half-duplex capable
			// 0 = Not 100BASE-TX half-duplex capable
			if( settings & 0x0080 )
				ns_printf( pio, "\n100BASE-TX half-duplex capable");
			else
				ns_printf( pio, "\nNot 100BASE-TX half-duplex capable");

			// TAF10FD  10BASE-T Full-Duplex		0x0040
			// 1 = 10BASE-T full-duplex capable
			// 0 = Not 10BASE-T full-duplex capable
			if( settings & 0x0040 )
				ns_printf( pio, "\n10BASE-T full-duplex capable");
			else
				ns_printf( pio, "\nNot 10BASE-T full-duplex capable");

			// TAF10HD  10BASE-T Half-Duplex		0x0020
			// 1 = 10BASE-T half-duplex capable
			// 0 = Not 10BASE-T half-duplex capable
			if( settings & 0x0020 )
				ns_printf( pio, "\n10BASE-T half-duplex capable");
			else
				ns_printf( pio, "\nNot 10BASE-T half-duplex capable");
    	}
	
	
	if( autodone )
    {
    	if (!fec_mii_read(0, 5, &settings))
    		ns_printf(pio, "\n\nLink Partner Ability Register    =  Error  \n");
    	else
    	{
			ns_printf(pio, "\n\nLink Partner Ability Register    = %04x\n",settings);

			// TAF100T4  100BASE-T4 Full-Duplex
			// 1 = Link partner is 100BASE-T4 capable
			// 0 = Link partner is not 100BASE-T4 capable
			// This function is not implemented in the EPHY.
			if( settings & 0x0200 )
				ns_printf( pio, "\nLink partner is 100BASE-T4 capable");
			else
				ns_printf( pio, "\nLink partner is not 100BASE-T4 capable");
		
			// TAF100FD  100BASE-TX Full-Duplex
			// 1 = Link partner is 100BASE-TX full-duplex capable
			// 0 = Link partner is not 100BASE-TX full-duplex capable
			if( settings & 0x0100 )
				ns_printf( pio, "\nLink partner is 100BASE-TX full-duplex capable");
			else
				ns_printf( pio, "\nLink partner is not 100BASE-TX full-duplex capable");

			// TAF100HD  100BASE-TX Half-Duplex
			// 1 = Link partner is 100BASE-TX half-duplex capable
			// 0 = Link partner is not 100BASE-TX half-duplex capable
			if( settings & 0x0080 )
				ns_printf( pio, "\nLink partner is 100BASE-TX half-duplex capable");
			else
				ns_printf( pio, "\nLink partner is not 100BASE-TX half-duplex capable");

			// TAF10FD  10BASE-T Full-Duplex
			// 1 = Link partner is10BASE-T full-duplex capable
			// 0 = Link partner is not 10BASE-T full-duplex capable
			if( settings & 0x0040 )
				ns_printf( pio, "\nLink partner is10BASE-T full-duplex capable");
			else
				ns_printf( pio, "\nLink partner is not 10BASE-T full-duplex capable");

			// TAF10HD  10BASE-T Half-Duplex
			// 1 = Link partner is 10BASE-T half-duplex capable
			// 0 = Link partner is not 10BASE-T half-duplex capable
			if( settings & 0x0020 )
				ns_printf( pio, "\nLink partner is 10BASE-T half-duplex capable");
			else
				ns_printf( pio, "\nLink partner is not 10BASE-T half-duplex capable");
		}
    
    	if (!fec_mii_read(0, 6, &settings))
    		ns_printf(pio, "\n\nAuto-Neg. Expansion Register     =  Error  \n");
    	else
			ns_printf(pio, "\n\nAuto-Neg. Expansion Register     = %04x\n",settings);

    	if (!fec_mii_read(0, 7, &settings))
    		ns_printf(pio, "\n\nAuto-Neg. Next Page Transmit     =  Error  \n");
    	else
			ns_printf(pio, "\n\nAuto-Neg. Next Page Transmit     = %04x\n",settings);

    	if (!fec_mii_read(0, 16, &settings))
    		ns_printf(pio, "\n\nInterrupt Register Register      =  Error  \n");
    	else
			ns_printf(pio, "\n\nInterrupt Register Register      = %04x\n",settings);

    	if (!fec_mii_read(0, 17, &settings))
    		ns_printf(pio, "\n\nProprietary Status Register      =  Error  \n");
    	else
			ns_printf(pio, "\n\nProprietary Status Register      = %04x\n",settings);

    	if (!fec_mii_read(0, 18, &settings))
    		ns_printf(pio, "\n\nProprietary Control Register     =  Error  \n");
    	else
			ns_printf(pio, "\n\nProprietary Control Register     = %04x\n",settings);

    	if (!fec_mii_read(0, 20, &settings))
    		ns_printf(pio, "\n\nRegister 20                      =  Error  ");
    	else
			ns_printf(pio, "\n\nRegister 20                      = %04x\n",settings);
    }
    
    ns_printf(pio, "\n\n" );
	return 0;
}





/* FUNCTION: linkstats()
 * 
 * printf stats for link interface. default to iface 1
 *
 * PARAM1: void * pio   - output stream
 *
 * RETURNS: 0 if OK, else -1 if arg is bad.
 */

int
linkstats(void * pio)
{
   int      iface =  1;
   char *   cp;

   cp = nextarg(((GEN_IO)pio)->inbuf);

   if (*cp)
   {
      iface = atoi(cp);
      if (iface < 1 || iface > (int)ifNumber)
      {
         ns_printf(pio,"interface number must be 1 to %d.\n", ifNumber);
         return -1;
      }
   }
   iface--;

   if (nets[iface]->n_stats)
      nets[iface]->n_stats(pio,iface);  /* actually call the stats */
   else
      ns_printf(pio,"no stats routine for interface %d\n", iface+1);
   return 0;
}


/* FUNCTION: settrapsize()
 *
 * settrapsize() - This sets a value which will cause the memory 
 * managment monitoring code to trap to a debugger if a malloc is 
 * attempted for this size. This is usefull when a memory leak is 
 * occuring of memory buffers on an unusual size, as often happens 
 * with networking code. It has nothing to do with SNMP traps. 
 * Returns no meaningfull value; declared as int to make compiler 
 * happy. 
 *
 * 
 * PARAM1: void * pio   - output stream
 *
 * RETURNS: 0
 */

extern   int   memtrapsize;   /* in memman.c or user heap code */

int
settrapsize(void * pio) 
{
   char *   cp;

   cp = nextarg(((GEN_IO)pio)->inbuf);

   memtrapsize = atoi(cp);
   ns_printf(pio,"malloc trap size set to %d\n", memtrapsize);
   return 0;
}


//*****************************************************************************
// int   set_baud(void * pio)   Written By Eric Gregori   (847) 651 - 1971
//						        eric.gregori@freescale.com
//
// User command to set baud rate for serial port.
//
//*****************************************************************************
int   set_baud(void * pio)
{
   	char 	*cp;


   	cp = nextarg(((GEN_IO)pio)->inbuf);
   	
   	if( *cp )
		iuart_set_baud( 0, atoi(cp) );	
	
	return(0);
}


//*****************************************************************************
// int   cat(void * pio)   Written By Eric Gregori   (847) 651 - 1971
//						   eric.gregori@freescale.com
//
// Outputs file to console.
//
//*****************************************************************************
int   cat(void * pio)
{
   	char 			*cp;
   	unsigned char	*data;
   	uint8			bad_char;
   	uint32			bytes;
   	uint32			index;
   	uint32			i;


	ns_printf(pio, "\n\n" );

   	cp = nextarg(((GEN_IO)pio)->inbuf);
	if( emg_open(cp, &index, &bytes) == 0 )
	{
		data 		= (unsigned char *)index;
		index 		= 0;
		bad_char 	= 0;
		while( (index < bytes) && !bad_char )
		{
			for( i=0; i<10; i++ )
			{
				if( (data[index] < 8) || (data[index] > 127) )
				{
//					bad_char = data[index];
//					break;	
				}
				else		
					ns_printf( pio, "%c", data[index]);
				
				index++;
				
				if( index == bytes )
					break;
			}
			
			tk_sleep(2);
		}		
	}
	else
	   ns_printf(pio, "File Not Found" );
	
	if( bad_char )
	   ns_printf(pio, "\n\nCAT stopped, non-ascii character detected 0x%x", bad_char );
	
	ns_printf(pio, "\n\n" );
	
	return(0);
}

