/* FILENAME: iuart.c 
 *
 * Copyright  2005 By InterNiche Technologies Inc. All rights reserved
 *
 * Code for interfacing with UARTs
 *
 * FUNCTIONS: uart_init(), uart_getc(), uart_putc(), 
 *          uart_flush(), uart_ready(), uart_present(),
 *          uart_close(), uart_stats(), uart_check(),
 *          uart_isr(), uart0_isr(), uart1_isr(),
 * PPP FUNCTIONS: ln_uinit(), ln_uconnect(), ln_udisconnect(),
 *          ln_uputc(), uart_baud()
 *            
 * If POLLED_UART is defined, then UART I/O is done via polling, else
 * UART I/O is done using interrups.
 *
 * When POLLED_UART is defined
 * 1. Console works, PPP doesn't work (we loose data)
 * 2. uart_check() calls uart_isr() to process Tx and Rx FIFOs.
 *
 * When POLLED_UART is not defined
 * 1. UART I/O is done using interrupts
 * 2. In uart_init(), interrupts are enabled for the UART0, UART1
 * 3. When interrupt occurs, from the vector table (vectors.s),
 *    uartN_isr() is called. uartN_isr() calls uart_isr(N)
 *    It checks Tx and Rx status to determine I/O processing.
 *    If new data is received, get it and place it in the Rx buffer.
 *    If there is data in the Tx buffer, transmit it. 
 *
 * To use interrupts on EVB, we need to do the following
 * 1. UART0 and UART1 have unique interrupt vector numbers.
 * 2. Configure the port bits (done in device initialization).
 * 3. Enable the interrupts for the UARTs (done in uart_init())
 *    and enable the recievers. Set the transmitters to "idle".
 * 4. When the interrupt occurs, check the Rx status for new
 *    characters and the Tx status for room to transmit more
 *    characters. Continue until Rx and Tx clear.
 *
 */
   
#include "ipport.h"
#include "comline.h"
#include "uart.h"
#ifdef USE_PPP
#include "ppp_port.h"
#include "mppp.h"
#endif

/* define POLLED_UART to poll the uarts to do I/O */
//#define POLLED_UART    

#ifdef POLLED_UART
/* Define BLOCKING_IO to block till complete output is sent to console 
 * It is not needed when ISR is used for console I/O. */
#define BLOCKING_IO    1  
#endif

#ifndef UART_RXBUFSIZE
#define UART_RXBUFSIZE        32	// EMG
#endif

#ifndef UART_TXBUFSIZE
#define UART_TXBUFSIZE        256	// EMG - I decreased this
#endif

#ifndef UART0_SPEED
#define UART0_SPEED           115200
#endif

#ifndef UART1_SPEED
#define UART1_SPEED           19200
#endif

struct uart_desc
{
   int unit;                  /* unit number */
   unsigned long speed;       /* baud rate */
   unsigned char *rx_buf;
   volatile unsigned rx_in;
   volatile unsigned rx_out;
   unsigned char *tx_buf;
   volatile unsigned tx_in;
   volatile unsigned tx_out;
   volatile unsigned tx_idle;
   int initdone;
};

struct uart_desc uarts[] = {
   { 
     0,
     UART0_SPEED,
     NULL,
     NULL,
     0, 0,
     NULL,
     0, 0,
     0
   },
   { 
     1,
     UART1_SPEED,
     NULL,
     NULL,
     0, 0,
     NULL,
     0, 0,
     0
   }
};

unsigned char uart_rxbufs[sizeof(uarts)/sizeof(uarts[0])][UART_RXBUFSIZE];
unsigned char uart_txbufs[sizeof(uarts)/sizeof(uarts[0])][UART_TXBUFSIZE];

__declspec(interrupt) void uart0_isr(void);
__declspec(interrupt) void uart1_isr(void);
void uart_isr(int);

/* For the 16K RAM build, network stats are disabled to save space.
 * Hence we use the following variables to find out any runtime problems.
 * They are printed in "uart_stats". From the INET> prompt, when 
 * "linkstats" command is executed, uart_stats is called and these
 * values get printed.
 */
static long utxbuffull=0;   /* num of times tx buffer got full */
static long utxbytedrop=0;  /* num of bytes dropped bcoz tx buffer was full */
static long urxbytedrop=0;  /* num of rx bytes dropped bcoz rx buf was full */

int uart_yield = 0;         /* TRUE = UART can use tk_yield when Tx full */

int	autobaud = 0;		    /* Autobaud index */
int fe_count = 0;			/* counts framing error failures */
#define FE_COUNT_THRES		3

int	autobaud_table[] = {115200, 38400, 0};

void iuart_set_baud( int dev, int baud );

/* FUNCTION: uart_init()
 *
 * Sets up UART device
 *
 * PARAM1: unit;
 *
 * RETURNS: 0 if successful, otherwise -1
 */

int
uart_init(int unit)
{
   struct uart_desc *uart;
   int dev;
   int divisor;

   if ((unit < 0) || (unit >= sizeof(uarts)/sizeof(uarts[0])))
   {
      dtrap();
      return (-1);
   }
   uart = &uarts[unit];
   dev  = uart->unit;

   /* empty buffers */
   uart->rx_buf = &uart_rxbufs[dev][0];
   uart->tx_buf = &uart_txbufs[dev][0];
   uart->rx_in  = uart->rx_out = 0;
   uart->tx_in  = uart->tx_out = 0;
   uart->tx_idle = 1;
#ifdef USE_LEDS
   out_led(uart->tx_idle);
#endif

   /* reset the UART */
   /* disable interrupts, reset the device, and disable Rx and Tx */
   MCF_UART_UIMR(dev) = 0;
   MCF_UART_UCR(dev) = MCF_UART_UCR_RESET_RX;
   MCF_UART_UCR(dev) = MCF_UART_UCR_RESET_TX;
   MCF_UART_UCR(dev) = MCF_UART_UCR_RESET_ERROR;
   MCF_UART_UCR(dev) = MCF_UART_UCR_BKCHGINT;
   MCF_UART_UCR(dev) = ( MCF_UART_UCR_RESET_MR |
                             MCF_UART_UCR_TX_DISABLED |
                             MCF_UART_UCR_RX_DISABLED );

   /* configure for 8,N,1 with flow control */
   MCF_UART_UMR(dev) = ( //jpw MCF_UART_UMR_RXRTS |
                             MCF_UART_UMR_PM_NONE |
                             MCF_UART_UMR_BC_8 );
   MCF_UART_UMR(dev) = ( MCF_UART_UMR_CM_NORMAL |
//jpw                             MCF_UART_UMR_TXCTS |
                             MCF_UART_UMR_SB_STOP_BITS_1 );

   /* baud rate is set by calculating a clock divider:
    *    divider = (f_sys/2) / (32 * baudrate)
    */
//jpw   divisor = ((((SYS_CLK_KHZ * 1000) / uart->speed) + 16) / 32);
   divisor = (uint16)((SYSTEM_CLOCK*1000000)/(UART_BAUD * 32));
   MCF_UART_UBG1(dev) = (divisor >> 8) & 0xff;
   MCF_UART_UBG2(dev) = divisor & 0xff;
   MCF_UART_UCSR(dev) = ( MCF_UART_UCSR_RCS_SYS_CLK |
                              MCF_UART_UCSR_TCS_SYS_CLK );
   
	MCF_UART0_UBG1 = (uint8)((divisor & 0xFF00) >> 8);
	MCF_UART0_UBG2 = (uint8)(divisor & 0x00FF);
   /* initialize the UART interrupts */
//jpw - for mulitple parts this needs fixed
   if (dev == 0)
   {
//      mcf5223_interrupt_init(13, 2, uart0_isr);
	  MCF_INTC0_ICR13 = MCF_INTC_ICR_IL(2);
	  MCF_INTC0_IMRL &= ~MCF_INTC_IMRL_MASK13;
   }
   else
   {
//      mcf5223_interrupt_init(14, 2, uart1_isr);
	  MCF_INTC0_ICR14 = MCF_INTC_ICR_IL(2);
	  MCF_INTC0_IMRL &= ~MCF_INTC_IMRL_MASK14;
   }

   /* enable receiver and Rx & Tx interrupts */

#ifdef POLLED_UART
   MCF_UART_UCR(dev) = ( MCF_UART_UCR_RX_ENABLED |
                             MCF_UART_UCR_TX_ENABLED );
#else
   MCF_UART_UCR(dev) = ( MCF_UART_UCR_RX_ENABLED );
   MCF_UART_UIMR(dev) = ( MCF_UART_UIMR_TXRDY |
                              MCF_UART_UIMR_RXRDY_FU );
#endif
         
   uart->initdone = 1;

   return (0);
}


/* FUNCTION: uart_putc()
 *
 * Send a character to a comm port 
 *
 * PARAM1: unit;        UART device number
 * PARAM2: ch;          character to send
 *
 * RETURN: 0 if successful, otherwise an error code
 *
 * The character is placed in the output buffer (tx_buf).
 * If the UART transmitter is idle, it is started, and the 
 * interrupt handler will move the character to the UART.
 */

int
uart_putc(int unit, unsigned char ch)
{
   struct uart_desc *uart;
   int dev;
   int loopcount = 0;
   int tx_next;

   if ((unit < 0) || (unit >= sizeof(uarts)/sizeof(uarts[0])))
   {
      dtrap();                /* illegal unit number */
      return (-1);
   }
   uart = &uarts[unit];
   dev = uart->unit;

   if (uart->initdone != 1)   /* unit initialized? */
      return (0);

   if ((tx_next = uart->tx_in + 1) >= UART_TXBUFSIZE)
      tx_next -= UART_TXBUFSIZE;

   if (tx_next == uart->tx_out)
   {
      utxbuffull++;
      do
      {
         /* if tx_buf is full, and iniche task is initialized,
            let system run to free up some space */
         if (uart_yield)
         {
            tk_yield();
            if (++loopcount > 10000)
            {
               utxbytedrop++;
               return (-1);
            }
         }
//#ifdef POLLED_UART
         else
         {
            uart_isr(dev);
         }
//#endif
      }
      while (tx_next == uart->tx_out);
   }

   uart->tx_idle = 0;
   uart->tx_buf[uart->tx_in] = ch;
   uart->tx_in = tx_next;
   MCF_UART_UCR(dev) = MCF_UART_UCR_TX_ENABLED;

#ifdef POLLED_UART
   uart_isr(dev);
#endif

   return (0);
}


/* FUNCTION: uart_getc()
 *
 * Read the next buffered character from the com port
 *
 * PARAM1: unit;        UART device number
 *
 * RETURN: next buffered character or -1 if buffer is empty
 */

int
uart_getc(int unit)
{
   struct uart_desc *uart;
   int ch;

   if ((unit < 0) || (unit >= sizeof(uarts)/sizeof(uarts[0])))
   {
      dtrap();
      return (-1);
   }
   uart = &uarts[unit];

#ifdef POLLED_UART
   uart_isr(uart->unit);
#endif

   if(uart->rx_in == uart->rx_out)
      return (-1);     /* no char ready */

   ch = (int)uart->rx_buf[uart->rx_out];
   if (++uart->rx_out >= UART_RXBUFSIZE)  uart->rx_out -= UART_RXBUFSIZE;

   return (ch);
}


/* FUNCTION: uart_flush()
 *
 * Check if the transmitter buffer is empty
 *
 * PARAM1: unit;        UART device number
 *
 * RETURN: 1 if tx_buf is empty, otherwise 0
 */

int
uart_flush(int unit)
{
   struct uart_desc *uart;

   if ((unit < 0) || (unit >= sizeof(uarts)/sizeof(uarts[0])))
   {
      dtrap();
      return (0);
   }
   uart = &uarts[unit];

#ifdef POLLED_UART
   uart_isr(uart->unit);
#endif

   return (uart->tx_in == uart->tx_out ? 1 : 0);
}


/* FUNCTION: uart_ready()
 *
 * Check if the transmitter buffer is NOT full
 *
 * PARAM1: unit;        UART device number
 *
 * RETURN: 0 if tx_buf is full, otherwise 1
 */

int
uart_ready(int unit)
{
   struct uart_desc *uart;
   int tx_next;

   if ((unit < 0) || (unit >= sizeof(uarts)/sizeof(uarts[0])))
   {
      dtrap();
      return (0);
   }
   uart = &uarts[unit];

#ifdef POLLED_UART
   uart_isr(uart->unit);
#endif

   if ((tx_next = uart->tx_in + 1) >= UART_TXBUFSIZE)
      tx_next -= UART_TXBUFSIZE;

   return (tx_next != uart->tx_out ? 1 : 0);
}


/* FUNCTION: uart_present()
 *
 * Check if there is there is a character ready to be read
 *
 * PARAM1: unit         UART device number
 *
 * RETURN: -1 if character is available, otherwise 0
 */

int
uart_present(int unit)
{
   struct uart_desc *uart;

   if ((unit < 0) || (unit >= sizeof(uarts)/sizeof(uarts[0])))
   {
      dtrap();
      return (0);
   }
   uart = &uarts[unit];

#ifdef POLLED_UART
   uart_isr(uart->unit);
#endif

   return (uart->rx_in != uart->rx_out ? -1 : 0);
}


/* FUNCTION: uart_close()
 *
 * Closes the UART
 *
 * PARAM1: unit;        UART device number
 *
 * RETURN: none
 */

void
uart_close(int unit)
{
   struct uart_desc *uart;
   int dev;

   if ((unit < 0) || (unit >= sizeof(uarts)/sizeof(uarts[0])))
   {
      dtrap();
      return;
   }
   uart = &uarts[unit];
   dev = uart->unit;

   /* turn off the Tx and Rx hardware */
   MCF_UART_UCR(dev) = ( MCF_UART_UCR_TX_DISABLED |
                             MCF_UART_UCR_RX_DISABLED );
   
   /* empty buffers */
   uart->rx_in = uart->rx_out = 0;
   uart->tx_in = uart->tx_out = 0;
   uart->tx_idle = 1;
}


/* FUNCTION: uart_stats()
 *
 * Print UART statistics 
 *
 * PARAM1: pio;         pointer to output device structure
 * PARAM2: unit;        UART device number 
 *
 * RETURN: 0 if successful, otherwise -1
 */

#ifndef MEM_WRAPPERS
extern char *maxheaploc ;
extern long maxheaploclen ;
#endif

int
uart_stats(void *pio, int unit)
{
   struct uart_desc *uart;

   if ((unit < 0) || (unit >= sizeof(uarts)/sizeof(uarts[0])))
   {
      dtrap();
      return (-1);
   }
   uart = &uarts[unit];

   ns_printf(pio, "baud = %ld rx_in = %ld, rx_out =%ld", 
      uart->speed, uart->rx_in, uart->rx_out);
   ns_printf(pio, "tx_in = %ld, tx_out=%ld, tx_idle =%ld\n", 
      uart->tx_in, uart->tx_out, uart->tx_idle);
   ns_printf(pio, "txbuffull = %ld, txbytedrop =%ld, rxbytedrop =%ld\n", 
      utxbuffull, utxbytedrop, urxbytedrop);
#ifndef MEM_WRAPPERS
   ns_printf(pio,"maxheaploc = %lx, size =%ld\n", maxheaploc, maxheaploclen);
#endif

   return (0);
}


/* FUNCTION: uart_check()
 *
 * check for UART activity when devices are polled
 *
 * PARAMS: none
 *
 * RETURN: none
 *
 * A character (if available) is read from each UART that is being 
 * used for a PPP session.
 */

void
uart_check()
{
   struct uart_desc *uart;
   int unit;
   int ch;

   if (uarts[0].initdone != 1)
      return;

   for (unit = 0; unit < sizeof(uarts)/sizeof(uarts[0]); unit++)
   {
      uart = &uarts[unit];

#ifdef POLLED_UART /* used in polled mode */
      uart_isr(uart->unit);  /* finish off pending send/receive */
#endif

#ifdef USE_PPP
      while ((uart->rx_out != uart->rx_in) &&
             (uart->line) && (uart->line->ln_getc))
      {
         ch = uart_getc(uart->unit);
         if (ch == -1)
            break;
         if (uart->line->ln_state == LN_DISCONNECTED)
            ln_uconnect(uart->line);
         uart->line->ln_getc(uart->line, ch);
      }
#endif

   }
}


/* FUNCTION: uart_isr()
 *
 * UART interrupt service request handler
 *
 * PARAM1: unit;        UART device number
 *
 * RETURN: none
 *
 * Copy characters from the UART Rx buffer to rx_buf.
 * Copy characters from tx_buf to the UART Tx buffer.
 * Turn off the Tx buffer if it is idle.
 *
 * The 'unit' parameter is guarenteed to be valid, so we
 * don't do any error checking.
 */

void
uart_isr(int unit)
{
   struct uart_desc *uart = &uarts[unit];
   int dev = uart->unit;
   uint8 ch;
   uint8 usr;
   int   rx_next, tx_next;

   usr = MCF_UART_USR(dev);
   /* UART receiver */
   if (usr & MCF_UART_USR_RXRDY)
   {
      ch = MCF_UART_URB(dev);
      /* discard character if there was an error */
      if ( !(usr & (MCF_UART_USR_RB |
                    MCF_UART_USR_FE |
                    MCF_UART_USR_PE |
                    MCF_UART_USR_OE )) )
      {
         if ((rx_next = uart->rx_in + 1) >= UART_RXBUFSIZE)
            rx_next -= UART_RXBUFSIZE;
         if (rx_next != uart->rx_out)
         {
            uart->rx_buf[uart->rx_in] = ch;
            uart->rx_in = rx_next;
         }
         else
            urxbytedrop++;    /* lost character - buffer full */
		
		if( ch > 127 )
		{
			fe_count++;
			if( fe_count > FE_COUNT_THRES )
			{
				fe_count = 0;
				autobaud++;
				if( autobaud_table[autobaud] == 0 )
					autobaud = 0;
			
				iuart_set_baud( 0, autobaud_table[autobaud] );
			}			
		}
		else
			fe_count = 0;
      }
      else
      {
      	MCF_UART_UCR(dev) = MCF_UART_UCR_RESET_ERROR;
      	fe_count++;

		if( fe_count > FE_COUNT_THRES )
		{
			fe_count = 0;
			autobaud++;
			if( autobaud_table[autobaud] == 0 )
				autobaud = 0;
			
			iuart_set_baud( 0, autobaud_table[autobaud] );
		}
      }
   }

   /* UART transmitter */
   if (usr & MCF_UART_USR_TXRDY)
   {
      if (uart->tx_out != uart->tx_in)
      {
         MCF_UART_UCR(dev) = MCF_UART_UCR_TX_ENABLED;
         ch = uart->tx_buf[uart->tx_out++];
         if (uart->tx_out >= UART_TXBUFSIZE)
            uart->tx_out -= UART_TXBUFSIZE;
         MCF_UART_UTB(dev) = ch;
      }
      else
      {
         MCF_UART_UCR(dev) = MCF_UART_UCR_TX_DISABLED;
         uart->tx_idle = 1;
      }
   }
}


/* FUNCTION: uart0_isr()
 *
 * interrupt handler for UART #0
 *
 * PARAMS: none
 *
 * RETURN: none
 */

__declspec(interrupt) 
void
uart0_isr(void)
{
    uart_isr(0);
}


/* FUNCTION: uart1_isr()
 *
 * interrupt handler for UART #0
 *
 * PARAMS: none
 *
 * RETURN: none
 */

__declspec(interrupt) 
void
uart1_isr(void)
{
    uart_isr(1);
}


#ifdef USE_COMPORT
/*
 * Next few routines are support for Interniche line structure.
 */

int units_set = 0;


/* FUNCTION: ln_uinit()
 *
 * Initializes the device specified in the com_line struct
 *
 * PARAM1: line         pointer to com_line structure
 *
 * RETURN: 0 if successful, otherwise an error code
 */

int
ln_uinit(LINEP line)
{
   int   unit;
   int   err;

   if (line->ln_state != LN_INITIAL)
   {
      dtrap();       /* don't call this multiple times */
      return (0);      /* return OK anyway... */
   }

   if (units_set++ > 0)
   {
      dtrap();    /* trying to set up multiple UARTs unsupported */
   }
   unit = 1;  /* channel 1 == unit 1 */
   err = uart_init(unit);  /* open UART and assign comport */
   
   if (err)
      return (err);    /* report hardware failures */

   line->lower_unit = unit;
   line->ln_state = LN_CONNECTED;
   uarts[unit].line = line;

   return (0);
}


/* FUNCTION: ln_uconnect()
 *
 * Put PPP device into the 'connected' state
 *
 * PARAM1: line;        pointer to com_line structure 
 * 
 * RETURN: 0 if successful, otherwise an error code
 */
int
ln_uconnect(LINEP line)
{
   int   err;
   
   if (line->ln_state == LN_INITIAL) 
   {
      err = ln_uinit(line);
      if (err)
         return (err);   /* report hardware failures */
   }
   line->ln_state = LN_CONNECTED;

   /* Indicate to PPP that lower link is now up */
   ppp_lowerup((M_PPP)(line->upper_unit), LCP_STATE);

   return (0);
}


/* FUNCTION: ln_udisconnect()
 *
 * 'Disconnect' the PPP device
 *
 * PARAM1: line         pointer to com_line structure
 *
 * RETURN: 0 (always successful)
 */

int
ln_udisconnect(LINEP line)
{
   line->ln_state = LN_DISCONNECTED;
   return (0);
}


/* FUNCTION: ln_putc()
 *
 * send a single byte to the PPP device 
 *
 * PARAM1: line         pointer to com_line structure
 * PARAM2: ch           character to be sent
 *
 * RETURN: 0 if successful, otherwise an error code
 */

int
ln_uputc(LINEP line, int ch)
{
   return (uart_putc(line->lower_unit, (u_char)ch));
}

#endif   /* USE_PPP */

// EMG
void iuart_set_baud( int dev, int baud )
{
   	int 	divisor;

   	divisor = (uint16)((SYSTEM_CLOCK*1000000)/(baud * 32));
   	MCF_UART_UBG1(dev) = (divisor >> 8) & 0xff;
   	MCF_UART_UBG2(dev) = divisor & 0xff;
}


