/*
 * File:		sysinit.c
 * Purpose:		Reset configuration of the M5223EVB
 *
 * Notes:
 */

#include "m5223evb.h"
#include "common.h"

/********************************************************************/

void mcf5223_init(void);
void mcf5223_wtm_init(void);
void mcf5223_pll_init(void);
void mcf5223_uart_init(void);
void mcf5223_scm_init(void);
void mcf5223_gpio_init(void);
void mcf5223_ePHY_init(void);
void mcf5223_powerup_config(void);


extern int SoftEthernetNegotiation( int seconds );
extern int poll_switches( void );


uint8	powerup_config_flags = 0;

/********************************************************************/
void
mcf5223_init(void)
{
	extern char __DATA_ROM[];
	extern char __DATA_RAM[];
	extern char __DATA_END[];
	extern char __BSS_START[];
	extern char __BSS_END[];
	extern char  __heap_addr[];
	extern char  __heap_size[];
	extern uint32 VECTOR_TABLE[];
	extern uint32 __VECTOR_RAM[];
	extern uint8 __SRAM[];
	extern uint8 __SRAM_SIZE[];
	extern uint32 __SP_INIT[];
	register uint32 n;
	register uint8 *dp, *sp;

	MCF_GPIO_PDDPAR = 0x0F;//enable pst[0..3]
		
	mcf5xxx_wr_vbr((uint32)VECTOR_TABLE);
			
	/*
	 * Move initialized data from ROM to RAM.
	 */
	if (__DATA_ROM != __DATA_RAM)
	{
		dp = (uint8 *)__DATA_RAM;
		sp = (uint8 *)__DATA_ROM;
		n = (uint32)(__DATA_END - __DATA_RAM);
		while (n--)
			*dp++ = *sp++;
	}

	/*
	 * Zero uninitialized data
	 */
	if (__BSS_START != __BSS_END)
	{
		sp = (uint8 *)__BSS_START;
		n = (uint32)(__BSS_END - __BSS_START);
		while (n--)
			*sp++ = 0;
	}

	/*
	 * Set Port UA to initialize URXD0/UTXD0
	 */
    MCF_GPIO_PUAPAR = 0
        | MCF_GPIO_PUAPAR_RXD0_RXD0
        | MCF_GPIO_PUAPAR_TXD0_TXD0;

    MCF_GPIO_PUBPAR = 0
        | MCF_GPIO_PUBPAR_RXD1_RXD1
        | MCF_GPIO_PUBPAR_TXD1_TXD1;

	/* Set real time clock freq */

	MCF_CLOCK_RTCDR = 25000000;
	
	mcf5223_wtm_init();
	mcf5223_pll_init();
	mcf5223_scm_init();
	mcf5223_gpio_init();

    uart_init(0);
    uart_init(1);

	mcf5223_powerup_config();
	mcf5223_ePHY_init();
}
/********************************************************************/
void
mcf5223_wtm_init(void)
{
	/*
	 * Disable Software Watchdog Timer
	 */
	MCF_SCM_CWCR = 0;
}
/********************************************************************/
void
mcf5223_pll_init(void)
{
	/* The PLL pre-divider affects this!!! 
	 * Multiply 25Mhz reference crystal /CCHR by 12 to acheive system clock of 60Mhz
	 */

	MCF_CLOCK_SYNCR = MCF_CLOCK_SYNCR_MFD(4) | MCF_CLOCK_SYNCR_CLKSRC| MCF_CLOCK_SYNCR_PLLMODE | MCF_CLOCK_SYNCR_PLLEN ;

	while (!(MCF_CLOCK_SYNSR & MCF_CLOCK_SYNSR_LOCK))
	{
	}
}
/********************************************************************/
void
mcf5223_scm_init(void)
{
	/*
	 * Enable on-chip modules to access internal SRAM
	 */
	MCF_SCM_RAMBAR = (0
		| MCF_SCM_RAMBAR_BA(SRAM_ADDRESS)
		| MCF_SCM_RAMBAR_BDE);
}
/********************************************************************/
tU08	gotlink;   						/**<Global Variable For Determination if
                                      * link is active (1=active)
                                      * defined in "main.c" */
                                      
void
mcf5223_gpio_init(void)
{
    /* 
     * Allow interrupts from ABORT, SW1, SW2, and SW3 (IRQ[1,4,7,11]) 
     */
     
#if 0  // EMG - Need to poll Switch state from web server
    /* Enable IRQ signals on the port */
    MCF_GPIO_PNQPAR = 0
        | MCF_GPIO_PNQPAR_IRQ1_IRQ1
        | MCF_GPIO_PNQPAR_IRQ4_IRQ4
        | MCF_GPIO_PNQPAR_IRQ7_IRQ7;
    
    MCF_GPIO_PGPPAR = 0
        | MCF_GPIO_PGPPAR_IRQ11_IRQ11;
        
    /* Set EPORT to look for rising edges */
    MCF_EPORT_EPPAR0 = 0
        | MCF_EPORT_EPPAR_EPPA1_RISING
        | MCF_EPORT_EPPAR_EPPA4_RISING
        | MCF_EPORT_EPPAR_EPPA7_RISING;
        
    MCF_EPORT_EPPAR1 = 0
        | MCF_EPORT_EPPAR_EPPA11_RISING;
        
    /* Clear any currently triggered events on the EPORT  */
    MCF_EPORT_EPIER0 = 0
        | MCF_EPORT_EPIER_EPIE1
        | MCF_EPORT_EPIER_EPIE4
        | MCF_EPORT_EPIER_EPIE7;
       
    MCF_EPORT_EPIER1 = 0
        | MCF_EPORT_EPIER_EPIE11;
       
    /* Enable interrupts in the interrupt controller */
	MCF_INTC0_IMRL &= 	~(MCF_INTC_IMRL_MASK1); // 
	MCF_INTC0_IMRL &= 	~(MCF_INTC_IMRL_MASK4); // 
	MCF_INTC0_IMRL &= 	~(MCF_INTC_IMRL_MASK7); // 

	MCF_INTC1_ICR35 = MCF_INTC_ICR_IL(4);
    MCF_INTC1_IMRH &=  ~(MCF_INTC_IMRH_MASK35);


#else
	  MCF_GPIO_PNQPAR = 0;
	  MCF_GPIO_PGPPAR = 0;

#endif
	/*
	 * Clear mask all bit to allow intrrupts
	 */
    MCF_INTC0_IMRL &= ~(MCF_INTC_IMRL_MASKALL);

	/*
	 * Initialize PLDPAR to enable Ethernet Leds
     * Ethernet status LEDs - Using GPIO for ACT/RX/TX
     *  is only for rev0 Si.	
	 */
	if ((MCF_CIM_CIR & 0x003F)==0)
	{
		MCF_GPIO_PLDPAR = (0
						| MCF_GPIO_PORTLD_PORTLD1 // LNKLED
						| MCF_GPIO_PORTLD_PORTLD2 // SPDLED
						| MCF_GPIO_PORTLD_PORTLD3 // DUPLED
						| MCF_GPIO_PORTLD_PORTLD4 // COLLED
						);

		MCF_GPIO_DDRLD =  MCF_GPIO_DDRLD_DDRLD0 // ACTLED
						| MCF_GPIO_DDRLD_DDRLD5 // RXLED
						| MCF_GPIO_DDRLD_DDRLD6 // TXLED
						;
	}
	else //not first Si
	{
		MCF_GPIO_PLDPAR = (0
						| MCF_GPIO_PORTLD_PORTLD0 // ACTLED
						| MCF_GPIO_PORTLD_PORTLD1 // LNKLED
						| MCF_GPIO_PORTLD_PORTLD2 // SPDLED
						| MCF_GPIO_PORTLD_PORTLD3 // DUPLED
						| MCF_GPIO_PORTLD_PORTLD4 // COLLED
						| MCF_GPIO_PORTLD_PORTLD5 // RXLED
						| MCF_GPIO_PORTLD_PORTLD6 // TXLED
						);
	}	  
	/*
	 * Initialize Port TA to enable Axcel control
	 */
	MCF_GPIO_PTAPAR = 0x00; 
	MCF_GPIO_DDRTA  = 0x0F;
	MCF_GPIO_PORTTA = 0x04;
}

/********************************************************************/
void
mcf5223_uart_init(void)
{
	/*
	 * Initialize all three UARTs for serial communications
	 */

	register uint16 ubgs;

	/*
	 * Set Port UA to initialize URXD0/UTXD0
	 */
    MCF_GPIO_PUAPAR = 0
        | MCF_GPIO_PUAPAR_RXD0_RXD0
        | MCF_GPIO_PUAPAR_TXD0_TXD0;

    MCF_GPIO_PUBPAR = 0
        | MCF_GPIO_PUBPAR_RXD1_RXD1
        | MCF_GPIO_PUBPAR_TXD1_TXD1;

    MCF_GPIO_PUCPAR = 0
        | MCF_GPIO_PUCPAR_RXD2_RXD2
        | MCF_GPIO_PUCPAR_TXD2_TXD2;

	/*
	 * Reset Transmitter
	 */
	MCF_UART0_UCR = MCF_UART_UCR_RESET_TX;
	MCF_UART1_UCR = MCF_UART_UCR_RESET_TX;
	MCF_UART2_UCR = MCF_UART_UCR_RESET_TX;

	/*
	 * Reset Receiver
	 */
	MCF_UART0_UCR = MCF_UART_UCR_RESET_RX;
	MCF_UART1_UCR = MCF_UART_UCR_RESET_RX;
	MCF_UART2_UCR = MCF_UART_UCR_RESET_RX;

	/*
	 * Reset Mode Register
	 */
	MCF_UART0_UCR = MCF_UART_UCR_RESET_MR;
	MCF_UART1_UCR = MCF_UART_UCR_RESET_MR;
	MCF_UART2_UCR = MCF_UART_UCR_RESET_MR;

	/*
	 * No parity, 8-bits per character
	 */
	MCF_UART0_UMR = (0
		| MCF_UART_UMR_PM_NONE
		| MCF_UART_UMR_BC_8 );
	MCF_UART1_UMR = (0
		| MCF_UART_UMR_PM_NONE
		| MCF_UART_UMR_BC_8 );
	MCF_UART2_UMR = (0
		| MCF_UART_UMR_PM_NONE
		| MCF_UART_UMR_BC_8 );

	/*
	 * No echo or loopback, 1 stop bit
	 */
	MCF_UART0_UMR = (0
		| MCF_UART_UMR_CM_NORMAL
		| MCF_UART_UMR_SB_STOP_BITS_1);
	MCF_UART1_UMR = (0
		| MCF_UART_UMR_CM_NORMAL
		| MCF_UART_UMR_SB_STOP_BITS_1);
	MCF_UART2_UMR = (0
		| MCF_UART_UMR_CM_NORMAL
		| MCF_UART_UMR_SB_STOP_BITS_1);

	/*
	 * Set Rx and Tx baud by SYSTEM CLOCK
	 */
	MCF_UART0_UCSR = (0
		| MCF_UART_UCSR_RCS_SYS_CLK
		| MCF_UART_UCSR_TCS_SYS_CLK);
	MCF_UART1_UCSR = (0
		| MCF_UART_UCSR_RCS_SYS_CLK
		| MCF_UART_UCSR_TCS_SYS_CLK);
	MCF_UART2_UCSR = (0
		| MCF_UART_UCSR_RCS_SYS_CLK
		| MCF_UART_UCSR_TCS_SYS_CLK);

	/*
	 * Mask all UART interrupts
	 */
	MCF_UART0_UIMR = 0;
	MCF_UART1_UIMR = 0;
	MCF_UART2_UIMR = 0;

	/*
	 * Calculate baud settings
	 */
	ubgs = (uint16)((SYSTEM_CLOCK*1000000)/(UART_BAUD * 32));

	MCF_UART0_UBG1 = (uint8)((ubgs & 0xFF00) >> 8);
	MCF_UART0_UBG2 = (uint8)(ubgs & 0x00FF);
	MCF_UART1_UBG1 = (uint8)((ubgs & 0xFF00) >> 8);
	MCF_UART1_UBG2 = (uint8)(ubgs & 0x00FF);
	MCF_UART2_UBG1 = (uint8)((ubgs & 0xFF00) >> 8);
	MCF_UART2_UBG2 = (uint8)(ubgs & 0x00FF);

	/*
	 * Enable receiver and transmitter
	 */
	MCF_UART0_UCR = (0
		| MCF_UART_UCR_TX_ENABLED
		| MCF_UART_UCR_RX_ENABLED);
	MCF_UART1_UCR = (0
		| MCF_UART_UCR_TX_ENABLED
		| MCF_UART_UCR_RX_ENABLED);
	MCF_UART2_UCR = (0
		| MCF_UART_UCR_TX_ENABLED
		| MCF_UART_UCR_RX_ENABLED);

}
/********************************************************************/


/********************************************************************/
// Init ePHY
/********************************************************************/
void mcf5223_ePHY_init(void)
{
	uint32 		myctr; 					//generic counter variable
	uint16 		mymrdata, mymwdata;    	//temp variable for MII read/write data
  	uint16		reg0, reg1, reg4;				//


	fec_mii_init((SYSTEM_CLOCK));

  	// set phy address to zero
  	MCF_PHY_EPHYCTL1 = MCF_PHY_EPHYCTL1_PHYADDR(FEC_PHY0); //	FEC_PHY0 from m5223evb.h 

  	//Enable EPHY module with PHY clocks disabled
  	//Do not turn on PHY clocks until both FEC and EPHY are completely setup (see Below)
//  	MCF_PHY_EPHYCTL0 = (uint8)(MCF_PHY_EPHYCTL0_DIS100 | MCF_PHY_EPHYCTL0_DIS10);
  	MCF_PHY_EPHYCTL0 = (uint8)(MCF_PHY_EPHYCTL0  & ~(MCF_PHY_EPHYCTL0_DIS100 | MCF_PHY_EPHYCTL0_DIS10)); 

  	//Disable auto_neg at start-up
  	MCF_PHY_EPHYCTL0 = (uint8)(MCF_PHY_EPHYCTL0 | (MCF_PHY_EPHYCTL0_ANDIS));

  	//Enable EPHY module
  	MCF_PHY_EPHYCTL0 = (uint8)(MCF_PHY_EPHYCTL0_EPHYEN | MCF_PHY_EPHYCTL0);

		// Force ePHY to manual, 100mbps, Half Duplexe
		(void)fec_mii_read(0, 0, &reg0);
		reg0 |= 0x2000;								// 100Mbps
		reg0 &= ~0x0100;							// Half Duplexe
		reg0 &= ~0x1000;							// Manual Mode	
		(void)fec_mii_write( 0, 0, reg0 );
//		(void)fec_mii_write( 0, 0, (reg0|0x0200) ); // Force re-negotiate 

	// Startup delay
//	for (myctr=150000; myctr >0; myctr--){uart_isr(0);}

	//Enable PHY interrupts in Reg 16 (PHY Interrupt Control Register)
	//Set PHY Interrupt Control Register
	mymwdata = PHY_R16_ACKIE | PHY_R16_PRIE | PHY_R16_LCIE | PHY_R16_ANIE;
	mymwdata = mymwdata | PHY_R16_PDFIE | PHY_R16_RFIE | PHY_R16_JABIE;
	while (!(fec_mii_write(FEC_PHY0, PHY_REG_IR, mymwdata)))
	{		
	};
	MCF_INTC0_ICR36 = MCF_INTC_ICR_IL(3);
	MCF_INTC0_IMRH &=  ~(MCF_INTC_IMRH_MASK36);
	MCF_PHY_EPHYCTL0 = MCF_PHY_EPHYCTL0 | (MCF_PHY_EPHYCTL0_EPHYIEN );

	for (myctr=10000; myctr >0; myctr--){uart_check();}

	//*****************************************************************************
	//
	// Work-around for bug in hardware autonegotiation.
	// Attempt to connect at 100Mbps - Half Duplexe
	// Wait for seconds
	// Attempt to connect at 10Mbps - Half Duplexe
	// 
	// Returns 10, or 100 on success, 0 on failure
	//*****************************************************************************
	if( 1 )
	{
		// Force ePHY to manual, 100mbps, Half Duplexe
		while( !fec_mii_read(0, 0, &reg0) ){};
		reg0 |= 0x2000;								// 100Mbps
		reg0 &= ~0x0100;							// Half Duplexe
		reg0 &= ~0x1000;							// Manual Mode	
		while( !fec_mii_write( 0, 0, reg0 ) ){};
		while( !fec_mii_write( 0, 0, (reg0|0x0200) )){}; // Force re-negotiate 
	
		for( myctr=400000; myctr; myctr-- )
		{
			uart_check();
			(void)fec_mii_read(0, 1, &reg1);
			if( reg1 & 0x0004 )
			{
//				printf( "\nLink UP - 100 HD" );				
				return;
			}
		}
	
		// Force ePHY to manual, 10mbps, Half Duplexe
		while( !fec_mii_read(0, 0, &reg0) ){};
		reg0 &= ~0x2000;							// 10Mbps
		reg0 &= ~0x0100;							// Half Duplexe
		reg0 &= ~0x1000;							// Manual Mode	
		while( !fec_mii_write( 0, 0, reg0 ) ){};
		while( !fec_mii_write( 0, 0, (reg0|0x0200) )){}; // Force re-negotiate 
	
#if 0
		for( myctr=20000; myctr; myctr-- )
		{
			uart_check();
			(void)fec_mii_read(0, 1, &reg1);
			printf( "\nLink UP - 10 HD" );
			if( reg1 & 0x0004 )
			{
				printf( "\nLink UP - 10 HD" );				
				return;
			}
		}
#endif		
	}
	
//	printf("\nLink DOWN" );
	
	return;
}


/********************************************************************/
// powerup_config		Written by Eric Gregori
//
// This function reads the switches at power-up, and sets a power-up
// config flag.
/********************************************************************/
void mcf5223_powerup_config( void )
{
	powerup_config_flags = poll_switches() | 0x80;		
}
