% Chebyschev IIR filter design example.
% It demonstrates a proper design procedure that allows design also quite
% complex (low passband ripple, high attenuation, narrow transient region)
% IIR filters. Note that it directly uses poles and zeros of the transfer
% function.
% Direct design based on A and B polynomials does not produce correct
% results. These problems wit A and B polynomials are caused by high
% numerical sensitivity of direct approach for complex IIR filters.

% M. Drutarovsky, KEMT FEI TU Kosice
% 2.3.2003

Fs = 48000;			% sampling frequency
F1 = 4000*pi/(Fs/2);		% normalized stop-band 1
F2 = 5000*pi/(Fs/2);		% normalized pass-band 1
F3 = 7000*pi/(Fs/2);		% normalized pass-band 2
F4 = 8000*pi/(Fs/2);		% normalized stop-band 2
B = 16;				% number of bits
Npts = 1024;			% number of points of computed frequency response

% Chebyschev 1 (classical design based on standard cheby1 function produces
% unacteptable results for this specification)
dp = 0.001;			% passband ripple
ds = [0.0001 0.0001]; 		% stop-band riple
[b,a,v,u,c]=iirdes('ch1','p',[F1 F2 F3 F4], dp, ds);
%[b,a] = cheby1(11, 20*log10(1+dp), [F2 F3]/pi);

% DECOMPOSITION OF A,B POLYNOMIALS IS NOT NECESSARY (moreover it causes
% unaceptable errors for complex IIR filters)!!!
% v = roots(a)			% these values can be re-computed from b,a
% u = roots(b)
% c = b(1)			% gain
[nsec,dsec] = pairpz(v,u);	% matrices of 2nd-order sections
[M, junk] = size(nsec);

c    				% numerators of biquad sections
nsec				% are quite simple
dsec
pause


nsec(1,:) = nsec(1,:)*c;	% include constant gain into the 1st section
% proper scaling of each bikqad section (less numerically stable solution,
% it can be used with also with H1, H2, Hinf norms)
%for k=1:M
%t1 = 1;
%t2 = 1;
%  for kk =1:k
%        nt = nsec(kk,:); dt = dsec(kk,:);
%        t1 = conv(t1,nt);
%        t2 = conv(t2,dt);
%  end;
%  [h1t,H1t,H2t,Hinft] = filnorm(t1,t2);
%  nsec(k,:) = nsec(k,:)/h1t;			% scaled section (h1 norm)
%end;

% proper scaling of each bikqad section (numerically stable solution)
for k=1:M
  nt = nsec(1:k,:); dt = dsec(1:k,:);
  h1t = bfilnorm(nt,dt);
  nsec(k,:) = nsec(k,:)/h1t;			% scaled section (h1 norm)
end;

w = (0:Npts-1)*pi/Npts;
h = frqcasc(nsec,dsec, 1.0, Npts);		% gain is included in nsec!!!

figure(1)
plot(w,abs(h),'r'); hold on

figure(2);
plot(w,20*log10(abs(h)),'r'); hold on

zoom=(214:300);					% define zoom range
figure(3);
plot(w(zoom),abs(h(zoom)),'r'); hold on;

nsec  						% final scaled sections
dsec
pause



nsecq = quant(nsec, 'r', B);			% quantized scaled sections
dsecq = quant(dsec, 'r', B);

h = frqcasc(nsecq,dsecq, 1.0, Npts);		% gain is included in nsec!!!

figure(1);
plot(w,abs(h),'g'); hold off;
title('Frequency characteristic (Linear scale)');
xlabel('Normalized frequency');
ylabel('Atenuation');
grid;

figure(2)
plot(w,20*log10(abs(h)),'g'); hold off
title('Frequency characteristic (Logarithmic scale)');
xlabel('Normalized frequency');
ylabel('Atenuation [dB]');
grid;

figure(3);
plot(w(zoom),abs(h(zoom)),'g'); hold off;
title('Frequency characteristic (16 bit!) - Zoom (Linear scale)');
xlabel('Normalized frequency');
ylabel('Atenuation');
grid;

nsecq  						% final scaled sections
dsecq

