/*
; timer.c
;
; Demonstration program for timer usage and output to the port in C language
;
; Description:
;
;       Program toggles pin P3.4 (Red LED on the ADuC836EB1 board) with 1s period.
;       Period is defined by timer T0 and software counter.
;       Timer T0 is working in mode 1 (16-bit counter).
;
;       To setting:
;
;       TMOD: GATE  C/T  M1  M0  GATE  C/T  M1  M0
;               X    X    X   X    0    0    0  1       Mode 1 of Timer T0
;
;       TCON:  TF1  TR1  TF0  TR0  IE1  IT1   IE0  IT0
;               x    x   x    0/1   x    x     x    x   on/off
;                                                       Timer T0
;
;       IE:     EA  ET2  ES  ET1  EX1  ET0  EX0
;               1    x   x    x    x    1    x          enable interrupt from
;                                                       T0
;
;       Interrupt vector of T0 is at the address 000Bh.
;       State of the pin p3.4 is changed in the inettrupt
;
;       Period of timer T0 je 500 us. For external 12 MHz crystal
;       it requires initialization of T0: 65536-500=65036 (in decimal)
;
;
;       TIMER0 - interrupt routine for T0 timer
;
;       MAIN  - main program, after timer T0 initialization main program
;               runs in the endless loop (just demostration of some
;               activity that can be done "in paralel" with interrupt routine)
;
*/

#include <reg51.h>

#define PERIOD  (65536-500)     // defines T0 interrupt priod
#define CNT          (2000)     // defines number of "PERIODS" between LED changes

sbit LED = P3 ^ 4;

unsigned int pocitadlo;         // software counter (unsigned integer),
                                // 16 bit integer range ensures suitable range

void timer0 (void) interrupt 1  // handling of interrupt caused by timer 0
{
  TR0=0;                        // stop timer T0
  TL0=PERIOD%256;               // define timer period
  TH0=PERIOD/256;
  TR0=1;                        // start timer T0
  if( pocitadlo == 0) {         // test of condition ==0 is simpler than == CNT
          LED = ~LED;           // toggling of LED signal
      pocitadlo = CNT;
  }
  else
          pocitadlo--;          // could be possible to write more efficiently ...
}


//----------------------------------------------
// HLAVNY PROGRAM
//----------------------------------------------

void main (void)
{
  TMOD=0x01;                    // mode 1 of T0 timer
  IE=0x82;                      // eneable interrupt from T0 and enable global interrupts
  TL0=PERIOD%256;               // define interrupt period
  TH0=PERIOD/256;
  pocitadlo = CNT;              // initialize software counter
  TR0=1;                        // start T0 timer

  while(1) {
        ;                       // some "useful activity" in the main program
 }                              // can be added here
}
