//-----------------------------------------------------------------------------
// F326_AES_InvCipher.c
//-----------------------------------------------------------------------------
// Copyright 2007 Silicon Laboratories, Inc.
// http://www.silabs.com
//
// Program Description:
//
//
// How To Test:    See Readme.txt
//
//         
// Target:         C8051F326
// Tool chain:     Keil C51 7.50 / Keil EVAL C51
//                 Silicon Laboratories IDE version 2.91
// Command Line:   See Readme.txt
// Project Name:   F326_AES
//
//
// Release 1.0
//    -Initial Revision (CG/GP)
//    -11 JUN 2007
//

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

#include <C8051F326.h>
#include "F326_AES_Typedef.h"
#include "F326_AES_Parameters.h"
#include "F326_AES_InvCipher.h"

//-----------------------------------------------------------------------------
// Global Constants
//-----------------------------------------------------------------------------

extern pdata byte EXP_KEYS[];

static byte data State[4][4];          // State vector; stored explicitly in
                                       // RAM for speed purposes

static byte data CurrentKey[4][4];     // Stores the keys for the current round
                                       // of encryption

//-----------------------------------------------------------------------------
// Prototypes
//-----------------------------------------------------------------------------

void InvCipher (byte *in, byte *out);

// Functions defined in the AES specification; Core functions of the decryption
void InvSubBytes    (void);
void InvShiftRows   (void);
void InvMixColumns  (void);
void AddRoundKey    (void);

void StateIn     (byte *in);
void StateOut    (byte *out);
void LoadKeys    (char i);
byte FFMultiply (byte x, byte y);

//-----------------------------------------------------------------------------
// Support Subroutines
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// InvCipher
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : 1) byte *in - pointer to array of input, encrypted data
//                2) byte *out - pointer to array output, plaintext data
//
// Procedure that decrypts a message using AES/Rijndael 
//
//-----------------------------------------------------------------------------

void InvCipher (byte *in, byte *out)
{
   char r = Nr;                        // Number of rounds (from Parameters.h)

   StateIn(in);                        // Load string to be decrypted

   LoadKeys(Nr);                       // Load last round of keys
   AddRoundKey();                      // Initial round key add

   for(r = (Nr-1); r > 0; r--)         // Initiate normal rounds
   {
      InvShiftRows ();
      InvSubBytes ();
      LoadKeys (r);
      AddRoundKey ();
      InvMixColumns ();
   }

   InvShiftRows ();
   InvSubBytes ();
   LoadKeys (0);
   AddRoundKey ();

   StateOut (out);                     // Write decrypted string back out
}

//-----------------------------------------------------------------------------
// StateIn
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : 1) byte *in - pointer to the input which is plaintext data
//
// Copies the plaintext data to the global State array
//
// 809 cycles
//-----------------------------------------------------------------------------

static void StateIn(byte *in)
{
   char col,row;

   for(col = 0; col < 4; col++) {
      for(row = 0; row < 4; row++) {
         State[row][col] = *(in++); }}
}

//-----------------------------------------------------------------------------
// StateOut
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : 1) byte *out - pointer to the output which is encrypted data
//
// Copies the encrypted data from the global State array to the output array
//
// 825 cycles
//-----------------------------------------------------------------------------

static void StateOut (byte *out)
{
   byte col,row;

   for(col = 0; col < 4; col++) {
      for(row = 0; row < 4; row++) {
         *(out++) = State[row][col]; }}
}

//-----------------------------------------------------------------------------
// InvSubBytes
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Executes the inverse byte substitution using the inverse 
// Substitution Table (Si-box)
//
// Since the indices are hardcoded, Keil optimizes this code as effectively
// as a single dimensional array.
//
// 73 cycles
//-----------------------------------------------------------------------------

void InvSubBytes (void)
{
   	State[0][0] = InvSbox[State[0][0]];
   	State[0][1] = InvSbox[State[0][1]];
   	State[0][2] = InvSbox[State[0][2]];
   	State[0][3] = InvSbox[State[0][3]];
   	State[1][0] = InvSbox[State[1][0]];
   	State[1][1] = InvSbox[State[1][1]];
   	State[1][2] = InvSbox[State[1][2]];
   	State[1][3] = InvSbox[State[1][3]];
   	State[2][0] = InvSbox[State[2][0]];
   	State[2][1] = InvSbox[State[2][1]];
   	State[2][2] = InvSbox[State[2][2]];
   	State[2][3] = InvSbox[State[2][3]];
   	State[3][0] = InvSbox[State[3][0]];
   	State[3][1] = InvSbox[State[3][1]];
   	State[3][2] = InvSbox[State[3][2]];
   	State[3][3] = InvSbox[State[3][3]];
}

//-----------------------------------------------------------------------------
// ShiftRows
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Executes the row shifts; Only Rows 1, 2, and 3 shift.  Row 0 does not.
//
// Since the indices are hardcoded, Keil optimizes this code as effectively
// as a single dimensional array.
//
// 40 cycles
//-----------------------------------------------------------------------------

void InvShiftRows (void)
{
    byte hold;
    
	// Shift Row 1 right one column and wrap around
    hold = State[1][3];               
    State[1][3] = State[1][2];
    State[1][2] = State[1][1];
    State[1][1] = State[1][0];
    State[1][0] = hold;
    
    // Shift Row 2 right two columns and wrap around
    hold = State[2][2];
    State[2][2] = State[2][0];
    State[2][0] = hold;
    hold = State[2][3];
    State[2][3] = State[2][1];
    State[2][1] = hold;

    // Shift Row 3 right three columns;  Implemented as a left-shift for speed
    hold = State[3][0];                
	State[3][0] = State[3][1];
	State[3][1] = State[3][2];
	State[3][2] = State[3][3];
	State[3][3] = hold;
}

//-----------------------------------------------------------------------------
// InvMixColumns
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Inverse of the mix columns operation
//
// Directly assigning State[3][x] to aux3 didn't save any cycles;  Keil 
// is already optimizing this move.
//
// 3092 cycles
//-----------------------------------------------------------------------------

void InvMixColumns (void)
{
   byte aux0, aux1, aux2, aux3;

   aux0 = FFMultiply(0x0E, State[0][0]) ^ FFMultiply(0x0B, State[1][0]) ^
          FFMultiply(0x0D, State[2][0]) ^ FFMultiply(0x09, State[3][0]);
   aux1 = FFMultiply(0x09, State[0][0]) ^ FFMultiply(0x0E, State[1][0]) ^
          FFMultiply(0x0B, State[2][0]) ^ FFMultiply(0x0D, State[3][0]);
   aux2 = FFMultiply(0x0D, State[0][0]) ^ FFMultiply(0x09, State[1][0]) ^
          FFMultiply(0x0E, State[2][0]) ^ FFMultiply(0x0B, State[3][0]);
   aux3 = FFMultiply(0x0B, State[0][0]) ^ FFMultiply(0x0D, State[1][0]) ^
          FFMultiply(0x09, State[2][0]) ^ FFMultiply(0x0E, State[3][0]);
   State[0][0] = aux0;
   State[1][0] = aux1;
   State[2][0] = aux2;
   State[3][0] = aux3;

   aux0 = FFMultiply(0x0E, State[0][1]) ^ FFMultiply(0x0B, State[1][1]) ^
          FFMultiply(0x0D, State[2][1]) ^ FFMultiply(0x09, State[3][1]);
   aux1 = FFMultiply(0x09, State[0][1]) ^ FFMultiply(0x0E, State[1][1]) ^
          FFMultiply(0x0B, State[2][1]) ^ FFMultiply(0x0D, State[3][1]);
   aux2 = FFMultiply(0x0D, State[0][1]) ^ FFMultiply(0x09, State[1][1]) ^
          FFMultiply(0x0E, State[2][1]) ^ FFMultiply(0x0B, State[3][1]);
   aux3 = FFMultiply(0x0B, State[0][1]) ^ FFMultiply(0x0D, State[1][1]) ^
          FFMultiply(0x09, State[2][1]) ^ FFMultiply(0x0E, State[3][1]);
   State[0][1] = aux0;
   State[1][1] = aux1;
   State[2][1] = aux2;
   State[3][1] = aux3;

   aux0 = FFMultiply(0x0E, State[0][2]) ^ FFMultiply(0x0B, State[1][2]) ^
          FFMultiply(0x0D, State[2][2]) ^ FFMultiply(0x09, State[3][2]);
   aux1 = FFMultiply(0x09, State[0][2]) ^ FFMultiply(0x0E, State[1][2]) ^
          FFMultiply(0x0B, State[2][2]) ^ FFMultiply(0x0D, State[3][2]);
   aux2 = FFMultiply(0x0D, State[0][2]) ^ FFMultiply(0x09, State[1][2]) ^
          FFMultiply(0x0E, State[2][2]) ^ FFMultiply(0x0B, State[3][2]);
   aux3 = FFMultiply(0x0B, State[0][2]) ^ FFMultiply(0x0D, State[1][2]) ^
          FFMultiply(0x09, State[2][2]) ^ FFMultiply(0x0E, State[3][2]);
   State[0][2] = aux0;
   State[1][2] = aux1;
   State[2][2] = aux2;
   State[3][2] = aux3;

   aux0 = FFMultiply(0x0E, State[0][3]) ^ FFMultiply(0x0B, State[1][3]) ^
          FFMultiply(0x0D, State[2][3]) ^ FFMultiply(0x09, State[3][3]);
   aux1 = FFMultiply(0x09, State[0][3]) ^ FFMultiply(0x0E, State[1][3]) ^
          FFMultiply(0x0B, State[2][3]) ^ FFMultiply(0x0D, State[3][3]);
   aux2 = FFMultiply(0x0D, State[0][3]) ^ FFMultiply(0x09, State[1][3]) ^
          FFMultiply(0x0E, State[2][3]) ^ FFMultiply(0x0B, State[3][3]);
   aux3 = FFMultiply(0x0B, State[0][3]) ^ FFMultiply(0x0D, State[1][3]) ^
          FFMultiply(0x09, State[2][3]) ^ FFMultiply(0x0E, State[3][3]);
   State[0][3] = aux0;
   State[1][3] = aux1;
   State[2][3] = aux2;
   State[3][3] = aux3;
}


//-----------------------------------------------------------------------------
// AddRoundKey
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Procedure that adds round keys to the state
//
// 64 cycles
//-----------------------------------------------------------------------------

static void AddRoundKey (void)
{
    State[0][0] ^= CurrentKey[0][0];
	State[0][1] ^= CurrentKey[0][1];
	State[0][2] ^= CurrentKey[0][2];
	State[0][3] ^= CurrentKey[0][3];

	State[1][0] ^= CurrentKey[1][0];
	State[1][1] ^= CurrentKey[1][1];
	State[1][2] ^= CurrentKey[1][2];
	State[1][3] ^= CurrentKey[1][3];

    State[2][0] ^= CurrentKey[2][0];
	State[2][1] ^= CurrentKey[2][1];
	State[2][2] ^= CurrentKey[2][2];
	State[2][3] ^= CurrentKey[2][3];

    State[3][0] ^= CurrentKey[3][0];
	State[3][1] ^= CurrentKey[3][1];
	State[3][2] ^= CurrentKey[3][2];
	State[3][3] ^= CurrentKey[3][3];
}

//-----------------------------------------------------------------------------
// LoadKeys
//-----------------------------------------------------------------------------
//
// Return Value : None
// Parameters   : None
//
// Procedure that loads the current key from Flash into RAM
//
// 191 cycles
//-----------------------------------------------------------------------------

static void LoadKeys (char i)
{
   // Change index i from the number of the round to the start of round in
   // in the linear array of keys
   unsigned char index = (unsigned char) i * 16;           

   CurrentKey[0][0] = EXP_KEYS[index++];
   CurrentKey[1][0] = EXP_KEYS[index++];
   CurrentKey[2][0] = EXP_KEYS[index++];
   CurrentKey[3][0] = EXP_KEYS[index++];

   CurrentKey[0][1] = EXP_KEYS[index++];
   CurrentKey[1][1] = EXP_KEYS[index++];
   CurrentKey[2][1] = EXP_KEYS[index++];
   CurrentKey[3][1] = EXP_KEYS[index++];

   CurrentKey[0][2] = EXP_KEYS[index++];
   CurrentKey[1][2] = EXP_KEYS[index++];
   CurrentKey[2][2] = EXP_KEYS[index++];
   CurrentKey[3][2] = EXP_KEYS[index++];

   CurrentKey[0][3] = EXP_KEYS[index++];
   CurrentKey[1][3] = EXP_KEYS[index++];
   CurrentKey[2][3] = EXP_KEYS[index++];
   CurrentKey[3][3] = EXP_KEYS[index++];
}

//-----------------------------------------------------------------------------
// FFMultiply
//-----------------------------------------------------------------------------
//
// Return Value : 1) Returns the product
// Parameters   : 1) byte x - multiplier
//                2) byte y - multiplier
//
// Performs a finite field multiply using a table lookup
//
// 30 cycles
//-----------------------------------------------------------------------------

static byte FFMultiply (byte x, byte y)
{
   unsigned char temp_result;

   if (y==0) {         // Multiplying by 0 always returns 0, 
      return 0; }

   // !! Since we are checking carry flag, must be aware of what is happening
   // to PSW register after the addition.  If an interrupt is posted after
   // the addition but before the CY bit is set, the behavior of this function
   // is not necessarily correct.

   // save EA bit status and clear it.

   temp_result = log_table[x] + log_table[y];
   if (CY) {                          // carry flag set
      temp_result += 1; }

   // restore EA status
   
   return (byte) (exp_table[temp_result]);
}

//-----------------------------------------------------------------------------
// End Of File
//-----------------------------------------------------------------------------